<?php
/**
 * Please see weepie-framework.php for more details.
 */

namespace WpieFw\Hooks;

use WpieFw\Wpie\WpieGlobals;
use WpieFw\Helpers\WpieMiscHelper;
use WpieFw\Settings\WpieSettingsProcessor;
use WpieFw\Exceptions\WpieExceptionInterface;
use WpieFw\Exceptions\WpieExceptionLogger;
use WpieFw\Notices\WpieNotices;

if( ! defined( 'ABSPATH' ) ) exit;

/**
 * WpieHooksAdmin Class
 *
 * @author $Author: Vincent Weber <vincent@webrtistik.nl> $
 * @since 1.4.0
 */
final class WpieHooksAdmin extends WpieHooks
{
	/**
	 * Flag if has settings
	 *
	 * @since 1.4.4
	 *
	 * @var boolean
	 */
	private $doSettings = false;

	/**
	 * WpieSettingsProcessor instance
	 *
	 * @since 1.4.0
	 * @since 2.0 WpieSettingsProcessor instead of WpiePluginSettingsProcessor
	 *
	 * @var WpieSettingsProcessor
	 */
	private $settingsProcessor = false;

	/**
	 * Constructor
	 *
	 * @access public
	 *
	 * @param WpieSettingsProcessor $settingsProcessor
	 * @param WpieGlobals $globals
	 *
	 * @since 1.4.0
	 */
	public function __construct( WpieSettingsProcessor $settingsProcessor, WpieGlobals $globals )
	{
		parent::__construct( $globals );

		$this->settingsProcessor = $settingsProcessor;
	}

	/**
	 * {@inheritDoc}
	 * @see WpieHooks::add()
	 */
	public function add()
	{
		// Network vs non network admin hooks
		if( is_network_admin() ) {
			add_action( 'wpmu_new_blog', [ $this, 'activatePluginForNewBlog' ] );
		} elseif( !is_network_admin() && is_admin() ) {

		}

		// Admin hooks
		if( is_admin() ) {
			add_action( 'admin_menu', [ $this, 'setAdminPages' ], 10 );
			add_action( 'admin_menu', [ $this, 'removeAdminPages' ], 11 );
			add_action( 'admin_enqueue_scripts', [ $this, 'setScriptsAdmin' ] );
			add_action( 'admin_enqueue_scripts', [ $this, 'setStylesAdmin' ] );
			add_action( 'admin_print_styles', [ $this, 'printStylesAdmin' ], 100 );
			add_action( 'admin_print_scripts', [ $this, 'printScriptsAdminHeaderVars' ] );
			add_action( 'admin_print_scripts', [ $this, 'printScriptsAdminHeader' ], 11 );
			add_action( 'admin_print_footer_scripts', [ $this, 'printScriptsAdminFooter' ] );
			add_action( 'admin_footer', [ $this, 'unsetActivating' ], 99999 );

			// frontend ajax requests
			add_action( 'wp_ajax_wpie-action', [ $this, 'processAjaxRequest' ] );
			add_action( 'wp_ajax_nopriv_wpie-action', [ $this, 'processAjaxRequest' ] );
		}
	}

	/**
	 * Callback for the admin_enqueue_scripts hook
	 *
	 * @acces public
	 *
	 * @uses wp_enqueue_script()
	 * @uses [YOUR_PLUGIN_NAMESPACE]_scripts_admin_before
	 * @uses [YOUR_PLUGIN_NAMESPACE]_scripts_admin to let WeePie Framework Plugins enqueue scripts for admin pages
	 * @uses [YOUR_PLUGIN_NAMESPACE]_scripts_admin_after
	 *
	 * @since 0.1
	 */
	public function setScriptsAdmin( $hook_suffix )
	{
		global $wp_scripts;

		$uriAssetsJs = $this->globals->get( 'wfJsUri' );
		$pluginPath = $this->globals->get( 'pluginPath' );
		$jsUri = $this->globals->get( 'jsUri' );
		$isScriptDebug = WpieMiscHelper::isScriptDebug();
		$ext = ( $isScriptDebug ) ? '.js' : '.min.js';

		try {
			/**
			 * Let others enqueue scripts on the admin before
			 *
			 * @param string $hook_suffix
			 * @param WP_Scripts $wp_scripts
			 * @param boolean $isScriptDebug
			 *
			 * @since 1.4.7
			 */
			do_action( $this->globals->nameSpace . '_scripts_admin_before', $hook_suffix, $wp_scripts, $isScriptDebug );
		} catch( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}

		if( null !== $uriAssetsJs ) {
			// enqueue common functions
			$file = $uriAssetsJs . '/wpie-common' . $ext;
			wp_enqueue_script( 'wpie-common', $file, [], $this->globals->get( 'versionWf' ), true );

			$file = $uriAssetsJs . '/wpie-notices' . $ext;
			wp_enqueue_script( 'wpie-notices', $file, [], $this->globals->get( 'versionWf' ), true );
		}

		if( file_exists( $pluginPath . '/assets/js/global.admin' . $ext ) ) {
			wp_enqueue_script( $this->globals->nameSpace.'-global-admin', $jsUri . '/global.admin' . $ext, [ 'jquery', 'wpie-common' ], $this->globals->get('version'), true );
		}

		try {
			/**
			 * Let others enqueue scripts on the admin after
			 *
			 * @param string $hook_suffix
			 * @param WP_Scripts $wp_scripts
			 * @param boolean $isScriptDebug
			 *
			 * @since 1.4.6 added $wp_scripts and $isModeDev param
			 * @since 1.4.7 added $isScriptDebug param instead of $isModeDev
			 */
			do_action( $this->globals->nameSpace . '_scripts_admin', $hook_suffix, $wp_scripts, $isScriptDebug );

			/**
			 * Let others enqueue scripts on the admin after
			 *
			 * @param string $hook_suffix
			 * @param WP_Scripts $wp_scripts
			 * @param boolean $isScriptDebug
			 *
			 * @since 1.4.7
			 */
			do_action( $this->globals->nameSpace . '_scripts_admin_after', $hook_suffix, $wp_scripts, $isScriptDebug );
		} catch( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}
	}

	/**
	 * Callback for the admin_print_scripts hook: setup global JavaScript params
	 *
	 * @access public
	 *
	 * @uses [YOUR_PLUGIN_NAMESPACE]_script_admin_vars to let WeePie Framework Plugins modify the $wpieVars array
	 * @uses json_encode() to safely create a JavaScript array
	 *
	 * @since 1.0
	 */
	public function printScriptsAdminHeaderVars()
	{
		static $did = false;

		if( !$did && $this->globals->get( 'doSettings' ) && $this->settingsProcessor->hasSettingsPage && $this->settingsProcessor->settingsPage->onSettingsPage ) {
			$wpieVarsGlobal = [];
			$wpieVarsGlobal['curr_sett_ns'] = $this->globals->nameSpace;
			$did = true;
		}

		$wpieVars = [];
		$wpieVars['ns'] = $this->globals->nameSpace;
		$wpieVars['nonce'] = $this->globals->get( 'nonce' );

		try {
			$wpieVars = apply_filters( $this->globals->nameSpace . '_script_admin_vars' , $wpieVars );
		} catch ( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}
		?>
		<script type='text/javascript'>
		/* <![CDATA[ */
		<?php if( isset( $wpieVarsGlobal['curr_sett_ns'] ) ): ?>var wpieData = <?php echo json_encode( $wpieVarsGlobal ) ?>;<?php echo "\n"; endif ?>
		var <?php echo $this->globals->get( 'jsNamespace' ) ?> = <?php echo json_encode( $wpieVars ) ?>;
		/* ]]> */
		</script>
		<?php
	}

	/**
	 * Callback for the admin_print_scripts hook
	 *
	 * @access public
	 *
	 * @uses [YOUR_PLUGIN_NAMESPACE]_print_scripts_admin_header to let WeePie Framework Plugins print scripts in the admin head
	 *
	 * @since 1.0
	 */
	public function printScriptsAdminHeader()
	{
		try {
			/**
			 * @todo description for hook {$nameSpace}_print_scripts_admin_header
			 */
			do_action( $this->globals->nameSpace . '_print_scripts_admin_header' );
		} catch ( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}
	}

	/**
	 * Callback for the admin_print_footer_scripts hook
	 *
	 * @access public
	 *
	 * @uses [YOUR_PLUGIN_NAMESPACE]_print_scripts_admin_footer to let WeePie Framework Plugins print scripts in the admin footer
	 *
	 * @since 1.0
	 */
	public function printScriptsAdminFooter()
	{
		try {
			/**
			 * @todo description for hook {$nameSpace}_print_scripts_admin_footer
			 */
			do_action( $this->globals->nameSpace . '_print_scripts_admin_footer' );
		} catch ( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
		}
	}

	/**
	 * Callback for the admin_enqueue_scripts hook
	 *
	 * Enqueue styles for the admin Plugin page
	 *
	 * @acces public
	 *
	 * @uses [YOUR_PLUGIN_NAMESPACE]_styles_admin to let WeePie Framework Plugins enqueue styles for in the admin head
	 *
	 * @since 0.1
	 */
	public function setStylesAdmin( $hook_suffix )
	{
		global $wp_styles;

		$isScriptDebug = WpieMiscHelper::isScriptDebug();

		try {
			/**
			 * @todo description for hook {$nameSpace}_styles_admin
			 */
		    do_action( $this->globals->nameSpace . '_styles_admin', $hook_suffix, $wp_styles, $isScriptDebug );
		} catch ( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}
	}

	/**
	 * Callback for the admin_print_styles hook
	 *
	 * @access public
	 *
	 * @uses [YOUR_PLUGIN_NAMESPACE]_print_styles_admin to let WeePie Framework Plugins print styles for in the admin head
	 *
	 * @since 1.0
	 */
	public function printStylesAdmin()
	{
		global $hook_suffix;

		try {
			/**
			 * @todo description for hook {$nameSpace}_print_styles_admin
			 */
			do_action( $this->globals->nameSpace . '_print_styles_admin', $hook_suffix );
		} catch ( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}
	}

	/**
	 * Callback for the admin_menu hook
	 *
	 * @access public
	 *
	 * @uses [YOUR_PLUGIN_NAMESPACE]_add_admin_pages to let WeePie Framework Plugins add admin page(s)
	 *
	 * @since 0.1
	 */
	public function setAdminPages()
	{
		try {
			/**
			 * @todo description for hook {$nameSpace}_add_admin_pages
			 */
			do_action( $this->globals->nameSpace . '_add_admin_pages' );
		} catch ( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}
	}

	/**
	 * Callback for the admin_menu hook
	 *
	 * @access public
	 *
	 * @uses [YOUR_PLUGIN_NAMESPACE]_add_admin_pages to let WeePie Framework Plugins remove admin page(s)
	 *
	 * @since 1.0
	 */
	public function removeAdminPages()
	{
		try {
			/**
			 * @todo description for hook {$nameSpace}_remove_admin_pages
			 */
			do_action( $this->globals->nameSpace . '_remove_admin_pages' );
		} catch ( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}
	}

	/**
	 * Callback for the wpmu_new_blog hook
	 *
	 * Ensure that new blogs are configured automaticly
	 *
	 * @access public
	 *
	 * @param int $blogId
	 *
	 * @since 1.1
	 */
	public function activatePluginForNewBlog( $blogId )
	{
		switch_to_blog( $blogId );

		try {
			/**
			 * @todo description for hook activate_{$pluginFile}
			 */
			do_action( 'activate_'  . $this->globals->get( 'pluginFile' ), false );
		} catch ( WpieExceptionInterface $e ) {
			WpieExceptionLogger::log( $e );
			WpieNotices::add( $this->globals->nameSpace, $e->getMessage() );
		}

		restore_current_blog();
	}

	/**
	 * Callback for admin_footer-plugins.php hook
	 *
	 * Unset the WeePieFramework::activatingOptionName option
	 *
	 * @TODO: consider a WP Cronjob for removing this option in case it failed removing
	 * @TODO: implement a WpieDefenition for this option
	 *
	 * @uses	delete_option()
	 *
	 * @since	1.1.8
	 */
	public function unsetActivating()
	{
		delete_option( $this->globals->get( 'optionActivating' ) );
	}
}