<?php

namespace WpieFw\Settings\Parser;

use WpieFw\Files\WpieFile;
use WpieFw\Exceptions\WpieInvalidArgumentException;

if( ! defined( 'ABSPATH' ) ) exit;

abstract class WpieAbstractSettingsParser implements WpieSettingsParserInterface
{
	/**
	 * @var string
	 */
	const KEY_GROUP_TITLE = 'group_title';

	/**
	 * @var string
	 */
	const KEY_GROUP_DESCR = 'group_descr';

	/**
	 * @var string
	 */
	const KEY_GROUP_WARNING = 'group_warning';

	/**
	 * @var string
	 */
	const KEY_TITLE = 'title';

	/**
	 * @var string
	 */
	const KEY_DESCR = 'descr';

	/**
	 * @var string
	 */
	const KEY_INNER = 'inner';

	/**
	 * @var WpieFile
	 */
	protected $file;

	/**
	 * @var string
	 */
	protected $content = '';

	/**
	 * @var array
	 */
	protected $defaults = [];

	/**
	 * @var array
	 */
	protected $I18n = [];

	/**
	 * @var integer
	 */
	protected $priority = 1;

	/**
	 * The settings index
	 *
	 * @var string
	 */
	protected $idx = '';

	/**
	 * Whether or not the settings is marked as default
	 *
	 * @var bool
	 */
	protected $isDefault = false;

	/**
	 * The parsed settings array
	 *
	 * @var array
	 */
	protected $array = [];

	/**
	 * Constuctor
	 *
	 * @param WpieFile $file
	 *
	 * @uses WpieFile::getContent()
	 * @uses self::load()
	 */
	public function __construct( WpieFile $file )
	{
		$this->file = $file;
	}

	/**
	 * {@inheritDoc}
	 * @see \WpieFw\Settings\Parser\WpieSettingsParserInterface::getFile()
	 */
	public function getFile()
	{
		return $this->file;
	}

	/**
	 * {@inheritDoc}
	 * @see \WpieFw\Settings\Parser\WpieSettingsParserInterface::getContents()
	 */
	public function getContents()
	{
		return $this->content;
	}

	/**
	 * Get parsed defaults
	 *
	 * @return array
	 */
	public function getParsedDefaults()
	{
		return $this->defaults;
	}

	/**
	 * Get parsed I18n values
	 *
	 * @return array
	 */
	public function getParsedI18n()
	{
		return $this->I18n;
	}

	/**
	 * Get parsed priority
	 *
	 * @return integer
	 */
	public function getParsedPriority()
	{
		return $this->priority;
	}

	/**
	 * Get parsed Idx
	 *
	 * @return string
	 */
	public function getParsedIdx()
	{
		return $this->idx;
	}

	/**
	 * Get parsed 'is default'
	 *
	 * @return boolean
	 */
	public function getParsedIsDefault()
	{
		return $this->isDefault;
	}

	/**
	 * Get parsed setting fields as an array
	 *
	 * @return array
	 */
	public function getParsedArray()
	{
		return $this->array;
	}

	/**
	 * {@inheritDoc}
	 * @see \WpieFw\Settings\Parser\WpieSettingsParserInterface::parseFile()
	 */
	public function parseFile( WpieFile $file )
	{
		$this->content = trim( $file->getContent() );

		if( '' === $this->content ) {
			throw new WpieInvalidArgumentException( sprintf( 'Could not parse settings file %s. File is empty.', $file->getFilename() ) );
		}

		return $this->content;
	}

	/**
	 * Parse settings default values
	 *
	 * @return array
	 */
	abstract public function parseDefaults();

	/**
	 * Parse settings translatable I18n values
	 *
	 * @return array
	 */
	abstract public function parseI18n();

	/**
	 * Parse setting priority
	 *
	 * @return integer
	 */
	abstract public function parsePriority();

	/**
	 * Parse setting index
	 *
	 * @return string
	 */
	abstract public function parseIdx();

	/**
	 * Parse setting (is) default
	 *
	 * @return bool
	 */
	abstract public function parseIsDefault();

	/**
	 * Parse setting array
	 *
	 * @return array
	 */
	abstract public function parseToArray();

	/**
	 * Parse all data
	 *
	 * @uses self::load()
	 * @uses self::parseDefaults()
	 * @uses self::parseI18n()
	 * @uses self::parsePriority()
	 * @uses self::parseIdx()
	 * @uses self::parseIsDefault()
	 * @uses self::parseToArray()
	 */
	public function parse()
	{
		// Parse the settings file and let child class load
		$this->load( $this->parseFile( $this->file ) );

		// Parse rest of settings data
		$this->parseDefaults();
		$this->parsePriority();
		$this->parseIdx();
		$this->parseIsDefault();

		// finally parse the settings as an php array
		$this->parseToArray();

		// parse I18n strings based on the parsed array
		$this->parseI18n();
	}
}