<?php

namespace WpieFw\Settings;

use WpieFw\Files\WpieFile;
use WpieFw\Settings\Iterator\WpieSetting;
use WpieFw\Settings\Parser\WpieXmlSettingsParser;
use WpieFw\Settings\Parser\WpieSettingsParserInterface;
use WpieFw\Settings\Iterator\WpieSettingMultilang;
use WpieFw\Multilang\WpieMultilangProcessor;
use WpieFw\Exceptions\WpieInvalidArgumentException;

if( ! defined( 'ABSPATH' ) ) exit;

/**
 * @author webRtistik
 *
 * An instance of this class should be injected in the WpieSettingsGenerator constructor.
 * The Factory will create a WpieSetting instance ( see WpieSettingsGenerator::create() )
 *
 */
final class WpieSettingsFactory implements WpieSettingsFactoryInterface
{
	/**
	 * {@inheritDoc}
	 * @see \WpieFw\Settings\WpieSettingsFactoryInterface::create()
	 */
	public function create( string $type, WpieFile $file, WpieSettingsValues $values )
	{
		$idx = $file->getIndex();
		$parser = $this->getParser( $type, $file );

		return new WpieSetting( $idx, $parser, $values );
	}

	/**
	 * {@inheritDoc}
	 * @see \WpieFw\Settings\WpieSettingsFactoryInterface::createForMultilang()
	 */
	public function createForMultilang( string $type, WpieFile $file, WpieSettingsValues $values, WpieMultilangProcessor $multilangProcessor )
	{
		$idx = $file->getIndex();
		$parser = $this->getParser( $type, $file );

		return new WpieSettingMultilang( $idx, $parser, $values, $multilangProcessor );
	}


	/**
	 * Get parser for given type
	 *
	 * @param string $type
	 * @param WpieFile $file
	 *
	 * @throws WpieInvalidArgumentException
	 *
	 * @return WpieSettingsParserInterface|NULL
	 */
	private function getParser( string $type, WpieFile $file )
	{
		switch( $type ) {
			case 'xml':
				return new WpieXmlSettingsParser( $file );
				break;
			case 'yaml':
				// not implemented yet
				return null;
				break;
			case 'json':
				// not implemented yet
				return null;
				break;
			default:
				throw new WpieInvalidArgumentException( sprintf( 'Factory could not create settings instance for type: %s. Invalid type.', $type ) );
				break;
		}

	}
}