<?php

namespace WpieFw\Settings;

use WpieFw\Settings\Definition\WpieSettingsDefinition;
use WpieFw\Settings\Resolver\WpieSettingValuesResolverFactory;
use WpieFw\Settings\Resolver\WpieSettingValuesWpOptionsResolver;
use WpieFw\Exceptions\WpieInvalidArgumentException;

if( ! defined( 'ABSPATH' ) ) exit;

/**
 * @author webRtistik
 *
 * An instance of this class retrieves the stored (saved) values through the WpieSettingValuesResolverFactory instance
 */
final class WpieSettingsValues
{
	/**
	 * @var WpieSettingValuesWpOptionsResolver
	 */
	private $resolver;

	/**
	 * Holds resolved values
	 *
	 * @todo implement resolved values?
	 *
	 * @var array
	 */
	private $resolved = [];

	/**
	 * @param WpieSettingValuesResolverFactory $factory
	 * @param WpieSettingsDefinition $definitions
	 *
	 * @uses WpieSettingValuesResolverFactory::getResolver()
	 */
	public function __construct( WpieSettingValuesResolverFactory $factory, WpieSettingsDefinition $definitions )
	{
		$this->resolver = $factory->getResolver( $definitions );
	}

	/**
	 * Get the value for a definition
	 *
	 * @param string $shortcut
	 * @param array $replacements
	 * @return string
	 */
	public function getDefinitionValue( $shortcut = '', array $replacements = [] )
	{
		return $this->resolver->getValueForShortcut( $shortcut, $replacements );
	}

	/**
	 * Get stored setting files
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::getFiles()
	 *
	 * @return array
	 */
	public function getFiles()
	{
		return $this->resolver->getFiles();
	}

	/**
	 * Get stored setting fields array
	 *
	 * @param string $idx
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::getArray()
	 *
	 * @return array
	 */
	public function getArray( $idx = '' )
	{
		return $this->resolver->getArray( $idx );
	}

	/**
	 * Get stored default setting
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::getDefault()
	 *
	 * @return string
	 */
	public function getDefault()
	{
		return $this->resolver->getDefault();
	}

	/**
	 * Get stored setting
	 *
	 * @param string $idx
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::getSetting()
	 *
	 * @return array
	 */
	public function getSetting( $idx = '' )
	{
		return $this->resolver->getSetting( $idx );
	}

	/**
	 * Get stored temporary setting
	 *
	 * @param string $idx
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::getSettingTmp()
	 *
	 * @return array
	 */
	public function getSettingTmp( $idx = '' )
	{
		return $this->resolver->getSettingTmp( $idx );
	}

	/**
	 * Set stored setting files
	 *
	 * @param array $files
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::persistFiles()
	 *
	 * @return WpieSettingsValues
	 */
	public function persistFiles( array $files )
	{
		$this->resolver->persistFiles( $files );

		return $this;
	}

	/**
	 * Set stored setting fields array
	 *
	 * @param string $idx
	 * @param array $fields
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::persistArray()
	 *
	 * @return WpieSettingsValues
	 */
	public function persistArray( string $idx, array $fields )
	{
		$this->resolver->persistArray( $idx, $fields );

		return $this;
	}

	/**
	 * Set stored default setting
	 *
	 * @param string $value
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::persistDefault()
	 *
	 * @return WpieSettingsValues
	 */
	public function persistDefault( $value = '' )
	{
		$this->resolver->persistDefault( $value );

		return $this;
	}

	/**
	 * Set stored setting
	 *
	 * @param string $idx
	 * @param array $setting
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::persistSetting()
	 *
	 * @return WpieSettingsValues
	 */
	public function persistSetting( string $idx, array $setting )
	{
		$this->resolver->persistSetting( $idx, $setting );

		return $this;
	}

	/**
	 * Set stored temporary setting
	 *
	 * @param string $idx
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::persistSettingTmp()
	 *
	 * @return WpieSettingsValues
	 */
	public function persistSettingTmp( $idx = '' )
	{
		$this->resolver->persistSettingTmp( $idx );

		return $this;
	}

	/**
	 * Persist shortcut to all persist methods
	 *
	 * @param string $name
	 * @param array $value
	 * @param string $key optional
	 * @return \WpieFw\Settings\WpieSettingsValues
	 */
	public function persist( string $name, array $value, string $key = '' )
	{
		switch( $name ) {
			case 'files':
				$this->persistFiles( $value );
				break;
			case 'array':
				$this->persistArray( $key, $value );
				break;
			case 'setting':
				$this->persistSetting( $key, $value );
				break;
			case 'setting_tmp':
				$this->persistSettingTmp( $key );
				break;
			default:
				throw new WpieInvalidArgumentException( sprintf( 'Could not persist for settings stored values on name: %s.', $name ) );
				break;
		}

		return $this;
	}

	/**
	 * Delete stored setting fields
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::deleteFiles()
	 *
	 * @return WpieSettingsValues
	 */
	public function deleteFiles()
	{
		$this->resolver->deleteFiles();

		return $this;
	}

	/**
	 * Delete stored setting fields array
	 *
	 * @param string $idx
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::deleteArray()
	 *
	 * @return WpieSettingsValues
	 */
	public function deleteArray( $idx = '' )
	{
		$this->resolver->deleteArray( $idx );

		return $this;
	}

	/**
	 * Delete stored settings default
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::deleteDefault()
	 *
	 * @return \WpieFw\Settings\WpieSettingsValues
	 */
	public function deleteDefault()
	{
		$this->resolver->deleteDefault();

		return $this;
	}

	/**
	 * Delete stored setting
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::deleteSetting()
	 *
	 * @param string $idx
	 *
	 * @return \WpieFw\Settings\WpieSettingsValues
	 */
	public function deleteSetting( $idx = '' )
	{
		$this->resolver->deleteSetting( $idx );

		return $this;
	}

	/**
	 * Delete stored temporary setting
	 *
	 * @uses WpieSettingValuesWpOptionsResolver::deleteSetting()
	 *
	 * @param string $idx
	 *
	 * @return \WpieFw\Settings\WpieSettingsValues
	 */
	public function deleteSettingTmp( $idx = '' )
	{
		$this->resolver->deleteSettingTmp( $idx );

		return $this;
	}

	/**
	 * Delete shortcut to all delete methods
	 *
	 * @param string $name
	 * @param string $key optional
	 *
	 * @return \WpieFw\Settings\WpieSettingsValues
	 */
	public function delete( $name = '', $key = '' )
	{
		switch( $name ) {
			case 'files':
				$this->deleteFiles();
				break;
			case 'array':
				$this->deleteArray( $key );
				break;
			case 'default':
				$this->deleteDefault();
				break;
			case 'setting':
				$this->deleteSetting( $key );
				break;
			case 'setting_tmp':
				$this->deleteSettingTmp( $key );
				break;
			default:
				throw new WpieInvalidArgumentException( sprintf( 'Could not delete for settings stored values on name: %s.', $name ) );
				break;
		}

		return $this;
	}
}