<?php
/**
 * Please see wp-cookie-allow.php for more details.
 *
 * @author $Vincent Weber <vincent@webrtistik.nl>$
 */

use WpieFw\Helpers\WpieAjaxHelper;
use WpieFw\Helpers\WpieSettingsHelper;
use WpieFw\PostTypes\WpiePostType;
use WpieFw\Helpers\WpieCookieHelper;
use WpieFw\Helpers\WpieMiscHelper;
use WpieFw\Helpers\WpieMultilangHelper;

if( ! defined( 'ABSPATH' ) ) exit;

if( !class_exists( 'WpcaModuleCore' ) ) {

	/**
	 * WpcaCore Class
	 *
	 * Module to handle the core functionality of the WeePie Cookie Allow Plugin
	 *
	 * @author $Vincent Weber <vincent@webrtistik.nl>$
	 *
	 * @since 2.0
	 */
	class WpcaModuleCore extends WpieFw\Modules\Iterator\WpieModule {

		/**
		 * The loading priority for the Module
		 *
		 * @since 2.0
		 *
		 * @var int
		 */
		protected $priority = 5;


		/**
		 * Flag if the Plugin is active
		 *
		 * @since 1.0
		 *
		 * @var bool
		 */
		public $pluginsStatus = false;


		/**
		 * The choosen cookies to allow on 1st visit
		 *
		 * @since 3.0
		 *
		 * @var string
		 */
		public $cookiesBeforeConsent = 1;


		/**
		 * The choosen accept method
		 *
		 * @since 2.2.6
		 *
		 * @var string
		 */
		public $consentMethod = 1;


		/**
		 * Flag if the plugin is disbaled for logged-in clients
		 *
		 * @since 1.0
		 *
		 * @var bool
		 */
		public $disableLoggedIn = false;


		/**
		 * Flag if to disable for non EU visitors
		 *
		 * Settings value for general_disable_for_non_eu_visitors
		 *
		 * @var boolean
		 *
		 * @since 3.2.6
		 * @since 3.2.11 part of module core, priot to 3.2.11, this property was part of the geo module
		 */
		public $disableForNonEU = false;

		/**
		 * Flag if GTM optimization is enabled
		 *
		 * @var boolean
		 *
		 * @since 3.2.13
		 */
		public $gtmOptimize = false;

		/**
		 * Flag for Google Analytics anonymized skipping
		 *
		 * @var boolean
		 *
		 * @since 3.2.13
		 */
		public $enableGaAnom = true;

		/**
		 * Value of the Cookie wpca_consent
		 *
		 * @since 2.0
		 *
		 * @var string default '1'
		 */
		public $cookieValueConsent = false;


		/**
		 * Cookie policy Post Type
		 *
		 * @since 3.1
		 *
		 * @var bool
		 */
		public $cookiePolicyPostType = 'page';


		/**
		 * The cookies before consent ID allowing Google Analytics
		 *
		 * @since 3.2
		 *
		 * @var integer
		 */
		public $cookiesBeforeConsentIdGoogleAnalytics = 2;


		/**
		 * The cookies before consent ID allowing 3rd parties
		 *
		 * @since 3.1.2
		 *
		 * @var integer
		 */
		public $cookiesBeforeConsentId3rdParties = 3;


		/**
		 * The consent method ID's without Accept button
		 *
		 * @since 3.1.2
		 *
		 * @var array
		 */
		public $consentMethodsIdsNoBtn = array( 5, 6, 7 );


		/**
		 * The consent method ID's without layer
		 *
		 * @since 3.1.2
		 *
		 * @var array
		 */
		public $consentMethodsIdsNoLayer = array( 2, 3, 4, 5, 6, 7 );


		/**
		 * Flag doing a frontend wpca ajax call
		 *
		 * @since 3.4.4
		 *
		 * @var boolean
		 */
		public $doingWpcaAjaxAction = false;

		/**
		 * Flag doing a frontend wpca REST api call
		 *
		 * @since 3.4.4
		 *
		 * @var boolean
		 */
		public $doingWpcaRestAction = false;

		/**
		 * Flag doing the frontend ajax call to render the layout
		 *
		 * @since 3.4.4
		 *
		 * @var boolean
		 */
		public $doingRenderLayoutAjax = false;


		/**
		 * AJAX action for the cookie policy page items
		 *
		 * @since 3.4
		 *
		 * @var string
		 */
		private $_ajaxActionListCookiePolicyItems = 'wpca-update-list-cookie-policy-items';


		/**
		 * AJAX action for rendering the layout
		 *
		 * @since 3.4
		 *
		 * @var string
		 */
		private $_ajaxActionRenderLayout = 'wpca-frontend-render-layout';


		/**
		 * Cookie name for Cookie that will be set if client confirmed Cookie Settings at front-end
		 *
		 * @since 2.0
		 *
		 * @since 2.3.3 renamed to wpca_consent
		 *
		 * @var string
		 */
		const COOKIE_NAME_CONSENT = 'wpca_consent';


		/**
		 * Cookie name for Cookie that will hold the Cookie Category Settings at front-end
		 *
		 * @since 3.2
		 *
		 * @var string
		 */
		const COOKIE_NAME_CC = 'wpca_cc';


		/**
		 * CSS file name
		 *
		 * @since 3.1
		 *
		 * @var string
		 */
		const FILE_NAME_CSS = 'wpca-core';


		/**
		 * Constructor
		 *
		 * @since 2.0
		 *
		 * @acces public
		 */
		public function __construct() {
			$this->doingWpcaAjaxAction = WpieAjaxHelper::doingWpieAction() && isset( $_REQUEST['wpca_action'] );
			$this->doingRenderLayoutAjax = $this->doingWpcaAjaxAction && $_REQUEST['wpca_action'] === $this->_ajaxActionRenderLayout;
			$this->doingWpcaRestAction = WpieAjaxHelper::doingRestRequest( 'wpca' );
		}


		/* (non-PHPdoc)
		 * @see WpiePluginModule::start()
		 */

		public function start()
		{
			parent::setPriority( $this->priority );
		}


		/**
		 * Helper to get one of the Cookie named that are defined as const in this class
		 *
		 * @acces public
		 *
		 * @param string $cookie, shortname for the Cookie
		 *
		 * @since 2.0
		 *
		 * @return string empty on failure or the name of the wanted Cookie
		 */
		public function getCookieName( $cookie = '' )
		{
			if ( '' === $cookie || !is_string( $cookie ) ) {
				return '';
			}

			$name = '';
			switch( $cookie ) {
				case 'consent': $name = self::COOKIE_NAME_CONSENT; break;
				case 'cc': $name = self::COOKIE_NAME_CC; break;
			}

			return $name;
		}


		/**
		 * Determine if current user IP address is from the EU
		 *
		 * @uses WpcaModuleGeo::fetch()
		 *
		 * @throws Throwable
		 * @throws Exception
		 *
		 * @since 3.2.11
		 *
		 * @return boolean|null
		 */
		public function isIpAddressFromEU()
		{
		    static $is = null;

		    if( null !== $is ) {
		        return $is;
		    }

	        try {
	            // if having a valid $record, continue
				$moduleGeo = $this->getModule( 'geo' );
				/** @var WpcaModuleGeo $moduleGeo */
	            if( $record = $moduleGeo->fetch() ) {
	                $is = $record->country->isInEuropeanUnion;
	            }
	        } catch ( \Throwable $e ) {
	            throw $e;
	        } catch( \Exception $e ) {
	            throw $e;
	        }

	        /**
	         * Filter the is In European Union retrieved from the Geo lite v2 database
	         *
	         * @var bool $is
	         * @var \GeoIp2\Model\Country $record
	         *
	         * @since 3.2.11
	         */
	        return apply_filters( 'wpca_geoip_is_in_european_union', $is, $record );
		}


		/**
		 * Callback for the wpca_styles_frontend hook
		 *
		 * Registers styles for the replacement blocks
		 *
		 * @access public
		 *
		 * @param array	$styles
		 *
		 * @since 2.3
		 */
		public function setStyles( $styles )
		{
			$relModulePath = parent::getIndex() . '/css/' . self::FILE_NAME_CSS;
			$path = $this->globals->get( 'moduleUri' ) . '/' . $relModulePath;

			$style = new stdClass();
			$style->path = $path;

			$styles[] = $style;

			return $styles;			
		}		


		/**
		 * Callback for the wpca_script_frontend_vars_before hook
		 *
		 * @access public
		 *
		 * @param array $vars
		 *
		 * @since 2.2
		 * @since 3.4.5 renamed to setScriptVars
		 *
		 * @return array
		 */
		public function setScriptVars( $vars ) {

			$vars['cookieNameConsent'] 	= $this->getCookieName( 'consent' );
			$vars['cookieNameCc'] 	= $this->getCookieName( 'cc' );

			return $vars;
		}


		/**
		 * Callback for the wpie_wp_query_args_post_type_dropdown hook
		 *
		 * Set post type for the dropdown with Cookie Policy select items
		 *
		 * @param array $args
		 *
		 * @since 3.1
		 *
		 * @return array
		 */
		public function setCookiePolicyQueryArgs( $args )
		{
			if( !$this->cookiePolicyPostType ) {
				return $args;
			}

			$args['post_type'] = $this->cookiePolicyPostType;

			return $args;
		}


		/**
		 * Determine if front-end logic must be executed
		 *
		 * @acces public
		 *
		 * @param bool $do default to true
		 *
		 * @since 2.0
		 *
		 * @return bool true or false
		 */
		public function doFrontendLogic( $do = true )
		{
			if( is_admin() ) {
				return false;
			}

			$isLocalRequest = ( WpieMiscHelper::isLocalhost() || WpieMiscHelper::isLan() );
            // is geo ip logic possible?
			$moduleGeo = $this->getModule( 'geo' );
			/** @var WpcaModuleGeo $moduleGeo */
			$isGeoLogicPossible = $moduleGeo->isGeoLogicPossible();

			if( '0' === $this->pluginsStatus ) {
			    $reason = 1;
				$do = false;
			} elseif( true === $this->disableLoggedIn && is_user_logged_in() ) {
			    $reason = 2;
				$do = false;
			} elseif( $this->disableForNonEU && !$isLocalRequest && $isGeoLogicPossible && !$this->isIpAddressFromEU() ) {
			    $reason = 3;
			    $do = false;
			} else {
			    $reason = 4;
				$do = true;
			}

			/**
			 * Let others decide to or not to do the frontend logic
			 *
			 * @param boolean $do
			 * @param int $reason
			 *
			 * @since 3.2.6
			 */
			return apply_filters( 'wpca_do_frontend_logic', $do, $reason );
		}


		/**
		 * Check if doing a frontend ajax request
		 *
		 * @since 3.4.4
		 * @return bool
		 */
		public function doingFrontendAjax() {
			return $this->doingWpcaRestAction || $this->doingWpcaAjaxAction;
		}


		/**
		 * Determine if front-end hooks must be added inside {@link WeePieFramework::init()}
		 *
		 * @acces public
		 *
		 * @uses WpcaModuleCore::doFrontendLogic()
		 * @uses WpcaModuleCore::doingFrontendAjax()
		 *
		 * @since 1.0
		 * @since 3.4.4 added WpcaModuleCore::doingFrontendAjax()
		 *
		 * @return bool
		 */
		public function doFrontendHooks( $do )
		{
			return $this->doFrontendLogic( $do ) && !$this->doingFrontendAjax();
		}


		/**
		 * Determine if 3rd party cookies are allowed before consent
		 *
		 * @since 3.1
		 *
		 * @return boolean
		 */
		public function do3rdPartyCookiesBeforeConsent()
		{
			return ( $this->cookiesBeforeConsentId3rdParties === $this->cookiesBeforeConsent );
		}


		/**
		 * Determine if Google Analytics cookies are allowed before consent
		 *
		 * @since 3.2
		 *
		 * @return boolean
		 */
		public function doGoogleAnalyticsCookiesBeforeConsent()
		{
			return ( $this->cookiesBeforeConsentIdGoogleAnalytics === $this->cookiesBeforeConsent );
		}

		/**
		 * Determine if need to disable for non EU IP Address
		 *
		 * @since 3.2.6
		 * @since 3.2.11 part of module core, priot to 3.2.11, this method was part of the geo module
		 *
		 * @return boolean
		 */
		public function doDisableForNonEUAddress()
		{
		    return ( $this->disableForNonEU && !$this->isIpAddressFromEU() );
		}

		/**
		 * Determine if NOT need to disable for non EU IP Address
		 *
		 * @since 3.2.11
		 * @since 3.2.11 part of module core, prior to 3.2.11, this method was part of the geo module
		 *
		 * @return boolean
		 */
		public function doDisableNOTForNonEUAddress()
		{
		    if( !$this->disableForNonEU ) {
		        return true;
		    } elseif ( $this->disableForNonEU && $this->isIpAddressFromEU() ) {
		        return true;
		    } else {
		        return false;
		    }
		}


		/**
		 * Determine if the accept button should be force to hide
		 *
		 * @since 3.1.2
		 *
		 * @return boolean
		 */
		public function forceHideBtnAccept()
		{
			return ( in_array( $this->consentMethod, $this->consentMethodsIdsNoBtn ) );
		}


		/**
		 * Callback for the hook wpca_settings_escape_groupfield_style
		 *
		 * Escape fields when not needed
		 *
		 * @param boolean $escape
		 * @param array $field
		 * @param array $vars
		 *
		 * @since 3.0.2
		 *
		 * @return bool
		 */
		public function escapeSettingsField( $escape, $field, $vars  )
		{
			$showLayer = $this->settingsProcessor->get('general')->get( 'general_show_layer' );
			$showCloseX = $this->settingsProcessor->get( 'general' )->get( 'general_show_x' );
			$showBtnDecline = $this->settingsProcessor->get('general')->get( 'general_show_btn_decline' );
			$showSettingsIcon = $this->settingsProcessor->get( 'general' )->get( 'general_show_icon_cc_settings' );

			switch( $vars['current_templ'] ) {
				case 'style':
					if( !$showLayer ) {
						$unsets = array( 'style_overlay_color', 'style_overlay_opacity' );
						if( in_array( WpieSettingsHelper::getFieldAttributeName( $field ), $unsets ) ) {
							return true;
						}
					}
					if( !$showCloseX ) {
						$unsets = array( 'style_close_color' );
						if( in_array( WpieSettingsHelper::getFieldAttributeName( $field ), $unsets ) ) {
							return true;
						}
					}
					if( !$showBtnDecline || $this->do3rdPartyCookiesBeforeConsent() ) {
						$unsets = array( 'style_button_decline_color',
								             'style_button_decline_color_hover',
								             'style_button_decline_text_color',
								             'style_button_decline_text_color_hover' );
						if( in_array( WpieSettingsHelper::getFieldAttributeName( $field ), $unsets ) ) {
							return true;
						}
					}
					if( !$showSettingsIcon ) {
						$unsets = array( 'style_icon_cc_settings_color' );
						if( in_array( WpieSettingsHelper::getFieldAttributeName( $field ), $unsets ) ) {
							return true;
						}
					}
					if( $this->do3rdPartyCookiesBeforeConsent() ) {
						$unsets = array( 'style_button_reset_consent_color',
								             'style_button_reset_consent_color_hover',
								             'style_button_reset_consent_text_color',
								             'style_button_reset_consent_text_color_hover' );
						if( in_array( WpieSettingsHelper::getFieldAttributeName( $field ), $unsets ) ) {
							return true;
						}
					}
					break;
				case 'content':
					if( !$showBtnDecline || $this->do3rdPartyCookiesBeforeConsent() ) {
						$unsets = array( 'content_button_decline_txt',
								             'content_button_decline_txt_declining' );
						if( in_array( WpieSettingsHelper::getFieldAttributeName( $field ), $unsets ) ) {
							return true;
						}
					}
					break;
			}

			return $escape;
		}


		/**
		 * Callback for the hook wpca_settings_escape_group_style
		 *
		 * Escape groups when not needed
		 *
		 * @param boolean $escape
		 * @param array $groupName the name of the group
		 * @param array $vars
		 *
		 * @since 3.1
		 *
		 * @return boolean
		 */
		public function escapeSettingsGroup( $escape, $groupName, $vars )
		{
			switch( $vars['current_templ'] ) {
				case 'style':
					if( 'automate_replace' === $groupName && $this->do3rdPartyCookiesBeforeConsent() ) {
						return true;
					}
					if( 'reconsider_icon' === $groupName ) {
						return !$this->settingsProcessor->get( 'general' )->get( 'general_show_reconsider_icon' );
					}

					break;
				case 'content':
					if( 'automate_replace' === $groupName && $this->do3rdPartyCookiesBeforeConsent() ) {
						return true;
					}
					if( 'reset_consent' === $groupName && $this->do3rdPartyCookiesBeforeConsent() ) {
						return true;
					}
					break;
			}

			return $escape;
		}


		/**
		 * Callback for the wpca_ajax_json_return hook
		 *
		 * @param mixed  $return
		 * @param string $action
		 * @param array  $data
		 *
		 * @since 3.1
		 *
		 * @return string|mixed $return
		 */
		public function processAjaxRequest( $return, $action = '', $data = array() )
		{
			if( $this->_ajaxActionListCookiePolicyItems !== $action && $this->_ajaxActionRenderLayout !== $action ) {
				return $return;
			}

			switch ($action) {
				case $this->_ajaxActionListCookiePolicyItems:
					$postType = ( isset( $data['post_type'] ) ) ? $data['post_type'] : $this->cookiePolicyPostType;
					$postTypeObj = new WpiePostType( $postType );

					$args = array(
							'orderby' => 'ID',
							'order' => 'DESC',
							'post_type' => $postType,
							'post_status' => 'publish'
							);

					// query for the posts
					$posts = $postTypeObj->retrieveCustom( $args, false, false );

					// flag if posts are available
					if( ! empty( $posts ) ) {
						$return = sprintf( '<option value="-1">%s</option>', __( '-- Select --', 'wpca' ) );
						foreach( $posts as $postId => $post ) {
							$isPdf = ( wp_attachment_is( 'pdf', $postId ) );
							if( 'attachment' === $postType && !$isPdf ) {
								continue;
							}
							if( 'attachment' === $postType && $isPdf ) {
								$title = basename( get_post_meta( $postId, '_wp_attached_file', true ) );
							} else {
								$title = $post->post_title;
							}
							$return .= sprintf( '<option value="%d">%s</option>', $postId, $title );
						}
					}	else {
						$return = sprintf( '<option value="-1">%s</option>', __( 'No items found for post type "'.$postType.'"', 'wpca' ) );
					}
					break;
					case $this->_ajaxActionRenderLayout:
						$return['content'] = $this->_getFrontendHTML();
					break;
			}

			return $return;
		}

		/**
		 * {@inheritDoc}
		 * @see \WpieFw\Modules\Iterator\WpieModuleInterface::init()
		 *
		 * @since 2.0
		 */
		public function init()
		{
			$generalSettings = $this->settingsProcessor->get('general');

			$onSettingsPage = $this->settingsProcessor->settingsPage->onSettingsPage;
			// load early or not
			$loadHtmlEarly = $generalSettings->get( 'general_load_html_early', true );
			// fetch method
			$fetchMethod = $generalSettings->get( 'general_fetch_method', 'adminajax' );
			// enable/disable GTM optimization
			$gtmOptimize = $generalSettings->get( 'general_gtm_optimize' );
			// set plugin status (enabled/disabled)
			$this->pluginsStatus = $generalSettings->get( 'general_plugin_status' );
			// enable/disable plugin for loggedin clients
			$this->disableLoggedIn = $generalSettings->get( 'general_disable_for_logged_in' );
			// enable/disable for non EU visitors
			$this->disableForNonEU = $generalSettings->get( 'general_disable_for_non_eu_visitors' );
			// the choosen cookies that should be allowde before consent (1,2,3)
			$this->cookiesBeforeConsent = (int) $generalSettings->get( 'general_cookies_before_consent' );
			// the choosen consent method (1,2,3,4,5,6,7)
			$this->consentMethod = (int) $generalSettings->get( 'general_consent_method' );
			// read the value of cookie wpca_consent into a class member
			$this->cookieValueConsent = WpieCookieHelper::read( $this->getCookieName( 'consent' ) );

			if( $onSettingsPage ) {
				$this->cookiePolicyPostType = $this->settingsProcessor->get('content')->get( 'content_cookie_policy_post_type' );

				if( $loadHtmlEarly && 'wpml' === WpieMultilangHelper::getActivePlugin() ) {
					if( function_exists( 'icl_set_setting' ) && class_exists( 'WPML_Cookie_Setting' )) {
						$generalSettings->set( 'general_load_html_early', false, true );
						icl_set_setting( WPML_Cookie_Setting::COOKIE_SETTING_FIELD, 1, true );

						$msg = __( 'We\'ve detected WPML is active. To correctly load your bar/box language, we automatically de-activated "Load HTML early" on the general settings and enabled WPML\'s "Language filtering for AJAX operations".', 'wpca' );
						$this->notice( $msg, 'info', $generalSettings->getName(), true );
					}
				}
			}

			if( '0' === $this->pluginsStatus && !is_network_admin() ) {
				$uri = $this->settingsProcessor->settingsPage->getTabParam( $this->settingsProcessor->settingsPage->getDefaultTabIdx(), 'uri' );
				$msg = sprintf( __( 'The %s Plugin is currently disabled.', 'wpca' ), $this->globals->get( 'pluginName' ) );
				$msg = ( $onSettingsPage ) ? $msg : $msg .' '. sprintf( __( '<a href="%s">Enable it on the settings page.</a>', 'wpca' ), $uri );
				$this->notice( $msg, 'error' );
			}

			// show a notice to the user about GA and GTM
			if( $gtmOptimize) {
			    $msg = __( 'You choosed to optimize for GTM. Please note that some configurations (e.g. "Keep anonymized Google Analytics") will not work out of the box anymore. The next step is to use our GTM event and variable to set triggers and variables for your tags', 'wpca' );
			    // add a notice with the 'once' flag to true (4th arg)
			    $this->notice( $msg, 'warning', $this->settingsProcessor->getName( 'general' ), true );
			}

			// if user seleceted REST api as fetch method
			if(  $fetchMethod === 'rest' || isset( $_REQUEST['wpca-do-rest'] ) && $_REQUEST['wpca-do-rest'] === '1' ) {
				add_filter( 'wpca_register_rest_routes', function( $routes ) {
					$routes[] = [
						'action' => 'fetch-layout',
						'callback' => function( $request ) {
							return [ 'content' => $this->_getFrontendHTML() ];
						}
					];

					return $routes;
				} );
			}

			add_action( 'wpca_styles_frontend', array( $this, 'setStyles' ) );
			add_filter( 'wpca_script_frontend_vars_before', array( $this, 'setScriptVars' ) );
			add_filter( 'wpca_do_frontend_hooks', array( $this, 'doFrontendHooks' ) );
			add_filter( "wpca_settings_escape_groupfield", array( $this, 'escapeSettingsField' ), 10, 3 );
			add_filter( "wpca_settings_escape_group", array( $this, 'escapeSettingsGroup' ), 10, 3 );
			add_filter( 'wpie_wp_query_args_post_type_dropdown', array( $this, 'setCookiePolicyQueryArgs' ) );
			add_filter( 'wpca_ajax_json_return', array( $this, 'processAjaxRequest' ), 10, 3 );

			// allow shortcodes to be added in text (custom HTML) widgets
			add_filter( 'widget_text', 'do_shortcode', 1 );

			// if PLugin is disabled or disabled for logged-in clients, stop here
			if( $this->doFrontendLogic() || $this->doingRenderLayoutAjax ) {
				define( 'WPCA_ACTIVE', true );
			}
		}

		/**
		 * Get the bar/box HTML through the frontend module
		 *
		 * @since 3.4.4
		 * @return string
		 */
		private function _getFrontendHTML()
		{
			/** @var WpcaModuleAutomate $moduleAutomate */
			$moduleAutomate = $this->getModule( 'automate' );
			$moduleAutomate->initAjaxHooksFrontend();

			/** @var WpcaModuleAutomateReplace $moduleAutomateReplace */
			$moduleAutomateReplace = $this->getModule( 'automate-replace' );
			$moduleAutomateReplace->initAjaxHooksFrontend();

			/** @var WpcaModuleFrontend $moduleFrontend */
			$moduleFrontend = $this->getModule( 'frontend' );
			$moduleFrontend->initParams();

			/** @var WpcaModuleCcpa $moduleCcpa */
			$moduleCcpa = $this->getModule( 'ccpa' );
			$moduleCcpa->initParams();

			return $moduleFrontend->render();
		}
	}

	if( !function_exists( 'wpca_disable_for_non_eu_ip_address' ) )
	{
	    /**
	     * Determine if need to disable for non EU IP Address
	     *
	     * @uses WpcaCore::disableForNonEUAddress()
	     *
	     * @since 3.2.6
	     * @since 3.2.11 part of module core, priot to 3.2.11, this function was part of the geo module
	     *
	     * @return boolean
	     */
	    function wpca_disable_for_non_eu_ip_address()
	    {
	        if( false !== ( $module = \WpieFw\modules\WpieModuleProcessor::getModule( 'wpca', 'core' ) ) )
	        {
				/** @var WpcaModuleCore $module */
	            $disable = $module->doDisableForNonEUAddress();
	            unset( $module );

	            return $disable;
	        }
	    }
	}

	if( !function_exists( 'wpca_disable_not_for_non_eu_ip_address' ) )
	{
	    /**
	     * Determine if need to disable for non EU IP Address
	     *
	     * @uses WpcaCore::doDisableNOTForNonEUAddress()
	     *
	     * @since 3.2.6
	     * @since 3.2.11 part of module core, priot to 3.2.11, this function was part of the geo module
	     *
	     * @return boolean
	     */
	    function wpca_disable_not_for_non_eu_ip_address()
	    {
	        if( false !== ( $module = \WpieFw\modules\WpieModuleProcessor::getModule( 'wpca', 'core' ) ) )
	        {
				/** @var WpcaModuleCore $module */
	            $disable = $module->doDisableNOTForNonEUAddress();
	            unset( $module );

	            return $disable;
	        }
	    }
	}
}