/**
 * Custom JS script loaded on GravityView view page
 *
 * @package   GravityView-Calendar-Ext
 * @license   GPL2+
 * @author    Katz Web Services, Inc.
 * @link      https://www.gravitykit.com
 * @copyright Copyright 2019, Katz Web Services, Inc.
 *
 * globals jQuery, gvCalendarWidget
 */

( function( $ ) {
	if ( ! window.gvCalendarWidget ) {
		return;
	}

	var widget = {
		data: window.gvCalendarWidget,

		formId: 0,

		/**
		 * Initialize widget logic
		 *
		 * @return {void}
		 */
		init: function() {
			widget.formId = widget.getViewFormId();

			var processAllWidgets = function() {
				$( '.active-drop > div[data-fieldid="gravityview-calendar"]' ).each( function() {
					widget.formId = widget.getViewFormId();

					widget.populateFeedSelection( this );
				} );
			};

			var processSingleWidget = function( e ) {
				widget.formId = widget.getViewFormId();

				widget.populateFeedSelection( $( e.target ).parents( 'div[data-fieldid="gravityview-calendar"]' ) );
			};

			$( 'body' )
				.on( 'click', '.gv-field-controls a[href=\'#settings\']', processSingleWidget )
				.on( 'dblclick', '.gv-fields', processSingleWidget )
				.on( 'gravityview_form_change', processAllWidgets );

			processAllWidgets();
		},

		/**
		 * Get form ID associated with the View
		 *
		 * @return {number}
		 */
		getViewFormId: function() {
			return parseInt( $( '#gravityview_form_id' ).val(), 10 ) || 0;
		},

		/**
		 * Popular widget feed selection based on the selected form
		 *
		 * @param {jQuery} widgetElement Widget element
		 *
		 * @return {void}
		 */
		populateFeedSelection: function( widgetElement ) {
			var $feedSelect = $( widgetElement ).find( 'div.gv-setting-container-feed select' );

			// Conditionally display empty feeds notice
			var $calendarContent = $( widgetElement ).find( '.gv-dialog-options' );
			var $emptyFeedsNotice = $calendarContent.find( '.empty-feeds-notice' );

			if ( ! widget.data.feeds[ widget.formId ] ) {
				// Since widget content is populated when it's added via AJAX request, let's hide the content and add a new container (if one doesn't already exist) for the empty feeds notice
				if ( ! $emptyFeedsNotice.length ) {
					$calendarContent.prepend( '<div class="gv-setting-container empty-feeds-notice"></div>' );

					$emptyFeedsNotice = $calendarContent.find( '.empty-feeds-notice' );
				}

				$calendarContent.children().hide();

				// Append the currently-selected Form ID to the URL in the View widget
				$emptyFeedsNotice.html( '<p>' + widget.data.translations.empty_feeds.replace( /&(amp;|#038;)?id=/, '&amp;id=' + widget.formId ) + '</p>' ).show();

				return;
			} else {
				// In case the empty feeds notice was displayed before, hide it and reveal the widget content
				$calendarContent.children().show();

				$emptyFeedsNotice.hide();
			}

			var currentSelection = $feedSelect.val();

			// Remove all but the first option ("Select a calendar") and populate with feeds associated with the form
			$feedSelect.find( 'option:gt(0)' ).remove();

			$.each( widget.data.feeds[ widget.formId ], function( i, feed ) {
				$feedSelect.append( $( '<option>', {
					value: feed.id,
					text: feed.title,
				} ) );
			} );

			// Since all options and removed and re-created every time settings are displayed, set the previous selection if it's one of the available options
			if ( $feedSelect.find( 'option[value="' + currentSelection + '"]' ).length ) {
				$feedSelect.val( currentSelection );
			} else {
				$feedSelect.val( '' );
			}
		},
	};

	widget.init();
}( jQuery ) );
