<?php
/**
 * GravityCalendar Add-On - Helpers for widget and block
 *
 * @since     1.1
 *
 * @package   GravityCalendar
 * @license   GPL2+
 * @author    Katz Web Services, Inc.
 * @link      https://www.gravitykit.com
 * @copyright Copyright 2019, Katz Web Services, Inc.
 */

class GVCalendarWidgetBlockHelpers {

	private static $forms_feeds;

	/**
	 * Returns translation strings to be used in widget/block UI
	 *
	 * @since 1.1
	 *
	 * @return array Array of strings, all sanitized by esc_html()
	 */
	static public function get_translations() {

		return array(
			'init_calendar'          => esc_html__( 'Initializing %s calendar, please wait...', 'gk-gravitycalendar' ),
			'calendar_init_failed'   => esc_html__( 'We could not initialize %s calendar.', 'gk-gravitycalendar' ),
			'select_form_and_feed'   => esc_html__( 'To display the calendar, please select a form and feed in the Block settings.', 'gk-gravitycalendar' ),
			'form'                   => esc_html__( 'Form', 'gk-gravitycalendar' ),
			'feed'                   => esc_html__( 'Feed', 'gk-gravitycalendar' ),
			'select_form'            => esc_html__( 'Select a form', 'gk-gravitycalendar' ),
			'select_feed'            => esc_html__( 'Select a feed', 'gk-gravitycalendar' ),
			'empty_forms'            => sprintf( '%s %s',
				esc_html__( 'There are no active Calendar feeds.', 'gk-gravitycalendar' ),
				sprintf( '<a href="%s">%s</a>', esc_url( admin_url( 'admin.php?page=gf_new_form' ) ), esc_html__( 'Click to create a Calendar feed.', 'gk-gravitycalendar' ) )
			),
			'empty_feeds'            => sprintf( '%s %s',
				esc_html__( 'This form does not have any Calendar feeds configured.', 'gk-gravitycalendar' ),
				sprintf( '<a href="%s">%s</a>',
					esc_url( admin_url( 'admin.php?page=gf_edit_forms&view=settings&subview=gravityview-calendar&fid=0&id=' ) ),
					esc_html__( 'Click to create a Calendar feed.', 'gk-gravitycalendar' )
				)
			),
			'calendar_settings'      => sprintf( '<a href="%s">%s</a>',
				esc_url_raw( admin_url( 'admin.php?page=gf_edit_forms&view=settings&subview=gravityview-calendar&fid=&id=' ) ),
				esc_html__( 'Edit the Calendar settings', 'gk-gravitycalendar' )
			),
			'calendar_options_error' => esc_html__( 'Calendar options could not be retrieved from the server. Please make sure that you have properly configured your calendar.', 'gk-gravitycalendar' ),
			'calendar_feed'          => esc_html__( 'Calendar Feed', 'gk-gravitycalendar' ),
			'calendar_configuration' => esc_html__( 'Calendar Configuration', 'gk-gravitycalendar' ),
		);

	}

	/**
	 * Get available forms and feeds
	 *
	 * @since 1.1
	 *
	 * @return array|false
	 */
	static public function get_forms_and_feeds() {

		if ( isset( self::$forms_feeds ) ) {
			return self::$forms_feeds;
		}

		global $wpdb;

		$feeds_table = $wpdb->prefix . 'gf_addon_feed';
		$forms_table = GFFormsModel::get_form_table_name();
		$calendar = GV_Extension_Calendar_Feed::get_instance();
		$addon_slug  = $calendar->get_slug();

		$db_results = $wpdb->get_results(
			$wpdb->prepare( "
				SELECT `form`.`id` as form_id, `form`.`title` as form_title, `feed`.`id` as feed_id, `feed`.`meta` as feed_meta 
				FROM {$feeds_table} feed 
				JOIN {$forms_table} form 
				ON `feed`.`form_id` = `form`.`id`
				WHERE 1 = 1
	        		AND `form`.`is_active` = 1 
					AND `feed`.`is_active` = 1
					AND `feed`.`addon_slug` = %s 
				ORDER BY `form`.`title` ASC", $addon_slug ),
			ARRAY_A
		);

		if ( empty( $db_results ) ) {
			self::$forms_feeds = false;

			return self::$forms_feeds;
		}

		$data = array(
			'forms'         => array(),
			'feeds'         => array(),
			'feed_form_map' => array(),
		);

		$strings = self::get_translations();

		foreach ( $db_results as $result ) {
			$feed_id    = (int) $result['feed_id'];
			$form_id    = (int) $result['form_id'];
			$form_title = $result['form_title'];
			$feed_meta  = json_decode( $result['feed_meta'], true );

			$data['forms'][ $form_id ] = array(
				'id'    => $form_id,
				'title' => $form_title
			);

			$feed = [
				'id'      => $feed_id,
				'form_id' => $form_id,
				'meta'    => $feed_meta,
			];

			$data['feeds'][ $form_id ][ $feed_id ] = [
				'id'          => $feed_id,
				'title'       => ! empty( $feed_meta['feedName'] )
					? $feed_meta['feedName']
					: sprintf( '%s %d', $strings['calendar_feed'], $feed_id ),
				'ics_enabled' => ! empty( $feed_meta['enable-ics-feed'] ) && ! empty( $feed_meta['export-ics-url'] ),
			];

			if ( $calendar->is_secure( $feed ) ) {
				$data['feeds'][ $form_id ][ $feed_id ]['secret'] = $calendar->get_validation_secret( $feed );
			}

			$data['feed_form_map'][ $feed_id ] = $form_id;
		}

		self::$forms_feeds = $data;

		return self::$forms_feeds;
	}
}
