<?php
/**
 * GravityCalendar Add-On - Widget
 *
 * @package   GravityCalendar
 * @license   GPL2+
 * @author    Katz Web Services, Inc.
 * @link      https://www.gravitykit.com
 * @copyright Copyright 2019, Katz Web Services, Inc.
 */

add_action( 'widgets_init', 'gv_calendar_register_widget' );

function gv_calendar_register_widget() {

	register_widget( 'GVCalendarWidget' );
}

class GVCalendarWidget extends WP_Widget {

	/**
	 * @var string Unique reference name for UI script
	 */
	protected $_assets_handle = 'gv_calendar_widget';

	/**
	 * GVCalendarWidget constructor.
	 *
	 * @see WP_Widget::__construct
	 */
	public function __construct() {

		$description = esc_html__( 'GravityCalendar Widget', 'gk-gravitycalendar' );

		WP_Widget::__construct(
			'gv-calendar_widget',
			__( 'GravityCalendar', 'gk-gravitycalendar' ),
			array( 'classname' => 'gv-calendar_widget', 'description' => $description ),
			array( 'width' => 200, 'height' => 250, 'id_base' => 'gv-calendar_widget' )
		);

		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Register, enqueue and localize UI assets
	 *
	 * @return void
	 */
	public function enqueue_scripts( $page ) {

		if ( $page !== 'widgets.php' ) {
			return;
		}

		$min = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) || isset( $_GET['gform_debug'] ) ? '' : '.min';

		$forms_and_feeds = GVCalendarWidgetBlockHelpers::get_forms_and_feeds();

		wp_enqueue_script( $this->_assets_handle, GV_CALENDAR_URL . 'assets/js/widget' . $min . '.js', array(), GV_CALENDAR_VERSION );

		wp_localize_script( $this->_assets_handle, 'GV_CALENDAR_WIDGET', $forms_and_feeds );
	}

	/**
	 * Handles outputting of the widget content
	 *
	 * @param array $args     Arguments provided to the widget
	 * @param array $instance Saved database values for the widget
	 */
	function widget( $args, $instance ) {

		echo $args['before_widget'];

		if ( $instance['feed_id'] ) {

			echo do_shortcode( '[gravitycalendar id="' . (int) $instance['feed_id'] . '" /]' );

		}

		echo $args['after_widget'];
	}

	/**
	 * Handles updates to the widget content
	 *
	 * @param array $new_instance The new instance of the widget
	 * @param array $old_instance The old instance of the widget
	 *
	 * @return array The widget instance, after changes have occurred
	 */
	public function update( $new_instance, $old_instance ) {

		$instance            = $old_instance;
		$instance['form_id'] = rgar( $new_instance, 'form_id' );
		$instance['feed_id'] = rgar( $new_instance, 'feed_id' );

		return $instance;
	}

	/**
	 * Outputs the form options for the widget
	 *
	 * @param array $instance The widget instance
	 *
	 * @return void
	 */
	public function form( $instance ) {

		$translations = GVCalendarWidgetBlockHelpers::get_translations();

		$forms_and_feeds = GVCalendarWidgetBlockHelpers::get_forms_and_feeds();

		if ( empty( $forms_and_feeds ) ) {
			echo "<p>{$translations['empty_forms']}</p>";

			return;
		}

		$widget_data = json_encode(
			array(
				'widget_id' => $this->id,
				'form_id'   => absint( rgar( $instance, 'form_id' ) ),
				'feed_id'   => absint( rgar( $instance, 'feed_id' ) ),
			)
		);

		$form_field_id   = esc_attr( $this->get_field_id( 'form_id' ) );
		$form_field_name = esc_attr( $this->get_field_name( 'form_id' ) );
		$feed_field_id   = esc_attr( $this->get_field_id( 'feed_id' ) );
		$feed_field_name = esc_attr( $this->get_field_name( 'feed_id' ) );

		$widget = <<<HTML
<p class="form_selection">
    <label for="{$form_field_id}">{$translations['form']}:</label>
    <select id="{$form_field_id}" name="{$form_field_name}" class="gv_calendar_feed_select widefat">
        <option value="" disabled selected>{$translations['select_form']}</option>
    </select>
</p>
<p class="feed_selection hidden" >
    <label for="{$feed_field_id}">{$translations['feed']}:</label>
    <select id="{$feed_field_id}" name="{$feed_field_name}" class="widefat">
        <option value="" disabled selected>{$translations['select_feed']}</option>
    </select>
</p>
<p class="empty_feed hidden">
    {$translations['empty_feeds']}
</p>

<script type="text/javascript">
    var GV_CALENDAR_WIDGET_DATA = window.GV_CALENDAR_WIDGET_DATA || [];
    GV_CALENDAR_WIDGET_DATA['{$this->id}'] = {$widget_data};

    if ( typeof GV_CALENDAR_WIDGET === 'object' && typeof GV_CALENDAR_WIDGET.init === 'function' ) {
        GV_CALENDAR_WIDGET.init('{$this->id}');
    }
</script>
HTML;

		echo $widget;
	}
}
