<?php

namespace GravityKit\GravityCalendar\Gutenberg\Blocks\Block;

// Exit if accessed directly
use GravityKit\GravityCalendar\Gutenberg\Blocks;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Calendar block.
 *
 * @since 1.1
 * @since 2.0 Refactored into a class.
 */
class Calendar extends Block {
	const BLOCK_NAME = 'calendar';

	static function register(): void {
		register_block_type( 'gv-calendar/' . self::BLOCK_NAME, array(
			'attributes'      => parent::get_block_attributes(),
		) );

		$_this = new self();

		add_action( 'enqueue_block_editor_assets',[ $_this, 'enqueue_assets' ], 9 );

		add_action( 'admin_head', [$_this, 'custom_metabox_styles' ] );

		add_action( 'wp_ajax_gv_calendar_get_calendar_options', [ $_this, 'get_calendar_options' ] );
	}

	/**
	 * Enqueues UI assets.
	 *
	 * @since 2.0
	 *
	 * @return void
	 */
	public function enqueue_assets(): void {
		$gv_calendar = \GV_Extension_Calendar_Feed::get_instance();
		$gv_calendar->enqueue_scripts();

		wp_print_styles( array(
			'fullcalendar-core',
			'fullcalendar-daygrid',
			'fullcalendar-timegrid',
			'fullcalendar-list',
			'gv-fullcalendar',
			'gv-calendar-tooltip',
		) );

		add_filter( 'gk/gravitycalendar/blocks/localization', function ( $localization ) {
			$localization['data']['calendar_block'] = [
				'ajax_action' => 'gv_calendar_get_calendar_options',
			];

			return $localization;
		} );
	}

	/**
	 * Fixes block overlapping the bottom meta boxes.
	 *
	 * @see   https://github.com/WordPress/gutenberg/issues/11378
	 *
	 * @since 2.0
	 */
	public function custom_metabox_styles() {

        if( ! \GV_Extension_Calendar_Feed::get_instance()->should_enqueue_assets() ) {
            return;
        }

		?>
        <!-- Style added by GravityCalendar. See https://github.com/WordPress/gutenberg/issues/11378 -->
        <style>
            .editor-writing-flow, .block-editor-writing-flow {
                height: auto;
            }

            .edit-post-layout__content .edit-post-visual-editor {
                flex-basis: auto; /** override the default flex-basis: 100%; */
            }
        </style>
		<?php
	}

	/**
	 * Handles AJAX action that fetches calendar options.
	 *
     * @since 2.0
     *
	 * @return void
	 */
	function get_calendar_options() {
		// Validate AJAX request
		$is_valid_nonce  = wp_verify_nonce( rgpost( 'nonce' ), Blocks::ASSETS_HANDLE );
		$is_valid_action = 'gv_calendar_get_calendar_options' === rgpost( 'action' );
		$form_id         = (int) rgpost( 'form_id' );
		$feed_id         = (int) rgpost( 'feed_id' );

		if ( !$is_valid_nonce || ! $is_valid_action || ! $is_valid_action || ! $form_id || ! $feed_id ) {
			// Return 'forbidden' response if nonce is invalid, otherwise it's a 'bad request'
			wp_die( false, false, array( 'response' => ( ! $is_valid_nonce ) ? 403 : 400 ) );
		}

		$gv_calendar = \GV_Extension_Calendar_Feed::get_instance();

		$options = $gv_calendar->get_calendar_instance_options( $feed_id, $form_id );

		wp_send_json_success( array(
				'_nonce'   => wp_create_nonce( 'gv-calendar-edit_' . $feed_id ),
				'feed_id'  => $feed_id,
				'form_id'  => $form_id,
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'options'  => wp_json_encode( $options ),
			)
		);
	}
}
