<?php

namespace GravityKit\GravityCalendar\Gutenberg;

use GravityKitFoundation;

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * GravityCalendar blocks.
 *
 * @since 2.0
 */
class Blocks {
	const ASSETS_HANDLE = 'gk-gravitycalendar-blocks';
	const MIN_WP_VERSION = '5.2';

	function __construct() {
		global $wp_version;

		if ( ! function_exists( 'register_block_type' ) ||
		     version_compare( $wp_version, self::MIN_WP_VERSION, '<' )
		) {
			return;
		}

		require_once( plugin_dir_path( __FILE__ ) . 'blocks/block.php' );

		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_admin_assets' ) );
		add_action( 'enqueue_block_assets', array( $this, 'enqueue_frontend_assets' ) );

		if ( version_compare( $wp_version, '5.8', '<' ) ) {
			add_filter( 'block_categories', array( $this, 'add_block_category' ) );
		} else {
			add_filter( 'block_categories_all', array( $this, 'add_block_category' ) );
		}

		$this->load_blocks();
	}

	/**
	 * Register block renderers.
	 *
	 * @since 2.0
	 *
	 * @return void
	 */
	public function load_blocks() {
		foreach ( glob( plugin_dir_path( __FILE__ ) . 'blocks/*/block.php' ) as $file ) {
			require_once( $file );

			$block_name  = basename( dirname( $file ) );
			$block_name  = explode( '-', $block_name );
			$block_name  = implode( '_', array_map( 'ucfirst', $block_name ) );
			$block_class = '\GravityKit\GravityCalendar\Gutenberg\Blocks\Block\\' . $block_name;

			if ( ! is_callable( array( $block_class, 'render' ) ) ) {
				continue;
			}

			$block_class::register();
		}
	}

	/**
	 * Add GravityCalendar category to WP block editor.
	 *
	 * @since 2.0
	 *
	 * @param array $categories
	 *
	 * @return array
	 */
	public function add_block_category( $categories ) {
		return array_merge(
			$categories,
			array(
				array( 'slug' => self::ASSETS_HANDLE, 'title' => esc_html__( 'GravityCalendar', 'gk-gravitycalendar' ) ),
			)
		);
	}

	/**
	 * Enqueue admin UI assets
	 *
	 * @since 2.0
	 *
	 * @return void
	 */
	public function enqueue_admin_assets() {
		$script     = 'assets/js/gk-gravitycalendar-blocks-admin.js';
		$style      = 'assets/css/gk-gravitycalendar-blocks-admin.css';
		$asset_file = include( GV_CALENDAR_PATH . 'assets/js/gk-gravitycalendar-blocks-admin.asset.php' );
		$handle     = self::ASSETS_HANDLE . '-admin';

		wp_enqueue_script(
			$handle,
			GV_CALENDAR_URL . $script,
			$asset_file['dependencies'],
			filemtime( GV_CALENDAR_PATH . $script )
		);

		wp_enqueue_style(
			$handle,
			GV_CALENDAR_URL . $style,
			array( 'wp-edit-blocks' ),
			filemtime( GV_CALENDAR_PATH . $style )
		);

		wp_localize_script(
			$handle,
			'GK_GRAVITYCALENDAR_BLOCKS',
			apply_filters( 'gk/gravitycalendar/blocks/localization', array(
				'home_page'  => home_url(),
				'admin_url'  => admin_url(),
				'ajax_url'   => admin_url( 'admin-ajax.php' ),
				'ajax_nonce' => wp_create_nonce( self::ASSETS_HANDLE ),
				'data'       => \GVCalendarWidgetBlockHelpers::get_forms_and_feeds(),
			) )
		);

		GravityKitFoundation::translations()->load_frontend_translations( 'gk-gravitycalendar' );
	}

	/**
	 * Enqueue frontend assets
	 *
	 * @since 2.0
	 *
	 * @return void
	 */
	public function enqueue_frontend_assets() {

		if( ! \GV_Extension_Calendar_Feed::get_instance()->should_enqueue_assets() ) {
			return;
		}

		$style  = 'assets/css/gk-gravitycalendar-blocks.css';

		\GV_Extension_Calendar_Feed::gv_calendar_clipboard_frontend_scripts();

		wp_enqueue_style(
			self::ASSETS_HANDLE,
			GV_CALENDAR_URL . $style,
			[],
			filemtime( GV_CALENDAR_PATH . $style )
		);
	}
}

new Blocks();
