<?php

if ( ! defined( 'ABSPATH' ) || ! class_exists( 'NF_Abstracts_Action' ) ) {
	exit;
}

use NinjaForms\Trello\Factories\MasterFactory;
/**
 * Class NF_Trello_Actions_Action
 */
final class NF_Trello_Actions_Action extends NF_Abstracts_Action {

	/**
	 * @var string
	 */
	protected $_name = 'trello';

	/**
	 * @var array
	 */
	protected $_tags = array();

	/**
	 * @var string
	 */
	protected $_timing = 'normal';

	/**
	 * @var int
	 */
	protected $_priority = '10';

	/**
	 * @var trello_oauth
	 */
	protected $trello;

	/** @var MasterFactory	 */
	protected $masterFactory;

	/**
	 * Constructor
	 */
	public function __construct( $access_token ) {
		parent::__construct();

		// Ensure access_token is success values, not a rejection notice
		if(!isset($access_token['oauth_token']) || !isset($access_token['oauth_token_secret'])){
			return;
		}

		$this->masterFactory =  new MasterFactory();

		if ( class_exists( 'NF_FU_File_Uploads' ) ) {
			new NF_Trello_Integrations_FileUploads_Attachments();
		}

		$this->_nicename = __( 'Trello', 'ninja-forms-trello' );

		$this->trello = $this->masterFactory->makeTrelloOauth( $access_token['oauth_token'], $access_token['oauth_token_secret'] );

		$settings = $this->get_action_settings();
		$settings = $this->edit_attachment_settings( $settings );

		$this->_settings = array_merge( $this->_settings, $settings );

		add_action( 'nf_admin_enqueue_scripts', array( $this, 'admin_assets' ) );

		add_action( 'wp_ajax_nf_trello_get_objects', array( $this, 'ajax_get_objects' ) );
	}

	protected function edit_attachment_settings( $settings ) {
		if ( ! isset( $settings['field_list_fu_email_attachments'] ) ) {
			return $settings;
		}

		$settings['field_list_fu_email_attachments']['group'] = 'primary';
		$settings['field_list_fu_email_attachments']['label'] = __( 'Attachments', 'ninja-forms-trello' );

		return $settings;
	}

	public function admin_assets() {
		if ( false === NF_Trello()->get_trello_access_token() ) {
			return;
		}

		wp_enqueue_script( 'ninja-forms-trello', plugins_url( 'assets/js/builder/controllers/actions/trello.js', NF_Trello()->plugin_file_path ), array( 'nf-builder' ) );
	}

	/**
	 * Get data from Trello
	 *
	 * @param array     $choices
	 * @param string    $type
	 * @param bool|true $select
	 *
	 * @return array
	 */
	protected function get_choices( $choices, $type = 'organization', $select = true ) {
		$choices = $this->trello->getDropdown( $choices, $type, $select );

		return $this->convert_choices( $choices );
	}

	/**
	 * Convert array to format for settings
	 *
	 * @param array $choices
	 *
	 * @return array
	 */
	protected function convert_choices( $choices ) {
		$new = array();
		foreach ( $choices as $key => $value ) {
			$new[] = array( 'label' => $value, 'value' => $key );
		}

		return $new;
	}

	/**
	 * Get the action settings
	 *
	 * @return array
	 */
	public function get_action_settings() {
		if ( ! isset( $_GET['form_id'] ) ) {
			return array();
		};

		$form_id        = $_GET['form_id'];
		$actions        = Ninja_Forms()->form( $form_id )->get_actions();
		$saved_settings = array();
		foreach ( $actions as $action ) {
			if ( 'trello' !== $action->get_setting( 'type' ) ) {
				continue;
			}

			$saved_settings = $action->get_settings();
		}

		$org_id = isset( $saved_settings['trello_organization'] ) ? $saved_settings['trello_organization'] : 0;

		$board_id = 0;
		if ( 'select' !== $org_id ) {
			$board_id = isset( $saved_settings['trello_boards'] ) ? $saved_settings['trello_boards'] : 0;
		}

		$list_id = 0;
		if ( $board_id ) {
			$list_id = isset( $saved_settings['trello_lists'] ) ? $saved_settings['trello_lists'] : 0;
		}

		$data = array(
			'org_id'   => $org_id,
			'board_id' => $board_id,
			'list_id'  => $list_id,
		);

		$settings = NF_Trello()->config( 'action-settings', $data );

		return $settings;
	}

	protected function should_process( $action_settings ) {
		$access_token = NF_Trello()->get_trello_access_token();
		if ( ! $access_token ) {
			return false;
		}

		if ( ! isset( $action_settings['trello_list'] ) || empty( $action_settings['trello_list'] ) || 0 == $action_settings['trello_list'] ) {
			return false;
		}

		if ( ! isset( $action_settings['card_name'] ) || empty( $action_settings['card_name'] ) ) {
			return false;
		}

		return $access_token;
	}

	protected function get_card_description( $action_settings ) {
		// Convert HTML to Markdown for the description
		$description = $action_settings['card_descp'];
		if ( empty( $description ) ) {
			return '';
		}

		$parser   = new NF_Trello_Lib_Parser( $description );
		$markdown = $parser->parse();

		if ( false === $markdown ) {
			return $description;
		}

		return $markdown;
	}

	protected function get_fieldset( $action_settings, $key_prefix ) {
		$data = array();
		foreach ( $action_settings as $key => $setting ) {
			if ( false === strpos( $key, $key_prefix . '_' ) ) {
				continue;
			}

			if ( '0' == $setting || 'off' == strtolower( $setting ) ) {
				continue;
			}

			$data[] = str_replace( $key_prefix . '_', '', $key );
		}

		return $data;
	}

	protected function get_card_labels( $action_settings ) {
		return $this->get_fieldset( $action_settings, 'trello_labels' );
	}

	protected function get_card_members( $action_settings ) {
		return $this->get_fieldset( $action_settings, 'trello_members' );
	}

	protected function get_file_field( $key, $fields ) {
		foreach ( $fields as $field ) {
			if ( $field['key'] === $key ) {
				return $field;
			}
		}

		return false;
	}

	protected function get_card_attachments( $action_settings, $fields ) {
		$card_files = array();
		foreach ( $action_settings as $key => $setting ) {
			if ( false === strpos( $key, 'field_list_fu_email_attachments-' ) ) {
				continue;
			}

			if ( '0' == $setting || 'off' == strtolower( $setting ) ) {
				continue;
			}

			$file_key = str_replace( 'field_list_fu_email_attachments-', '', $key );

			$field = $this->get_file_field( $file_key, $fields );
			if ( false === $field || ! isset( $field['files'] ) ) {
				continue;
			}

			foreach ( $field['files'] as $file ) {
				if ( ! isset( $file['data'] ) ) {
					continue;
				}

				if ( ! isset( $file['data']['file_path'] ) || ! file_exists( $file['data']['file_path'] ) ) {
					continue;
				}

				$card_files[] = $file['data'];
			}
		}

		return $card_files;
	}

	public function process( $action_settings, $form_id, $data ) {
		$access_token = $this->should_process( $action_settings );
		// Ensure access_token is success values, not a rejection notice
		if ( !isset($access_token['oauth_token']) || !isset($access_token['oauth_token_secret']) ) {
			return $data;
		}

		$params = array(
			'name'   => $action_settings['card_name'],
			'desc'   => $this->get_card_description( $action_settings ),
			'pos'    => $action_settings['trello_position'],
			'idList' => $action_settings['trello_list'],
		);

		// Labels
		$labels = $this->get_card_labels( $action_settings );
		if ( ! empty( $labels ) ) {
			$params['labels'] = implode( ',', $labels );
		}

		$members = $this->get_card_members( $action_settings );
		if ( is_array( $members ) && count( $members ) > 0 ) {
			$params['idMembers'] = implode( ',', $members );
		}

		$trello = $this->masterFactory->makeTrelloOauth( $access_token['oauth_token'], $access_token['oauth_token_secret'] );
		// Create card
		$card = $trello->postCard( $params );

		if ( ! empty( $action_settings['card_link'] ) ) {
			$card_links = preg_split( '/[\s]+/', $action_settings['card_link'] );
			foreach ( $card_links as $link ) {
				$trello->postCardAttachment( $card->id, array( 'url' => $link ) );
			}
		}

		// Attachment File uploads
		$card_files = $this->get_card_attachments( $action_settings, $data['fields'] );
		if ( ! empty( $card_files ) ) {
			foreach ( $card_files as $file ) {
				$this->trello->uploadAttachment( $card->id, $file['file_path'] );
			}
		}

		return $data;
	}

	public function ajax_get_objects() {
		$type = filter_input( INPUT_POST, 'type' );
		$id   = filter_input( INPUT_POST, 'id' );

		if ( ! isset( $type ) || ! isset( $id ) ) {
			wp_send_json_error( __( 'No Trello object type or ID.', 'ninja-forms-trello' ) );
		}

		$objects = $this->get_data( $type, $id );

		if ( false === $objects ) {
			wp_send_json_error( __( 'The was an error retrieving data from Trello.', 'ninja-forms-trello' ) );
		}

		wp_send_json_success( $objects );
	}

	protected function get_data( $type, $id ) {
		$access_token = NF_Trello()->get_trello_access_token();
		if ( !isset($access_token['oauth_token']) || !isset($access_token['oauth_token_secret']) ) {
			return false;
		}

		$trello = $this->masterFactory->makeTrelloOauth( $access_token['oauth_token'], $access_token['oauth_token_secret'] );
		$method = 'get' . ucfirst( $type ) . 's';
		$data   = call_user_func( array( $trello, $method ), $id );
		$data   = $trello->getDropdown( $data, $type );

		if ( 'BoardMember' === $type ) {
			$members = array();
			foreach ( $data as $member_id => $member_name ) {
				$members[] = array( 'name'  => 'trello_members_' . $member_id,
				                    'type'  => 'toggle',
				                    'label' => $member_name,
				                    'value' => $member_id,
				                    'width' => 'one-half',
				);
			}

			return $members;

		}

		return $data;
	}

}