<?php

require_once dirname( __FILE__ ) . '/oauth.php';

if ( ! class_exists( 'trello_oauth' ) ) {
	/**
	 * Trello OAuth class
	 */
	class trello_oauth {

		/* Contains the last HTTP status code returned. */
		public $http_code;
		/* Contains the last API call. */
		public $url;
		/* Set up the API root URL. */
		public $host = 'https://api.trello.com/1/';
		/* Set timeout default. */
		public $timeout = 30;
		/* Set connect timeout. */
		public $connecttimeout = 30;
		/* Verify SSL Cert. */
		public $ssl_verifypeer = false;
		/* Respons format. */
		public $format = 'json';
		/* Decode returned json data. */
		public $decode_json = true;
		/* Contains the last HTTP headers returned. */
		public $http_info;
		/* Set the useragnet. */
		public $useragent = 'Provider Oauth';
		/* Immediately retry the API call if the response was not successful. */
		//public $retry = TRUE;
		private $request_token_url = 'https://trello.com/1/OAuthGetRequestToken';
		private $access_token_url = 'https://trello.com/1/OAuthGetAccessToken';
		private $authorize_url = 'https://trello.com/1/OAuthAuthorizeToken';
		private $authenticate_token_url = '';
		private $app_name = 'Ninja Forms Trello';
		private $scope = array( 'read', 'write' );

		private $consumer_key = '21b4a1e4f755637319c979849147076e';
		private $consumer_secret = '7d5ab8655e15582041cab97459d1c6d9fca51e690247df4fb5214f2691ba786c';

		/**
		 * Set API URLS
		 */
		function accessTokenURL() {
			return $this->access_token_url;
		}

		function authenticateURL() {
			return $this->authorize_url;
		}

		function authorizeURL() {
			return $this->authorize_url;
		}

		function requestTokenURL() {
			return $this->request_token_url;
		}

		/**
		 * Debug helpers
		 */
		function lastStatusCode() {
			return $this->http_status;
		}

		function lastAPICall() {
			return $this->last_api_call;
		}

		/**
		 * Construct Provider oAuth object
		 *
		 * If consumer_key and consumer_secret are passed into __construct, use
		 * those values instead of hard-coded values
		 */
		function __construct($oauth_token = null, $oauth_token_secret = null, ?string $consumer_key = null, ?string $consumer_secret = null)
		{
			if (!is_null($consumer_key) && !is_null($consumer_secret)) {
				$this->consumer_key = $consumer_key;
				$this->consumer_secret = $consumer_secret;
			}

			$this->sha1_method = new OAuthSignatureMethod_HMAC_SHA1();
			$this->consumer    = new OAuthConsumer( $this->consumer_key, $this->consumer_secret );
			if ( ! empty( $oauth_token ) && ! empty( $oauth_token_secret ) ) {
				$this->token = new OAuthConsumer( $oauth_token, $oauth_token_secret );
			} else {
				$this->token = null;
			}
		}

		function get_authorise_url( $callback = '', $source = 'trello' ) {
			$request_token = $this->getRequestToken( $callback );

			if ( empty( $request_token ) ) {
				return false;
			}

			$_SESSION[ $source . '_oauth_token' ]        = $token = $request_token['oauth_token'];
			$_SESSION[ $source . '_oauth_token_secret' ] = $request_token['oauth_token_secret'];

			switch ( $this->http_code ) {
				case 200:
					$url = $this->getAuthorizeURL( $token );

					return $url;
					break;
			}

			return false;
		}

		/**
		 * Get a request_token from Provider
		 *
		 * @returns a key/value array containing oauth_token and oauth_token_secret
		 */
		function getRequestToken( $oauth_callback = null ) {
			$parameters = array();
			if ( ! empty( $oauth_callback ) ) {
				$parameters['oauth_callback'] = $oauth_callback;
			}
			$request     = $this->oAuthRequest( $this->requestTokenURL(), 'GET', $parameters );
			if ( empty ( $request ) ) {
				return false;
			}
			$token       = OAuthUtil::parse_parameters( $request );
			$this->token = new OAuthConsumer( $token['oauth_token'], $token['oauth_token_secret'] );

			return $token;
		}

		/**
		 * Get the authorize URL
		 *
		 * @param      $token
		 * @param bool $sign_in_with_twitter
		 *
		 * @return string
		 */
		function getAuthorizeURL( $token, $sign_in_with_twitter = true ) {
			if ( is_array( $token ) ) {
				$token = $token['oauth_token'];
			}
			$app_name = isset( $this->app_name ) ? '&name=' . urlencode( $this->app_name ) : '';
			$scope    = '';
			if ( isset( $this->scope ) ) {
				$scope = ( ! is_array( $this->scope ) ) ? array( $this->scope ) : $this->scope;
				$scope = implode( ',', $scope );
				$scope = '&scope=' . urlencode( $scope );
			}

			if ( empty( $sign_in_with_twitter ) ) {
				return $this->authorizeURL() . "?oauth_token={$token}&expiration=never" . $app_name . $scope;
			} else {
				return $this->authenticateURL() . "?oauth_token={$token}&expiration=never" . $app_name . $scope;
			}
		}

		/**
		 * Exchange request token and secret for an access token and
		 * secret, to sign API calls.
		 *
		 */
		function getAccessToken( $oauth_verifier = false, $return_uri = null ) {
			$parameters = array();
			if ( ! empty( $oauth_verifier ) ) {
				$parameters['oauth_verifier'] = $oauth_verifier;
			}
			$request     = $this->oAuthRequest( $this->accessTokenURL(), 'GET', $parameters );
			$token       = OAuthUtil::parse_parameters( $request );
			$this->token = new OAuthConsumer( $token['oauth_token'], $token['oauth_token_secret'] );

			return $token;
		}

		/**
		 * One time exchange of username and password for access token and secret.
		 *
		 */
		function getXAuthToken( $username, $password ) {
			$parameters                    = array();
			$parameters['x_auth_username'] = $username;
			$parameters['x_auth_password'] = $password;
			$parameters['x_auth_mode']     = 'client_auth';
			$request                       = $this->oAuthRequest( $this->accessTokenURL(), 'POST', $parameters );
			$token                         = OAuthUtil::parse_parameters( $request );
			$this->token                   = new OAuthConsumer( $token['oauth_token'], $token['oauth_token_secret'] );

			return $token;
		}

		/**
		 * GET wrapper for oAuthRequest.
		 */
		function get( $url, $parameters = array() ) {
			$response = $this->oAuthRequest( $url, 'GET', $parameters );
			if ( $this->format === 'json' && $this->decode_json ) {
				return json_decode( $response );
			}

			return $response;
		}

		/**
		 * POST wrapper for oAuthRequest.
		 */
		function post( $url, $parameters = array() ) {
			$response = $this->oAuthRequest( $url, 'POST', $parameters );
			if ( $this->format === 'json' && $this->decode_json ) {
				return json_decode( $response );
			}

			return $response;
		}

		/**
		 * DELETE wrapper for oAuthReqeust.
		 */
		function delete( $url, $parameters = array() ) {
			$response = $this->oAuthRequest( $url, 'DELETE', $parameters );
			if ( $this->format === 'json' && $this->decode_json ) {
				return json_decode( $response );
			}

			return $response;
		}

		/**
		 * Attach file to card
		 *
		 * @param $card_id
		 * @param $file
		 *
		 * @return array|mixed|string
		 */
		function uploadAttachment( $card_id, $file ) {
			$filename      = basename( $file );
			$url           = $this->getFormat( 'cards/' . $card_id . '/attachments/' );
			$file_contents = file_get_contents( $file );
			$request       = OAuthRequest::from_consumer_and_token( $this->consumer, $this->token, 'POST', $url, array() );
			$request->sign_request( $this->sha1_method, $this->consumer, $this->token );

			$eol           = "\r\n";
			$data          = '';
			$mime_boundary = md5( time() );
			$data .= '--' . $mime_boundary . $eol;
			$data .= 'Content-Disposition: form-data; name="file"; filename="' . $filename . '"' . $eol;
			$data .= 'Content-Type: ' . $this->_mime_type_content( $file ) . $eol . $eol;
			$data .= $file_contents . $eol;
			$data .= "--" . $mime_boundary . "--" . $eol . $eol;

			$params = array(
				'http' => array(
					'method'  => 'POST',
					'header'  => 'Content-Type: multipart/form-data; boundary=' . $mime_boundary . $eol,
					'content' => $data,
				),
			);

			$context  = stream_context_create( $params );
			$response = @file_get_contents( $request->to_url(), false, $context );

			if ( $this->format === 'json' && $this->decode_json ) {
				return json_decode( $response );
			}

			return $response;
		}

		/**
		 * Wrapper for mime_content_type
		 *
		 * @param string $file
		 *
		 * @return bool|string
		 */
		protected function _mime_type_content( $file ) {
			$realpath = realpath( $file );
			if ( $realpath && function_exists( 'finfo_file' ) && function_exists( 'finfo_open' ) && defined( 'FILEINFO_MIME_TYPE' ) ) {
				// Use the Fileinfo PECL extension (PHP 5.3+)
				return finfo_file( finfo_open( FILEINFO_MIME_TYPE ), $realpath );
			}

			if ( function_exists( 'mime_content_type' ) ) {
				// Deprecated in PHP 5.3
				return mime_content_type( $realpath );
			}

			return false;
		}

		function getFormat( $url ) {
			return "{$this->host}{$url}";
		}

		/**
		 * Format and sign an OAuth / API request
		 */
		function oAuthRequest( $url, $method, $parameters ) {
			if ( strrpos( $url, 'https://' ) !== 0 && strrpos( $url, 'http://' ) !== 0 ) {
				$url = $this->getFormat( $url );
			}
			$request = OAuthRequest::from_consumer_and_token( $this->consumer, $this->token, $method, $url, $parameters );
			$request->sign_request( $this->sha1_method, $this->consumer, $this->token );
			switch ( $method ) {
				case 'GET':
					return $this->http( $request->to_url(), 'GET' );
				default:
					return $this->http( $request->get_normalized_http_url(), $method, $request->to_postdata() );
			}
		}


		/**
		 * Format and sign an OAuth / API request
		 */
		function oAuthRequest2( $url, $method, $parameters ) {
			if ( strrpos( $url, 'https://' ) !== 0 && strrpos( $url, 'http://' ) !== 0 ) {
				$url = $this->getFormat( $url );
			}
			$defaults                 = array();
			$token                    = $this->token;
			$defaults['access_token'] = $token->key;
			$parameters               = array_merge( $defaults, $parameters );
			$request                  = new OAuthRequest( $method, $url, $parameters );
			switch ( $method ) {
				case 'GET':
					return $this->http( $request->to_url(), 'GET' );
				default:
					return $this->http( $request->get_normalized_http_url(), $method, $request->to_postdata() );
			}
		}


		/**
		 * Make an HTTP request
		 *
		 * @return API results
		 */
		function http( $url, $method, $postfields = null ) {
			$this->http_info = array();
			$ci              = curl_init();
			/* Curl settings */
			curl_setopt( $ci, CURLOPT_USERAGENT, $this->useragent );
			curl_setopt( $ci, CURLOPT_CONNECTTIMEOUT, $this->connecttimeout );
			curl_setopt( $ci, CURLOPT_TIMEOUT, $this->timeout );
			curl_setopt( $ci, CURLOPT_RETURNTRANSFER, true );
			curl_setopt( $ci, CURLOPT_HTTPHEADER, array( 'Expect:' ) );
			curl_setopt( $ci, CURLOPT_SSL_VERIFYPEER, $this->ssl_verifypeer );
			curl_setopt( $ci, CURLOPT_HEADERFUNCTION, array( $this, 'getHeader' ) );
			curl_setopt( $ci, CURLOPT_HEADER, false );

			switch ( $method ) {
				case 'POST':
					curl_setopt( $ci, CURLOPT_POST, true );
					if ( ! empty( $postfields ) ) {
						curl_setopt( $ci, CURLOPT_POSTFIELDS, $postfields );
					}
					break;
				case 'DELETE':
					curl_setopt( $ci, CURLOPT_CUSTOMREQUEST, 'DELETE' );
					if ( ! empty( $postfields ) ) {
						$url = "{$url}?{$postfields}";
					}
			}
			curl_setopt( $ci, CURLOPT_URL, $url );
			$response        = curl_exec( $ci );
			if ( curl_error( $ci ) ) {
				$this->log_error( curl_error( $ci ) );
			}
			$this->http_code = curl_getinfo( $ci, CURLINFO_HTTP_CODE );
			$this->http_info = array_merge( $this->http_info, curl_getinfo( $ci ) );
			$this->url       = $url;
			curl_close( $ci );

			return $response;
		}

		protected function log_error( $error ) {
			if ( WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'Ninja Forms Trello - oAuth error:' );
				if ( is_array( $error ) || is_object( $error ) ) {
					error_log( print_r( $error, true ) );
				} else {
					error_log( $error );
				}
			}
		}

		/**
		 * Get the header info to store.
		 */
		function getHeader( $ch, $header ) {
			$i = strpos( $header, ':' );
			if ( ! empty( $i ) ) {
				$key                       = str_replace( '-', '_', strtolower( substr( $header, 0, $i ) ) );
				$value                     = trim( substr( $header, $i + 2 ) );
				$this->http_header[ $key ] = $value;
			}

			return strlen( $header );
		}

		/**
		 * filter text
		 */
		function filter_text( $text ) {
			return trim( filter_var( $text, FILTER_SANITIZE_STRING, FILTER_FLAG_STRIP_LOW ) );
		}

		public function show_details() {
			return $this->settings;
		}

		function getMember() {
			$params = array();
			$member = $this->get( 'members/my/', $params, 0 );

			return $member;
		}

		function getOrganizations( $id = '' ) {
			$params   = array();
			$all_orgs = $this->get( 'members/my/organizations/', $params, 0 );

			return $all_orgs;
		}

		function getOrganization( $id ) {
			$params = array();
			$org    = $this->get( 'organizations/' . $id, $params, 0 );

			return $org;
		}

		function getBoards( $org ) {
			$params = array();
			if ( $org != '1' ) {
				$url = 'organizations/' . $org . '/boards/';
			} else {
				$url    = 'members/my/boards/';
				$params = array( 'filter' => 'members' );
			}

			$all_boards = $this->get( $url, $params, 0 );

			return $all_boards;
		}

		function getBoard( $id ) {
			$params = array();
			$board  = $this->get( 'boards/' . $id, $params, 0 );

			return $board;
		}

		function getBoardMembers( $id ) {
			$params  = array();
			$members = $this->get( 'boards/' . $id . '/members', $params, 0 );

			return $members;
		}

		function getLists( $board ) {
			$params    = array();
			$all_lists = $this->get( 'boards/' . $board . '/lists', $params, 0 );

			return $all_lists;
		}

		function getList( $id ) {
			$params = array();
			$list   = $this->get( 'lists/' . $id, $params, 0 );

			return $list;
		}

		function getCards( $list ) {
			$params    = array();
			$all_cards = $this->get( 'lists/' . $list . '/cards/', $params, 0 );

			return $all_cards;
		}

		function getCard( $id ) {
			$params = array();
			$card   = $this->get( 'cards/' . $id, $params, 0 );

			return $card;
		}

		function postCard( $params ) {
			$card = $this->post( 'cards/', $params );

			return $card;
		}

		function postCardAttachment( $id, $params ) {
			$attachment = $this->post( 'cards/' . $id . '/attachments/', $params );

			return $attachment;
		}

		function getDropdown( $data, $object, $select_option = true ) {
			$select = array();
			if ( $select_option && $object != 'BoardMember' ) {
				$select[0] = 'Select ' . ucfirst( $object );
			}
			if ( $object == 'organization' ) {
				$select[1] = 'My Boards';
			}
			if ( $data && is_array( $data ) ) {
				foreach ( $data as $item ) {
					if ( $object == 'member' || $object == 'BoardMember' ) {
						$name = ( isset( $item->fullName ) ) ? $item->fullName : $item->username;
					} else {
						$name = ( isset( $item->displayName ) ) ? $item->displayName : $item->name;
					}

					$select[ $item->id ] = $name;
				}
			}

			return $select;
		}
	}
}
