<?php

/**
 * Modules  manager class
 *
 * @since 1.4.2
 */

class Weforms_Modules_Controller extends Weforms_REST_Controller {

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'modules';

    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_items' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base . '/toggle', array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'toggle_modules' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args' => array(
                        'module' => array(
                            'required'          => true,
                            'description'       => __( '', 'weforms-pro' ),
                            'type'              => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ),

                        'type' => array(
                            'required'          => true,
                            'description'       => __( '', 'weforms-pro' ),
                            'type'              => 'string',
                            'sanitize_callback' => 'sanitize_text_field',
                        ),
                    )
                ),
            )
        );
    }

    /**
     * Retrieves a collection of Modules.
     *
     * @since  1.3.9
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function  get_items( $request ) {
        $modules = array(
            'all'    => weforms_pro_get_modules(),
            'active' => weforms_pro_get_active_modules()
        );

        $response = $this->prepare_response_for_collection( $modules);
        $response = rest_ensure_response( $response );

        return $response;
    }

    /**
     * Toggle Module  function
     *
     * @since 1.4.2
     *
     * @param  WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function toggle_modules( $request ) {
        $module = isset( $request['module'] ) ? $request['module'] : '';
        $type   = isset( $request['type'] ) ? $request['type'] : '';

        if ( ! $module ) {
            return new WP_Error( 'rest_invalid_request_type', __( 'Invalid module provided', 'weforms-pro' ), array('status' => 404 ) );
        }

        if ( ! in_array( $type, array( 'activate', 'deactivate' ) ) ) {
            return new WP_Error( 'rest_invalid_request_type', __( 'Invalid request type', 'weforms-pro' ), array( 'status' => 404 ) );
        }

        if ( 'all' === $module ) {
            $modules = weforms_pro_get_modules();
        } else {
            $modules = array( $module => weforms_pro_get_module( $module ) );
        }

        if ( 'activate' == $type ) {
            foreach ($modules as $module_file => $data) {
                weforms_pro_activate_module( $module_file );
            }

            $message = __( 'Activated', 'weforms-pro' );
        } else {
            foreach ($modules as $module_file => $data) {
                weforms_pro_deactivate_module( $module_file );
            }

            $message = __( 'Deactivated', 'weforms-pro' );
        }

        $data  = array(
            'message' => $module . ' ' . $message,
            'code' => ''
        );

        $response = $this->prepare_response_for_collection( $data);
        $response = rest_ensure_response( $response );

        return $response;
    }
}


