<?php

/**
 * weForms Pro Asset load
 */
class WeForms_Pro_Script_Styles {

	private $gmap_api_key = '';

	function __construct() {

		$this->gmap_api_key = weforms_get_settings( 'gmap_api' );

		// frontend and backend script/styles
		add_filter( 'weforms_admin_scripts', array( $this, 'admin_scripts' ) );
		add_filter( 'weforms_admin_styles', array( $this, 'admin_styles' ) );

		add_filter( 'weforms_frontend_scripts', array( $this, 'frontend_scripts' ) );
		add_filter( 'weforms_frontend_styles', array( $this, 'frontend_styles' ) );

		if ( $this->gmap_api_key ) {
			// dependencies
			add_action( 'weforms-form-builder-js-deps', array( $this, 'add_google_map' ) );
		}

		// mixin dependencies
		add_filter( 'weforms-builder-js-builder-stage-mixins', array( $this, 'add_builder_stage_mixin' ) );
		add_filter( 'weforms-builder-js-form-fields-mixins', array( $this, 'add_form_field_mixin' ) );

		// localizations
		add_action( 'wpuf-form-builder-localize-script', array( $this, 'localize_script' ) );
		add_filter( 'wpuf-form-builder-i18n', array( $this, 'i18n' ) );

	}

	/**
	 * Get file prefix
	 *
	 * @return string
	 */
	public function get_prefix() {
		$prefix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		return $prefix;
	}

	/**
	 * Add admin scripts
	 *
	 * @param  array $scripts
	 *
	 * @return array
	 */
	public function admin_scripts( $scripts ) {

		$prefix = $this->get_prefix();

		$pro_scripts = array(
			'weforms-pro-wpuf-form-mixins'     => array(
				'src'       => WEFORMS_PRO_ASSETS . '/pro-fields/wpuf-form-builder-mixins-pro' . $prefix . '.js',
				'deps'      => array( 'weforms-form-builder-mixins' ),
				'in_footer' => true,
			),
			'weforms-pro-wpuf-form-components' => array(
				'src'       => WEFORMS_PRO_ASSETS . '/pro-fields/wpuf-form-builder-components-pro' . $prefix . '.js',
				'deps'      => array( 'weforms-pro-wpuf-form-mixins' ),
				'in_footer' => true,
			),
			'weforms-pro-components'           => array(
				'src'       => WEFORMS_PRO_ASSETS . '/js/components' . $prefix . '.js',
				// 'deps'      => array( 'wp-color-picker' ),
				'in_footer' => true,
			),
		);


		if ( $this->gmap_api_key ) {
			$pro_scripts['weforms-google-maps'] = array(
				'src'       => 'https://maps.google.com/maps/api/js?libraries=places&key=' . $this->gmap_api_key,
				'in_footer' => true,
			);
		}

		// manage dependency, mixins should load before components
		if ( array_key_exists( 'weforms-form-builder-components', $scripts ) ) {
			array_push( $scripts['weforms-form-builder-components']['deps'], 'weforms-pro-wpuf-form-mixins' );
		}

		// manage dependency, components should load before main "weforms-app" script
		if ( array_key_exists( 'weforms-app', $scripts ) ) {
			array_push( $scripts['weforms-app']['deps'], 'weforms-pro-components' );
			array_push( $scripts['weforms-app']['deps'], 'weforms-pro-wpuf-form-components' );
		}

		return array_merge( $scripts, $pro_scripts );
	}

	/**
	 * Pro admin styles
	 *
	 * @param  array $styles
	 *
	 * @return array
	 */
	public function admin_styles( $styles ) {
		$pro_styles = array(
			'weforms-form-builder-pro' => array(
				'src'  => WEFORMS_PRO_ASSETS . '/pro-fields/wpuf-form-builder-pro.css',
				'deps' => array( 'wp-color-picker' ),
			),
			'weforms-rating-star-css'  => array(
				'src' => WEFORMS_PRO_ASSETS . '/pro-fields/css-stars.css',
			),
		);

		return array_merge( $styles, $pro_styles );
	}

	/**
	 * Frontend scripts
	 *
	 * @param  array $scripts
	 *
	 * @return array
	 */
	public function frontend_scripts( $scripts ) {

		$prefix = $this->get_prefix();

		$pro_scripts = array(
			'weforms-rating-js'            => array(
				'src'  => WEFORMS_PRO_ASSETS . '/pro-fields/jquery.barrating.min.js',
				'deps' => array( 'jquery' ),
			),
			'weforms-conditional-logic' => array(
				'src'       => WEFORMS_PRO_ASSETS . '/pro-fields/conditional-logic' . $prefix . '.js',
				'deps'      => array( 'jquery' ),
				'in_footer' => true,
			),
			'weforms-signature-lib'        => array(
				'src'       => WEFORMS_PRO_ASSETS . '/js/signature_pad.min.js',
				'deps'      => array( 'jquery' ),
				'in_footer' => true,
			),
			'weforms-signature-field'      => array(
				'src'       => WEFORMS_PRO_ASSETS . '/js/signature-field.js',
				'deps'      => array( 'jquery', 'weforms-signature-lib' ),
				'in_footer' => true,
			),
			'weforms-input-mask'           => array(
				'src'       => WEFORMS_PRO_ASSETS . '/pro-fields/jquery.mask.min.js',
				'deps'      => array( 'jquery' ),
				'in_footer' => true,
			),
			'weforms-report-js'           => array(
				'src'       => WEFORMS_PRO_ASSETS . '/js/frontend-report.min.js',
				'deps'      => array( 'jquery' ),
				'in_footer' => true,
			),
		);

		// load single min file of above scripts including chart js when script debug disabled
		// weforms-pro.min.js/weforms-pro.js file generate by grunt release command
		// check concat:frontendScripts in Gruntfile.js
		if ( ! SCRIPT_DEBUG ) {
			$pro_scripts['weforms-pro-frontend-scripts'] = array(
				'src'       => '', //WEFORMS_PRO_ASSETS . '/js/weforms-pro.min.js',
				'deps'      => array( 'jquery', 'wpuf-form' ),
				'in_footer' => true,
			);
		} else {
			$pro_scripts['weforms-pro-frontend-scripts'] = array(
				'src'       => '', //WEFORMS_PRO_ASSETS . '/js/weforms-pro.js',
				'deps'      => array( 'jquery', 'wpuf-form' ),
				'in_footer' => true,
			);
		}

		if ( $this->gmap_api_key ) {
			$pro_scripts['weforms-google-maps'] = array(
				'src'       => 'https://maps.google.com/maps/api/js?libraries=places&key=' . $this->gmap_api_key,
				'in_footer' => true,
			);
		}

		return array_merge( $scripts, $pro_scripts );
	}

	/**
	 * Frontend styles
	 *
	 * @param  array $styles
	 *
	 * @return array
	 */
	public function frontend_styles( $styles ) {
		$pro_styles = array(
			'weforms-rating-star-css' => array(
				'src' => WEFORMS_PRO_ASSETS . '/pro-fields/css-stars.css',
			),
		);

		return array_merge( $styles, $pro_styles );
	}

	/**
	 * Builder stage mixins
	 *
	 * See `WeForms_Form_Builder_Assets` class for mixin dependency: $wpuf_mixins
	 *
	 * @param array $mixins
	 *
	 * @return array
	 */
	public function add_builder_stage_mixin( $mixins ) {
		array_push( $mixins, 'mixin_builder_stage_pro' );
		array_push( $mixins, 'mixin_form_field_pro' );

		return $mixins;
	}

	/**
	 * Form field mixin
	 *
	 * See `WeForms_Form_Builder_Assets` class for mixin dependency: $wpuf_mixins
	 *
	 * @param array $mixins
	 *
	 * @return array
	 */
	public function add_form_field_mixin( $mixins ) {
		array_push( $mixins, 'mixin_form_field_pro' );

		return $mixins;
	}

	/**
	 * Add data to localize script data array
	 *
	 * @since 2.5
	 *
	 * @param array $data
	 *
	 * @return array
	 */
	public function localize_script( $data ) {
		return array_merge(
			$data,
			array(
				'gmap_api_key'               => $this->gmap_api_key,
				'is_rs_captcha_active'       => class_exists( 'ReallySimpleCaptcha' ) ? true : false,
				'countries'                  => weforms_pro_get_countries(),
				'wpuf_cond_supported_fields' => array( 'radio_field', 'checkbox_field', 'dropdown_field' ),
			)
		);
	}

	/**
	 * Add script dependencies
	 *
	 * @since 2.5
	 *
	 * @param array $deps
	 *
	 * @return array
	 */
	public function add_google_map( $deps ) {

		$deps[] = 'weforms-google-maps';

		return $deps;
	}

	/**
	 * i18n translatable strings
	 *
	 * @since 2.5
	 *
	 * @param array $i18n
	 *
	 * @return array
	 */
	public function i18n( $i18n ) {
		return array_merge(
			$i18n,
			array(
				'street_address'   => __( 'Address Line 1', 'wpuf-pro' ),
				'street_address2'  => __( 'Address Line 2', 'wpuf-pro' ),
				'city_name'        => __( 'City', 'wpuf-pro' ),
				'state'            => __( 'State', 'wpuf-pro' ),
				'zip'              => __( 'Zip Code', 'wpuf-pro' ),
				'country_select'   => __( 'Country', 'wpuf-pro' ),
				'show_all'         => __( 'Show all', 'wpuf-pro' ),
				'hide_these'       => __( 'Hide these', 'wpuf-pro' ),
				'only_show_these'  => __( 'Only show these', 'wpuf-pro' ),
				'select_countries' => __( 'Select Countries', 'wpuf-pro' ),
			)
		);
	}
}
