<?php
/**
 * File: class-updater.php
 *
 * Updater/activation class.
 *
 * @package WeForms_Pro_Updater
 */

require_once dirname( __FILE__ ) . '/libs/wedevs-updater.php';

/**
 * The Updater Class.
 */
class WeForms_Pro_Updater extends WeDevs_License_Update {
	/**
	 * Constructor.
	 *
	 * @param string $plan Plan.
	 */
	public function __construct( $plan ) {
		parent::__construct( WEFORMS_PRO_FILE, 'weForms Pro', 'weforms-pro', $plan, WEFORMS_PRO_VERSION );

		$this->base_plugin_key = 'weforms_pro';
		$this->option          = 'weforms_pro_license';
		$this->license_status  = 'weforms_pro_license_status';

		if ( ! $this->is_local_server() ) {
			add_action( 'wp_ajax_weforms_pro_get_license', array( $this, 'weforms_license_status' ) );

			add_action( 'weforms-admin-menu', array( $this, 'admin_menu' ), 10, 2 );
			add_filter( 'weforms_vue_routes', array( $this, 'vue_routes' ) );
		}
	}

	/**
	 * Add admin menu.
	 *
	 * @param string $hook       Hook.
	 * @param string $capability Capability.
	 */
	public function admin_menu( $hook, $capability ) {
		global $submenu;

		if ( current_user_can( 'manage_options' ) ) {
			$submenu['weforms'][] = array( __( 'License', 'weforms-pro' ), 'manage_options', 'admin.php?page=weforms#/license' ); // phpcs:ignore WordPress.Variables.GlobalVariables.OverrideProhibited
		}
	}

	/**
	 * Add license routes into the router.
	 *
	 * @param  array $routes Routes.
	 */
	public function vue_routes( $routes ) {
		$routes[] = array(
			'path'      => '/license',
			'name'      => 'license',
			'component' => 'License',
		);

		return $routes;
	}

	/**
	 * License status checking.
	 */
	public function weforms_license_status() {
		$data = array(
			'license' => $this->get_license_key(),
			'status'  => $this->get_license_status(),
			'message' => '',
		);

		if ( isset( $data['status']->activated ) && $data['status']->activated ) {
			$data['message'] = __( 'Congrats! License activated successfully. ', 'weforms-pro' );
		}

		if ( isset( $data['status']->update ) ) {
			$update  = strtotime( $data['status']->update );
			$expired = false;

			if ( time() > $update ) {
				// translators: 1: Expiration time ago.
				$string  = __( 'Your license has been expired %s ago', 'weforms-pro' );
				$expired = true;
			} else {
				// translators: 1: Expiration time from now.
				$string = __( 'Your license will expire in %s', 'weforms-pro' );
			}

			// translators: 1: Message abour the license expiration, 2: Expiration date.
			$data['message'] = sprintf( __( ' %1$s (%2$s).', 'weforms-pro' ), sprintf( $string, human_time_diff( $update, time() ) ), date( 'F j, Y', $update ) );
		}

		wp_send_json_success( $data );
	}

	/**
	 * Prompts the user to add license key if it's not already filled out.
	 */
	public function license_enter_notice() {
		if ( $key = $this->get_license_key() ) { // phpcs:ignore
			return;
		}
		?>
		<div id="weforms-pro-license-notice" class="notice notice-error is-dismissible">
			<p>
			<?php
			printf(
				// translators: 1: HTML anchor open tag with link, 2: HTML anchor close tag, 3: Plugin name.
				esc_html__( 'Please %1$senter%2$s your %3$s plugin license key to get regular update and support.', 'weforms-pro' ),
				'<a href="' . esc_url( admin_url( 'admin.php?page=weforms#/license' ) ) . '">',
				'</a>',
				'<strong>' . esc_html( $this->name ) . '</strong>'
			);
			?>
			</p>
		</div>
		<?php
	}
}
