<?php

/**
 * Parses module file and retrieves module metadata
 *
 * @param  string $module_file Path to module file
 *
 * @return array
 */
function weforms_pro_get_module_data( $module_file ) {
    $default_headers = array(
        'name'        => 'Plugin Name',
        'description' => 'Description',
        'plugin_uri'  => 'Plugin URI',
        'thumbnail'   => 'Thumbnail URL',
        'class'       => 'Integration Class',
        'author'      => 'Author',
        'author_uri'  => 'Author URI',
        'version'     => 'Version',
    );

    $module_data = get_file_data( $module_file, $default_headers, 'weforms_pro_modules' );

    if ( ! preg_match("/^http/", $module_data['thumbnail'] ) ) {
        $module_data['thumbnail'] = WEFORMS_PRO_ASSETS . '/images/' .$module_data['thumbnail'];
    }

    return $module_data;
}

/**
 * Gets all the available modules
 *
 * @return array
 */
function weforms_pro_get_modules() {
    $module_root  = WEFORMS_PRO_MODULES;
    $modules_dir  = @opendir( $module_root);
    $modules      = array();
    $module_files = array();

    if ( $modules_dir ) {

        while ( ( $file = readdir( $modules_dir ) ) !== false ) {

            if ( substr( $file, 0, 1 ) == '.' ) {
                continue;
            }

            if ( is_dir( $module_root . '/' . $file ) ) {
                $plugins_subdir = @opendir( $module_root . '/' . $file );

                if ( $plugins_subdir ) {

                    while ( ( $subfile = readdir( $plugins_subdir ) ) !== false ) {
                        if ( substr( $subfile, 0, 1 ) == '.' ) {
                            continue;
                        }

                        if ( substr($subfile, -4) == '.php' ) {
                            $module_files[] = "$file/$subfile";
                        }
                    }

                    closedir( $plugins_subdir );
                }
            }
        }

        closedir( $modules_dir );
    }

    if ( $module_files ) {

        foreach ( $module_files as $module_file ) {

            if ( ! is_readable( "$module_root/$module_file" ) ) {
                continue;
            }

            $module_data = weforms_pro_get_module_data( "$module_root/$module_file" );

            if ( empty ( $module_data['name'] ) ) {
                continue;
            }

            $file_base = wp_normalize_path( $module_file );


            $modules[ $file_base ] = $module_data;
        }
    }

    return $modules;
}

/**
 * Get a single module data
 *
 * @param  string $module
 *
 * @return WP_Error|Array
 */
function weforms_pro_get_module( $module ) {
    $module_root  = WEFORMS_PRO_MODULES;

    $module_data = weforms_pro_get_module_data( "$module_root/$module" );

    if ( empty ( $module_data['name'] ) ) {
        return new WP_Error( 'not-valid-plugin', __( 'This is not a valid plugin', 'weforms-pro' ) );
    }

    return $module_data;
}

/**
 * Get the meta key to store the active module list
 *
 * @return string
 */
function weforms_pro_active_module_key() {
    return 'weforms_pro_active_modules';
}

/**
 * Get active modules
 *
 * @return array
 */
function weforms_pro_get_active_modules() {
    return get_option( weforms_pro_active_module_key(), array() );
}

/**
 * Check if a module is active
 *
 * @param  string $module basename
 *
 * @return boolean
 */
function weforms_pro_is_module_active( $module ) {
    return in_array( $module, weforms_pro_get_active_modules() );
}

/**
 * Check if a module is inactive
 *
 * @param  string $module basename
 *
 * @return boolean
 */
function weforms_pro_is_module_inactive( $module ) {
    return ! weforms_pro_is_module_active( $module );
}

/**
 * Activate a module
 *
 * @param  string $module basename of the module file
 *
 * @return WP_Error|null WP_Error on invalid file or null on success.
 */
function weforms_pro_activate_module( $module ) {
    $current = weforms_pro_get_active_modules();

    $module_root = WEFORMS_PRO_MODULES;
    $module_data = weforms_pro_get_module_data( "$module_root/$module" );

    if ( empty ( $module_data['name'] ) ) {
        return new WP_Error( 'invalid-module', __( 'The module is invalid', 'weforms-pro' ) );
    }

    // activate if enactive
    if ( weforms_pro_is_module_inactive( $module ) ) {
        $current[] = $module;
        sort($current);

        update_option( weforms_pro_active_module_key(), $current);
    }

    return null;
}

/**
 * Deactivate a module
 *
 * @param  string $module basename of the module file
 *
 * @return boolean
 */
function weforms_pro_deactivate_module( $module ) {
    $current = weforms_pro_get_active_modules();

    if ( weforms_pro_is_module_active( $module ) ) {

        $key = array_search( $module, $current );

        if ( false !== $key ) {
            unset( $current[ $key ] );

            sort($current);
        }

        update_option( weforms_pro_active_module_key(), $current);

        return true;
    }

    return false;
}

add_filter( 'weforms_tracker_data', 'weforms_add_modules_data' );

/**
 * Send Modules data with tracking
 *
 * @since 1.2.5
 *
 * @param array $data
 *
 * @return array
 */
function weforms_add_modules_data( $data ) {
    $modules = weforms_pro_get_modules();

    foreach ( $modules as $key=>$module ) {
        if ( weforms_pro_is_module_active( $key ) ) {
            $data['modules']['active'][] = $module['name'];
        } else {
            $data['modules']['inactive'][] = $module['name'];
        }
    }
    return $data;
}