<?php
/**
 * Get countries
 *
 * @since 1.1.0
 *
 * @param  string $type (optional)
 *
 * @return array|string
 */
function weforms_pro_get_countries( $type = 'array' ) {
    $countries = include WEFORMS_PRO_INCLUDES . '/country-list.php';

    if ( $type == 'json' ) {
        $countries = json_encode( $countries );
    }

    return $countries;
}

/**
 * Get full list of currency codes.
 *
 * @since 2.4.2
 *
 * @return array
 */
function weforms_get_currencies() {
    $currencies = array(
        array( 'currency' => 'AED', 'label' => __( 'United Arab Emirates Dirham', 'weforms' ), 'symbol' => 'د.إ' ),
        array( 'currency' => 'AUD', 'label' => __( 'Australian Dollars', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'AZD', 'label' => __( 'Argentine Peso', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'BDT', 'label' => __( 'Bangladeshi Taka', 'weforms' ), 'symbol' => '&#2547;' ),
        array( 'currency' => 'BRL', 'label' => __( 'Brazilian Real', 'weforms' ), 'symbol' => '&#82;&#36;' ),
        array( 'currency' => 'BGN', 'label' => __( 'Bulgarian Lev', 'weforms' ), 'symbol' => '&#1083;&#1074;.' ),
        array( 'currency' => 'CAD', 'label' => __( 'Canadian Dollars', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'CLP', 'label' => __( 'Chilean Peso', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'CNY', 'label' => __( 'Chinese Yuan', 'weforms' ), 'symbol' => '&yen;' ),
        array( 'currency' => 'COP', 'label' => __( 'Colombian Peso', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'CZK', 'label' => __( 'Czech Koruna', 'weforms' ), 'symbol' => '&#75;&#269;' ),
        array( 'currency' => 'DKK', 'label' => __( 'Danish Krone', 'weforms' ), 'symbol' => 'kr.' ),
        array( 'currency' => 'DOP', 'label' => __( 'Dominican Peso', 'weforms' ), 'symbol' => 'RD&#36;' ),
        array( 'currency' => 'DZD', 'label' => __( 'Algerian Dinar', 'weforms' ), 'symbol' => 'DA;' ),
        array( 'currency' => 'EUR', 'label' => __( 'Euros', 'weforms' ), 'symbol' => '&euro;' ),
        array( 'currency' => 'HKD', 'label' => __( 'Hong Kong Dollar', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'HRK', 'label' => __( 'Croatia kuna', 'weforms' ), 'symbol' => 'Kn' ),
        array( 'currency' => 'HUF', 'label' => __( 'Hungarian Forint', 'weforms' ), 'symbol' => '&#70;&#116;' ),
        array( 'currency' => 'ISK', 'label' => __( 'Icelandic krona', 'weforms' ), 'symbol' => 'Kr.' ),
        array( 'currency' => 'IDR', 'label' => __( 'Indonesia Rupiah', 'weforms' ), 'symbol' => 'Rp' ),
        array( 'currency' => 'INR', 'label' => __( 'Indian Rupee', 'weforms' ), 'symbol' => '&#8377;' ),
        array( 'currency' => 'NPR', 'label' => __( 'Nepali Rupee', 'weforms' ), 'symbol' => 'Rs.' ),
        array( 'currency' => 'ILS', 'label' => __( 'Israeli Shekel', 'weforms' ), 'symbol' => '&#8362;' ),
        array( 'currency' => 'JPY', 'label' => __( 'Japanese Yen', 'weforms' ), 'symbol' => '&yen;' ),
        array( 'currency' => 'KIP', 'label' => __( 'Lao Kip', 'weforms' ), 'symbol' => '&#8365;' ),
        array( 'currency' => 'KRW', 'label' => __( 'South Korean Won', 'weforms' ), 'symbol' => '&#8361;' ),
        array( 'currency' => 'MYR', 'label' => __( 'Malaysian Ringgits', 'weforms' ), 'symbol' => '&#82;&#77;' ),
        array( 'currency' => 'MXN', 'label' => __( 'Mexican Peso', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'NGN', 'label' => __( 'Nigerian Naira', 'weforms' ), 'symbol' => '&#8358;' ),
        array( 'currency' => 'NOK', 'label' => __( 'Norwegian Krone', 'weforms' ), 'symbol' => '&#107;&#114;' ),
        array( 'currency' => 'NZD', 'label' => __( 'New Zealand Dollar', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'OMR', 'label' => __( 'Omani Rial', 'weforms' ), 'symbol' => 'ر.ع.' ),
        array( 'currency' => 'IRR', 'label' => __( 'Iranian Rial', 'weforms' ), 'symbol' => '﷼' ),
        array( 'currency' => 'PKR', 'label' => __( 'Pakistani Rupee', 'weforms' ), 'symbol' => 'Rs' ),
        array( 'currency' => 'PYG', 'label' => __( 'Paraguayan Guaraní', 'weforms' ), 'symbol' => '&#8370;' ),
        array( 'currency' => 'PHP', 'label' => __( 'Philippine Pesos', 'weforms' ), 'symbol' => '&#8369;' ),
        array( 'currency' => 'PLN', 'label' => __( 'Polish Zloty', 'weforms' ), 'symbol' => '&#122;&#322;' ),
        array( 'currency' => 'GBP', 'label' => __( 'Pounds Sterling', 'weforms' ), 'symbol' => '&pound;' ),
        array( 'currency' => 'RON', 'label' => __( 'Romanian Leu', 'weforms' ), 'symbol' => 'lei' ),
        array( 'currency' => 'RUB', 'label' => __( 'Russian Ruble', 'weforms' ), 'symbol' => '&#1088;&#1091;&#1073;.' ),
        array( 'currency' => 'SR', 'label'  => __( 'Saudi Riyal', 'weforms'), 'symbol' => 'SR' ),
        array( 'currency' => 'SGD', 'label' => __( 'Singapore Dollar', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'ZAR', 'label' => __( 'South African rand', 'weforms' ), 'symbol' => '&#82;' ),
        array( 'currency' => 'SEK', 'label' => __( 'Swedish Krona', 'weforms' ), 'symbol' => '&#107;&#114;' ),
        array( 'currency' => 'CHF', 'label' => __( 'Swiss Franc', 'weforms' ), 'symbol' => '&#67;&#72;&#70;' ),
        array( 'currency' => 'TWD', 'label' => __( 'Taiwan New Dollars', 'weforms' ), 'symbol' => '&#78;&#84;&#36;' ),
        array( 'currency' => 'THB', 'label' => __( 'Thai Baht', 'weforms' ), 'symbol' => '&#3647;' ),
        array( 'currency' => 'TRY', 'label' => __( 'Turkish Lira', 'weforms' ), 'symbol' => '&#8378;' ),
        array( 'currency' => 'USD', 'label' => __( 'US Dollar', 'weforms' ), 'symbol' => '&#36;' ),
        array( 'currency' => 'VND', 'label' => __( 'Vietnamese Dong', 'weforms' ), 'symbol' => '&#8363;' ),
        array( 'currency' => 'EGP', 'label' => __( 'Egyptian Pound', 'weforms' ), 'symbol' => 'EGP' ),
    );

    return apply_filters( 'weforms_currencies', $currencies );
}

/**
 * Get global currency
 *
 * @since 2.4.2
 *
 * @param  string $type
 *
 * @return mixed
 */
function weforms_get_currency( $type = '' ) {

    $currency_code = weforms_get_settings( 'currency', 'USD' );

    if ( $type == 'code' ) {
        return $currency_code;
    }

    $currencies = weforms_get_currencies();
    $index      = array_search( $currency_code, array_map( function( $element ){ return $element['currency']; }, $currencies) );
    $currency   = $currencies[ $index ];

    if ( $type == 'symbol' ) {
        return $currency['symbol'];
    }

    return $currency;
}


/**
 * Get the price format depending on the currency position.
 *
 * @return string
 */
function get_weforms_price_format() {
    $currency_pos = weforms_get_settings( 'currency_position', 'USD' );
    $format = '%1$s%2$s';

    switch ( $currency_pos ) {
        case 'left' :
            $format = '%1$s%2$s';
        break;
        case 'right' :
            $format = '%2$s%1$s';
        break;
        case 'left_space' :
            $format = '%1$s&nbsp;%2$s';
        break;
        case 'right_space' :
            $format = '%2$s&nbsp;%1$s';
        break;
    }

    return apply_filters( 'weforms_price_format', $format, $currency_pos );
}

/**
 * Return the thousand separator for prices.
 * @since  2.4.4
 * @return string
 */
function weforms_get_price_thousand_separator() {
    $separator = stripslashes( weforms_get_settings( 'price_thousand_sep', ',' ) );
    return $separator;
}

/**
 * Return the decimal separator for prices.
 * @since  2.4.4
 * @return string
 */
function weforms_get_price_decimal_separator() {
    $separator = stripslashes( weforms_get_settings( 'price_decimal_sep', '.' ) );
    return $separator;
}

/**
 * Return the number of decimals after the decimal point.
 * @since  2.4.4
 * @return int
 */
function weforms_get_price_decimals() {
    return absint( weforms_get_settings( 'price_num_decimals', 2 ) );
}

/**
 * Trim trailing zeros off prices.
 *
 * @param mixed $price
 * @return string
 */
function weforms_trim_zeros( $price ) {
    return preg_replace( '/' . preg_quote( wc_get_price_decimal_separator(), '/' ) . '0++$/', '', $price );
}

/**
 * Format the pricing number
 *
 * @since 2.4.2
 *
 * @param  number $number
 * @param  array
 *
 * @return mixed
 */
function weforms_format_price( $price, $formated = true, $args = array() ) {

    extract( apply_filters( 'weforms_price_args', wp_parse_args( $args, array(
        'currency'           => $formated ? weforms_get_currency( 'symbol' ) : '',
        'decimal_separator'  => weforms_get_price_decimal_separator(),
        'thousand_separator' => $formated ? weforms_get_price_thousand_separator() : '',
        'decimals'           => weforms_get_price_decimals(),
        'price_format'       => get_weforms_price_format()
    ) ) ) );

    $negative        = $price < 0;
    $price           = apply_filters( 'weforms_raw_price', floatval( $negative ? $price * -1 : $price ) );
    $price           = apply_filters( 'weforms_formatted_price', number_format( $price, $decimals, $decimal_separator, $thousand_separator ), $price, $decimals, $decimal_separator, $thousand_separator );

    if ( apply_filters( 'weforms_price_trim_zeros', false ) && $decimals > 0 ) {
        $price = weforms_trim_zeros( $price );
    }

    $formatted_price = ( $negative ? '-' : '' ) . sprintf( $price_format, $currency, $price );

    return apply_filters( 'weforms_format_price', $formatted_price, $price, $args );

}

function weforms_get_credit_card_names(){
    return array( 'visa','mastercard','amex','discover', 'diners', 'jcb', 'laser', 'maestro');
}

function weforms_signature_field_handler( $entry_fields, $form, $form_settings, $form_fields ) {
	$data = false;
	$signature_field_id = '';
	foreach ( array_keys( $entry_fields ) as $field_id ) {
		if ( false !== strpos( $field_id, 'signature_field' ) ) {
			$data = $entry_fields[ $field_id ];
			$signature_field_id = $field_id;
		}
	}
	if ( $data ) {
		if ( is_string( $data ) && strrpos( $data, "data:image/png;base64", - strlen( $data ) ) !== false ) {
			$data_pieces   = explode( ",", $data );
			$encoded_image = $data_pieces[1];
			$decoded_image = base64_decode( $encoded_image );

			$signature_dir = WP_CONTENT_DIR . '/uploads/weforms-signatures/';
			if ( ! file_exists( $signature_dir ) ) {
				mkdir( $signature_dir ,0777, true );
			}
			$filename    = sanitize_file_name( $form->id . "-" . time() . ".png" );
			$filepath    = wp_normalize_path( $signature_dir . $filename );
			// Writing signature
			if ( $handle = @fopen( $filepath, 'w' ) ) {
				fwrite( $handle, $decoded_image );
				fclose( $handle );
				@chmod( $filepath, 0644 );
			}

			if ( file_exists( $filepath ) ) {
				$url = wp_upload_dir()['baseurl'] . '/weforms-signatures/' . $filename;
				$entry_fields[ $field_id ] = $url;
			}
		}

	}

    return $entry_fields;

}
add_filter( 'weforms_before_entry_submission', 'weforms_signature_field_handler', 10, 4 );

/***
 *
 * Filter to add masking options to text field
 *
 * @param $text_options
 * @return array
 */
function weforms_add_mask_options_to_textfield( $text_options ) {

    $mask_options           = array(
        array(
            'name'          => 'mask_options',
            'title'         => 'Mask Options',
            'type'          => 'select',
            'options'       => array(
                'us_phone' => __( 'US Phone', 'weforms' ),
                'date'     => __( 'Date', 'weforms' ),
                'tax_id'   => __( 'Tax ID', 'weforms' ),
                'ssn'      => __( 'SSN', 'weforms' ),
                'zip'      => __( 'ZIP', 'weforms' ),
            ),
            'default'       => '',
            'section'       => 'advanced',
            'priority'      => 23,
            'help_text'     => __( 'Select this option to limit user input to unique values only. This will require that a value entered in a field does not currently exist in the entry database for that field.', 'weforms' ),
        )
    );

    return array_merge( $text_options, $mask_options );
}

add_filter( 'weforms_text_field_option_settings', 'weforms_add_mask_options_to_textfield' );

/***
 *
 * Conditional logic on submit button
 *
 * @param $form_id, $form_settings
 *
 * @since v1.3.6
 */
function conditional_logic_on_submit_button( $form_id, $form_settings ) {
    if ( !isset( $form_settings['submit_button_cond']->condition_status ) || $form_settings['submit_button_cond']->condition_status != 'yes' ) {
        return;
    }

    $cond_inputs                     = $form_settings['submit_button_cond'];
    $cond_inputs->condition_status   = isset( $cond_inputs->condition_status ) ? $cond_inputs->condition_status : '';

    if ( $cond_inputs->condition_status == 'yes' ) {
        $cond_field = array();
        $cond_operator = array();
        $cond_option = array();

        if ( isset($cond_inputs->conditions) && !empty($cond_inputs->conditions) ) {
            foreach ($cond_inputs->conditions as $condition ) {
                if ( isset($condition->name) && !empty($condition->name) ) {
                    array_push($cond_field, $condition->name);
                    array_push($cond_operator, $condition->operator);
                    array_push($cond_option, $condition->option);
                }
            }

            unset($cond_inputs->conditions);
        }

        $cond_inputs->cond_field    = $cond_field;
        $cond_inputs->cond_operator = $cond_operator;
        $cond_inputs->cond_option   = $cond_option;
        $cond_inputs->type          = 'submit';
        $cond_inputs->name          = 'submit';
        $cond_inputs->form_id       = $form_id;
        $condition                  = json_encode( $cond_inputs );

    } else {
        $condition = '';
    }

    $script = "wpuf_conditional_items.push({$condition});";
    wp_add_inline_script( 'wpuf-form', $script );
}
add_action( 'weforms_submit_btn', 'conditional_logic_on_submit_button', 10, 2 );


