<?php
/**
 * File: wedevs-updater.php
 *
 * @package WeDevs_License_Update
 *
 * phpcs:disable WordPress.VIP
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( class_exists( 'WeDevs_License_Update' ) ) {
	return;
}

/**
 * Class: WeDevs_License_Update
 *
 * WeDevs Plugin update checker.
 *
 * @author Tareq Hasan
 */
class WeDevs_License_Update {
	/**
	 * Base plugin key.
	 *
	 * @var string
	 */
	protected $base_plugin_key = 'weforms_pro';

	/**
	 * Base URL.
	 *
	 * @var string
	 */
	protected $base_url = 'https://weformspro.com/';

	/**
	 * License/update API endpoint.
	 *
	 * @var string
	 */
	protected $api_endpoint;

	/**
	 * Constructor.
	 *
	 * @param string $file       File path.
	 * @param string $name       Plugin name.
	 * @param string $product_id Product ID.
	 * @param string $slug       Plugin slug.
	 * @param string $version    Plugin version.
	 */
	public function __construct( $file, $name, $product_id, $slug, $version ) {
		// Bail out if it's a local server.
		if ( $this->is_local_server() ) {
			return;
		}

		$this->api_endpoint   = defined( 'WEFORMS_API_URL' ) ? WEFORMS_API_URL : 'https://api.boldgrid.com/api/weforms/licenses';
		$this->file           = $file;
		$this->name           = $name;
		$this->product_id     = $product_id;
		$this->slug           = $slug;
		$this->version        = $version;
		$this->option         = strtolower( str_replace( '-', '_', $this->slug ) ) . '_license';
		$this->plugin_name    = plugin_basename( $this->file );
		$this->license_status = strtolower( str_replace( '-', '_', $this->slug ) ) . '_license_status';

		if ( is_multisite() && is_main_site() ) {
			add_action( 'admin_notices', [ $this, 'license_enter_notice' ] );
			add_action( 'admin_notices', [ $this, 'license_check_notice' ] );
		} else {
			add_action( 'admin_notices', [ $this, 'license_enter_notice' ] );
			add_action( 'admin_notices', [ $this, 'license_check_notice' ] );
		}

		add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'check_update' ] );
		add_filter( 'pre_set_transient_update_plugins', [ $this, 'check_update' ] );
		add_filter( 'plugins_api', [ $this, 'check_info' ], 99, 3 );
		add_action( 'in_plugin_update_message-' . $this->plugin_name, [ $this, 'plugin_update_message' ] );
		add_action( 'wp_ajax_wedevs-license-form-action', [ $this, 'manage_license' ], 10 );
		add_action( 'wp_ajax_wedevs-license-delete-form-action', [ $this, 'delete_license' ], 10 );
	}

	/**
	 * Handle license submission.
	 *
	 * @since 1.0.0
	 **/
	public function manage_license() {
		if ( ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'wedevs-license-nonce' ) ) {
			wp_send_json_error( __( 'Nonce verification failied', 'weforms-pro' ) );
		}

		if ( isset( $_REQUEST['license_product_slug'] ) && ! empty( $_REQUEST['license_product_slug'] ) ) {
			$license_option     = strtolower( str_replace( '-', '_', $_REQUEST['license_product_slug'] ) ) . '_license';
			$license_status_key = strtolower( str_replace( '-', '_', $_REQUEST['license_product_slug'] ) ) . '_license_status';

			if ( empty( $_REQUEST['email'] ) ) {
				wp_send_json_error( __( 'Please enter your email address', 'weforms-pro' ) );
			}

			if ( empty( $_REQUEST['key'] ) ) {
				wp_send_json_error( __( 'Please enter your valid license key', 'weforms-pro' ) );
			}

			update_option(
				$license_option,
				[
					'email' => $_REQUEST['email'],
					'key'   => $_REQUEST['key'],
				]
			);
			delete_transient( $license_option );

			$license_status = get_option( $license_status_key );

			if ( ! isset( $license_status->activated ) || ! $license_status->activated ) {
				$response = $this->activation( 'activation' );

				if ( $response && isset( $response->activated ) && $response->activated ) {
					update_option( $license_status_key, $response );

					if ( isset( $response->update ) ) {
						$update  = strtotime( $response->update );
						$expired = false;

						if ( time() > $update ) {
							$greeting = __( 'Oops! license invalid.', 'weforms-pro' );
							// translators: 1: Human-readable time from now.
							$string  = __( 'has been expired %s ago', 'weforms-pro' );
							$expired = true;
						} else {
							$greeting = __( 'Congrats! License activated successfully. ', 'weforms-pro' );
							// translators: 1: Human-readable time from now.
							$string = __( 'will expire in %s', 'weforms-pro' );
						}

						$message = sprintf(
							'%1$s Your license %2$s (%3$s). ',
							$greeting,
							sprintf(
								$string,
								human_time_diff( $update, time() )
							),
							date( 'F j, Y', strtotime( $response->update ) )
						);

						if ( $expired ) {
							$message .= sprintf(
								'<a href="%1$s" target="_blank">%2$s</a>',
								'https://weformspro.com/account/',
								__( 'Renew License', 'weforms-pro' )
							);
						}
					} else {
						$message = __( 'Congrats! License activated successfully. ', 'weforms-pro' );
					}

					wp_send_json_success(
						[
							'data'    => $response,
							'message' => $message,
						]
					);
				}

				wp_send_json_success(
					[
						'data'    => $response,
						'message' => __( 'Invalid license', 'weforms-pro' ),
					]
				);
			} else {
				wp_send_json_success( $license_status );
			}
		}

		wp_send_json_error( __( 'Something went wrong', 'weforms-pro' ) );
	}

	/**
	 * Delete license
	 *
	 * @since 1.0.0
	 **/
	public function delete_license() {
		if ( ! wp_verify_nonce( $_POST['_wpnonce'], 'wedevs-license-delete-nonce' ) ) {
			return;
		}

		if ( isset( $_POST['license_product_slug'] ) && ! empty( $_POST['license_product_slug'] ) ) {
			$license_option     = strtolower( str_replace( '-', '_', $_POST['license_product_slug'] ) ) . '_license';
			$license_status_key = strtolower( str_replace( '-', '_', $_POST['license_product_slug'] ) ) . '_license_status';

			delete_option( $license_option );
			delete_transient( $license_option );
			delete_option( $license_status_key );

			wp_send_json_success( __( 'License successfully deleted', 'weforms-pro' ) );
		}

		wp_send_json_error( __( 'Something wrong, Please try again', 'weforms-pro' ) );
	}

	/**
	 * Check if the current server is localhost
	 *
	 * @return boolean
	 */
	protected function is_local_server() {
		return in_array( $this->get_ip(), [ '127.0.0.1', '::1' ], true );
	}

	/**
	 * Get current IP
	 *
	 * @since 1.0.0
	 **/
	public function get_ip() {
		switch ( true ) {
			case isset( $_SERVER['HTTP_CLIENT_IP'] ):
				$ip = $_SERVER['HTTP_CLIENT_IP'];
				break;
			case isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ):
				$ip = $_SERVER['HTTP_X_FORWARDED_FOR'];
				break;
			case isset( $_SERVER['HTTP_X_FORWARDED'] ):
				$ip = $_SERVER['HTTP_X_FORWARDED'];
				break;
			case isset( $_SERVER['HTTP_FORWARDED_FOR'] ):
				$ip = $_SERVER['HTTP_FORWARDED_FOR'];
				break;
			case isset( $_SERVER['HTTP_FORWARDED'] ):
				$ip = $_SERVER['HTTP_FORWARDED'];
				break;
			case isset( $_SERVER['REMOTE_ADDR'] ):
				$ip = $_SERVER['REMOTE_ADDR'];
				break;
			default:
				$ip = 'UNKNOWN';
				break;
		}

		return $ip;
	}

	/**
	 * Get license key
	 *
	 * @return array
	 */
	public function get_license_key() {
		return get_option( $this->option, array() );
	}

	/**
	 * Get license key
	 *
	 * @return array
	 */
	public function get_license_status() {
		return get_option( $this->license_status, array() );
	}

	/**
	 * Prompts the user to add license key if it's not already filled out.
	 *
	 * @return null
	 */
	public function license_enter_notice() {
		$key = $this->get_license_key();

		if ( $key ) {
			return;
		}
		?>
		<div class="notice notice-error is-dismissible">
			<p>
		<?php
		printf(
			// translators: 1: Link to the license form, 2: Name of this plugin.
			esc_html__( 'Please %1$s your %2$s plugin license key to get regular update and support.', 'weforms-pro' ),
			sprintf(
				'<a href="%1$s">%2$s</a>',
				esc_url( admin_url( 'admin.php?page=weforms#/license' ) ),
				esc_html__( 'enter', 'weforms-pro' )
			),
			sprintf(
				'<strong>%2$s</strong>',
				esc_html( $this->name )
			)
		);
		?>
			</p>
		</div>
		<?php
	}

	/**
	 * Check activation every 12 hours to the server.
	 *
	 * @return null
	 */
	public function license_check_notice() {
		$key = $this->get_license_key();

		if ( ! $key ) {
			return;
		}

		$error = __( ' Error: Please activate your license', 'weforms-pro' );

		$license_status = get_option( $this->license_status );

		if ( $license_status && $license_status->activated ) {
			$status = $this->get_info();

			if ( $status && $status->success ) {

				// Notice if validity expires.
				if ( isset( $status->update ) ) {
					$update = strtotime( $status->update );

					if ( time() > $update ) {
						echo '<div class="error">';
						echo '<p>Your <strong>' . esc_html( $this->plugin_name ) .
							'</strong> License has been expired. Please <a href="https://weformspro.com/account/" target="_blank">renew your license</a>.</p>';
						echo '</div>';
					}
				}
				return;
			}

			// Maybe the request didn't completed.
			if ( ! isset( $status->error ) ) {
				return;
			}

			$error = $status->error;
		}
		?>
		<div class="error">
			<p><strong><?php echo esc_html( $this->name ); ?></strong> <?php echo esc_html( $error ); ?></p>
		</div>
		<?php
	}

	/**
	 * Activation request to the plugin server.
	 *
	 * @param  string $request Request tyoe.
	 * @return object
	 */
	public function activation( $request = 'check' ) {
		global $wp_version;

		$license = $this->get_license_key();

		if ( ! $license ) {
			return;
		}

		if ( is_multisite() ) {
			$wp_install = network_site_url();
		} else {
			$wp_install = home_url( '/' );
		}

		list( $plugin_name, $plugin_version) = $this->get_current_plugin_info();

		$params = [
			'timeout'    => ( ( defined( 'DOING_CRON' ) && DOING_CRON ) ? 30 : 3 ),
			'user-agent' => 'WordPress/' . $wp_version . '; ' . home_url( '/' ),
			'body'       => [
				'wc-api'      => 'bg-api',
				'request'     => $request,
				'email'       => $license['email'],
				'license_key' => $license['key'],
				'product_id'  => $this->product_id,
				'instance'    => home_url(),
				'site_url'    => $wp_install,
				'slug'        => $this->slug,
				'version'     => $plugin_version,
				'wp_version'  => $wp_version,
				'php_version' => phpversion(),
			],
		];

		$response = wp_remote_post( $this->api_endpoint, $params );
		$update   = wp_remote_retrieve_body( $response );

		if ( is_wp_error( $response ) || 200 !== $response['response']['code'] ) {
			if ( is_wp_error( $response ) ) {
				echo '<div class="error"><p><strong>' . esc_html( $this->name ) .
					' Activation Error:</strong> ' . esc_html( $response->get_error_message() ) .
					'</p></div>';
				return false;
			}

			if ( 200 !== $response['response']['code'] ) {
				echo '<div class="error"><p><strong>' . esc_html( $this->name ) .
					' Activation Error:</strong> ' . esc_html( $response['response']['code'] ) .
					' - ' . esc_html( $response['response']['message'] ) . '</p></div>';
				return false;
			}

			printf(
				'<pre>%1$s</pre>',
				print_r( $response, true ) // phpcs:ignore
			);
		}

		return json_decode( $update );
	}

	/**
	 * Integrates into plugin update api check.
	 *
	 * @param  \stdClass $transient Update transient.
	 * @return \stdClass
	 */
	public function check_update( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		$remote_info = $this->get_info();

		if ( empty( $remote_info->latest ) ) {
			return $transient;
		}

		list( $plugin_name, $plugin_version) = $this->get_current_plugin_info();

		if ( version_compare( $plugin_version, $remote_info->latest, '<' ) ) {
			$obj              = new stdClass();
			$obj->slug        = $this->slug;
			$obj->new_version = $remote_info->latest;
			$obj->url         = $this->base_url;

			if ( isset( $remote_info->latest_url ) ) {
				$obj->package = $remote_info->latest_url;
			}

			$basefile = plugin_basename( $this->file );

			$transient->response[ $basefile ] = $obj;
		}

		return $transient;
	}

	/**
	 * Plugin changelog information popup.
	 *
	 * @param  \stdClass $transient WordPress plugin update transient.
	 * @param  string    $action    Action name.
	 * @param  array     $args      Optional arguments.
	 * @return \stdClass|bool
	 */
	public function check_info( $transient, $action, $args = [] ) {
		if ( 'plugin_information' !== $action ) {
			return $transient;
		}

		if ( $this->slug === $args->slug ) {
			$remote_info      = $this->get_info();
			$obj              = new \stdClass();
			$obj->name        = $this->name;
			$obj->slug        = $this->slug;
			$obj->new_version = $remote_info->latest;
			$obj->sections    = empty( $remote_info->sections ) ?
				[] : (array) $remote_info->sections;

			if ( isset( $remote_info->latest_url ) ) {
				$obj->download_link = $remote_info->latest_url;
			}

			return $obj;
		}

		return $transient;
	}

	/**
	 * Collects current plugin information.
	 *
	 * @return array
	 */
	public function get_current_plugin_info() {
		require_once ABSPATH . '/wp-admin/includes/plugin.php';

		$plugin_data    = get_plugin_data( $this->file );
		$plugin_name    = $plugin_data['Name'];
		$plugin_version = $plugin_data['Version'];

		return [
			$plugin_name,
			$plugin_version,
		];
	}

	/**
	 * Get license status and plugin update information.
	 *
	 * @return object
	 */
	public function get_info() {
		$status = get_transient( $this->option );
		if ( false === $status ) {
			$status   = $this->activation();
			$duration = 60 * 60 * 12; // 12 hour

			set_transient( $this->option, $status, $duration );
		}

		return $status;
	}

	/**
	 * Show plugin udpate message
	 *
	 * @since  1.0.0
	 *
	 * @param array $args Arguments.
	 */
	public function plugin_update_message( $args ) {
		$cache_key    = md5( $this->base_plugin_key . '_plugin_' . sanitize_key( $this->plugin_name ) . '_version_info' );
		$version_info = get_transient( $cache_key );

		if ( false === $version_info ) {
			$version_info = $this->get_info();
			set_transient( $cache_key, $version_info, 3600 );
		}

		if ( version_compare( $this->version, $version_info->latest, '<' ) && empty( $version_info->latest_url ) ) {
			$upgrade_notice = sprintf(
				'</p><p id="weforms-plugin-upgrade-notice" class="%1$s-plugin-upgrade-notice">Please <a href="%2$s" target="_blank">activate</a> your license key for getting regular updates and support',
				$this->base_plugin_key,
				esc_url( admin_url( 'admin.php?page=weforms#/license' ) )
			);
			echo apply_filters( $this->product_id . '_in_plugin_update_message', wp_kses_post( $upgrade_notice ) ); // phpcs:ignore
		}
	}
}
