<?php

/**
 * Campaign Monitor API  manager class
 *
 * @since 1.4.2
 */

class WeForms_Campaign_Monitor_Controller extends Weforms_REST_Controller {

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'campaignmonitor';

    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'campaign_fetch_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'campaign_update_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );
    }

        /**
     * get Collection of Campaign Lists
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function campaign_fetch_lists( $request ) {
        $lists    = get_option( 'wpuf_cm_lists', array( ) );
        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function campaign_update_lists( $request ) {
        $api_key = $this->get_api_key();

        if ( empty( $api_key ) ) {
            return new WP_Error( 'rest_weforms_campaign_invalid_api_key', __( 'Campaign Monitor API key is missing', 'weforms-pro') , array( 'status' => 404 ) );
        }

        $this->require_campaign_monitor();

        $auth         = array( 'api_key' => $api_key );
        $client_lists = $list_object = $lists = array();
        $cm_general   = new CS_REST_General($auth);
        $result       = $cm_general->get_clients();

        if ( $result ) {
            foreach ( $result->response as $client ) {
                if ( !class_exists('CS_REST_Clients') ) {
                    require_once dirname( __FILE__ ) . '/../cm-php-sdk/csrest_clients.php';
                }
                $client_class = new CS_REST_Clients( $client->ClientID, $auth );
                $client_lists[] = $client_class->get_lists();
            }
        }

        foreach ( $client_lists as $list ) {
            foreach ($list->response as $list_obj) {
                $list_object[] = $list_obj;
            }
        }

        foreach ($list_object as $list) {
            $lists[] = array(
                'id'     => $list->ListID,
                'name'   => $list->Name,
            );
        }

        update_option( 'wpuf_cm_lists', $lists );

        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'campaign_monitor_api_key' );

        return $api_key;
    }

    /**
     * Require the campaign monitor class if not exists
     *
     * @return void
     */
    public function require_campaign_monitor() {
        if ( ! class_exists( 'CS_REST_General' ) ) {
            require_once dirname( __FILE__ ) . '/../cm-php-sdk/csrest_general.php';
        }
    }
}
