<?php

/**
 * Integration class
 */
class WeForms_Integration_Campaign_Monitor extends WeForms_Pro_Integration {

    function __construct() {

        $this->id                = 'campaign-monitor';
        $this->title             = __( 'Campaign Monitor', 'weforms-pro' );
        $this->icon              = WEFORMS_ASSET_URI . '/images/icon-campaign-monitor.svg';
        $this->template          = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled' => false,
            'list'    => '',
            'double'  => false,
            'fields'  => array(
                'email'      => '',
                'first_name' => '',
                'last_name'  => ''
            )
        );

        $this->load_settings();

        add_action( 'admin_footer', array( $this, 'load_template' ) );

        add_action( 'wp_ajax_wpuf_campaign_monitor_fetch_lists', array( $this, 'fetch_lists' ) );
        add_action( 'wp_ajax_wpuf_campaign_monitor_update_lists', array( $this, 'update_lists' ) );

        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );

        add_action( 'weforms_entry_submission', array( $this, 'subscribe_user' ), 10, 4 );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scritps
     *
     * @return array
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-campaign-monitor'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['campaign-monitor'] = array(
            'label' => __( 'Campaign Monitor', 'weforms-pro' ),
            'icon'  => WEFORMS_ASSET_URI . '/images/integrations/campaign-monitor2.svg',
        );

        return $tabs;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'campaign_monitor_api_key' );

        return $api_key;
    }

    /**
     * Fetch campaign monitors saved list from server
     *
     * @return array
     */
    public function fetch_lists() {
        $lists = get_option( 'wpuf_cm_lists', array() );

        wp_send_json_success( $lists );
    }

    /**
     * Require the campaign monitor class if not exists
     *
     * @return void
     */
    public function require_campaign_monitor() {
        if ( ! class_exists( 'CS_REST_General' ) ) {
            require_once dirname( __FILE__ ) . '/cm-php-sdk/csrest_general.php';
        }
    }

    /**
     * Fetch the udpated list from campaign-monitor, save and return it
     *
     * @return array
     */
    public function update_lists() {

        $api_key = $this->get_api_key();

        if ( empty( $api_key ) ) {
            wp_send_json_error( __( 'Campaign Monitor API key is missing', 'weforms-pro' ) );
        }

        $this->require_campaign_monitor();
        $auth = array( 'api_key' => $api_key );
        $client_lists = $list_object = $lists = array();

        $cm_general = new CS_REST_General($auth);
        $result = $cm_general->get_clients();

        if ( $result ) {
            foreach ( $result->response as $client ) {
                if ( !class_exists('CS_REST_Clients') ) {
                    require_once dirname( __FILE__ ) . '/cm-php-sdk/csrest_clients.php';
                }
                $client_class = new CS_REST_Clients( $client->ClientID, $auth );
                $client_lists[] = $client_class->get_lists();
            }
        }

        foreach ( $client_lists as $list ) {
            foreach ($list->response as $list_obj) {
                $list_object[] = $list_obj;
            }
        }

        foreach ($list_object as $list) {
            $lists[] = array(
                'id'     => $list->ListID,
                'name'   => $list->Name,
            );
        }

        update_option( 'wpuf_cm_lists', $lists );
        wp_send_json_success( $lists );
    }

    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function subscribe_user( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->list ) || empty( $integration->fields->email ) ) {
            return;
        }

        $email = WeForms_Notification::replace_field_tags( $integration->fields->email, $entry_id );

        if ( empty( $email ) ) {
            return;
        }

        $first_name = WeForms_Notification::replace_name_tag( $integration->fields->first_name, $entry_id );
        $last_name  = WeForms_Notification::replace_name_tag( $integration->fields->last_name, $entry_id );

        $this->require_campaign_monitor();
        $auth = array( 'api_key' => $this->get_api_key() );

        if (!class_exists('CS_REST_Subscribers')) {
            require_once dirname( __FILE__ ) . '/cm-php-sdk/csrest_subscribers.php';
        }
        $wrap = new CS_REST_Subscribers( $integration->list, $auth );
        $result = $wrap->add(array(
            'EmailAddress' => $email,
            'Name' => $first_name . ' ' . $last_name,
            'CustomFields' => array(),
            'Resubscribe' => true
        ));
    }
}
