<?php

/**
 * Constatnt Contact API  manager class
 *
 * @since 1.4.2
 */

use Ctct\ConstantContact;
use Ctct\Components\Contacts\Contact;

class WeForms_Constant_Contact_Controller extends Weforms_REST_Controller {

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'constantcontact';

    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'constant_contact_fetch_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'constant_contact_update_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );
    }

    /**
     * get Collection of Constant Contact Lists
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function constant_contact_fetch_lists( $request ) {
        $lists    = get_option( 'wpuf_cc_lists', array( ) );
        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function constant_contact_update_lists( $request ) {
        $api_key = $this->get_api_key();

        if ( empty( $api_key ) ) {
            return new WP_Error( 'rest_weforms_constant_contact_invalid_api_key', __( 'Constant Contact API key is missing', 'weforms-pro') , array( 'status' => 404 ) );
        }

        $this->require_constant_contact();

        $lists        = array();
        $ctct         = new ConstantContact( $api_key );
        $contact_list = $ctct->listService->getLists($this->get_access_token());

        if ( $contact_list ) {
            foreach ($contact_list as $list) {
                $lists[] = array(
                    'id'     => $list->id,
                    'name'   => $list->name,
                );
            }

            update_option( 'wpuf_cc_lists', $lists );
        }

        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'constant_contact_api_key' );

        return $api_key;
    }

    /**
     * Get the Access Token
     *
     * @return string
     */
    private function get_access_token() {
        $access_token = weforms_get_settings( 'constant_contact_access_token' );

        return $access_token;
    }

    /**
     * Require the constant contact class if not exists
     *
     * @return void
     */
    public function require_constant_contact() {
        if ( ! class_exists( 'ConstantContact' ) ) {
            require_once dirname( __FILE__ ) . '/../library/autoload.php';
        }
    }
}
