<?php

/**
 * Integration class
 */
use Ctct\ConstantContact;
use Ctct\Components\Contacts\Contact;

class WeForms_Integration_Constant_Contact extends WeForms_Pro_Integration {

    function __construct() {

        $this->id                = 'constant-contact';
        $this->title             = __( 'Constant Contact', 'weforms-pro' );
        $this->icon              = WEFORMS_ASSET_URI . '/images/icon-constant-contact.svg';
        $this->template          = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled' => false,
            'list'    => '',
            'double'  => false,
            'fields'  => array(
                'email'      => '',
                'first_name' => '',
                'last_name'  => ''
            )
        );

        $this->load_settings();

        add_action( 'admin_footer', array( $this, 'load_template' ) );

        add_action( 'wp_ajax_wpuf_constant_contact_fetch_lists', array( $this, 'fetch_lists' ) );
        add_action( 'wp_ajax_wpuf_constant_contact_update_lists', array( $this, 'update_lists' ) );

        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );

        add_action( 'weforms_entry_submission', array( $this, 'subscribe_user' ), 10, 4 );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scritps
     *
     * @return array
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-constant-contact'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'constant_contact_api_key' );

        return $api_key;
    }

    /**
     * Get the Access Token
     *
     * @return string
     */
    private function get_access_token() {
        $access_token = weforms_get_settings( 'constant_contact_access_token' );

        return $access_token;
    }

    /**
     * Fetch constant contacts saved list from server
     *
     * @return array
     */
    public function fetch_lists() {
        $lists = get_option( 'wpuf_cc_lists', array() );

        wp_send_json_success( $lists );
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['constant-contact'] = array(
            'label' => __( 'Constant Contact', 'weforms-pro' ),
            'icon'  => WEFORMS_ASSET_URI . '/images/integrations/constant-contact2.svg',
        );

        return $tabs;
    }

    /**
     * Require the constant contact class if not exists
     *
     * @return void
     */
    public function require_constant_contact() {
        if ( ! class_exists( 'ConstantContact' ) ) {
            require_once dirname( __FILE__ ) . '/library/autoload.php';
        }
    }

    /**
     * Fetch the udpated list from constant contact, save and return it
     *
     * @return array
     */
    public function update_lists() {

        $api_key = $this->get_api_key();

        if ( empty( $api_key ) ) {
            wp_send_json_error( __( 'Constant Contact API key is missing', 'weforms-pro' ) );
        }

        $this->require_constant_contact();
        $ctct = new ConstantContact( $api_key );
        $contact_list = $ctct->listService->getLists($this->get_access_token());

        $lists     = array();

        if ( $contact_list ) {
            foreach ($contact_list as $list) {
                $lists[] = array(
                    'id'     => $list->id,
                    'name'   => $list->name,
                );
            }

            update_option( 'wpuf_cc_lists', $lists );
        }

        wp_send_json_success( $lists );
    }

    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function subscribe_user( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->list ) || empty( $integration->fields->email ) ) {
            return;
        }

        $email = WeForms_Notification::replace_field_tags( $integration->fields->email, $entry_id );

        if ( empty( $email ) ) {
            return;
        }

        $first_name = WeForms_Notification::replace_name_tag( $integration->fields->first_name, $entry_id );
        $last_name  = WeForms_Notification::replace_name_tag( $integration->fields->last_name, $entry_id );

        $this->require_constant_contact();

        $ctct = new ConstantContact($this->get_api_key());
        $action = "Creating Contact";
        $contact = new Contact();
        $contact->addEmail( $email );
        $contact->addList( $integration->list );
        $contact->first_name = $first_name;
        $contact->last_name = $last_name;
        $ctct->contactService->addContact( $this->get_access_token(), $contact, 'ACTION_BY_VISITOR' );
    }
}
