<?php

/**
 * Convertkit API  manager class
 *
 * @since 1.4.2
 */

class WeForms_Convertkit_Controller extends Weforms_REST_Controller {

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'convertkit';

    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'convertkit_fetch_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'convertkit_update_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );
    }

    /**
     * get Collection of ConvertKit  Lists
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function convertkit_fetch_lists( $request ) {
        $lists    = get_option( 'wpuf_ck_lists', array( ) );
        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function convertkit_update_lists( $request ) {
        $this->require_convertkit();

        $lists      = array();
        $convertkit = new ConvertKit( $this->get_api_key(), $version = 'v3' );
        $response   = $convertkit->getForms();

        if ( $response ) {
            $lists = json_decode( $response,true );

            foreach ($lists['forms'] as $key => $list) {
                $lists[] = array(
                    'id'     => $list['id'],
                    'name'   => $list['name']
                );
            }

            update_option( 'wpuf_ck_lists', $lists );
        }

        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    /**
     * Require the convertkit class if not exists
     *
     * @return void
     */
    public function require_convertkit() {
        if ( ! class_exists( 'ConvertKit' ) ) {
            require_once dirname( __FILE__ ) . '/../class-convertkit.php';
        }
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'convertkit_api_key' );

        return $api_key;
    }
}
