<?php

/**
 * ConvertKit Integration
 */
class WeForms_Integration_ConvertKit extends WeForms_Pro_Integration {

    function __construct() {

        $this->id              = 'convertkit';
        $this->title           = __( 'ConvertKit', 'weforms-pro' );
        $this->icon            = WEFORMS_ASSET_URI . '/images/icon-convertkit.svg';
        $this->template        = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled' => false,
            'list'    => '',
            'double'  => false,
            'fields'  => array(
                'email'      => '',
                'first_name' => '',
                'last_name'  => ''
            )
        );

        $this->load_settings();

        add_filter( 'admin_footer', array( $this, 'load_template' ) );

        add_action( 'wp_ajax_wpuf_convertkit_fetch_lists', array( $this, 'fetch_lists' ) );
        add_action( 'wp_ajax_wpuf_convertkit_update_lists', array( $this, 'update_lists' ) );

        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );

        add_action( 'weforms_entry_submission', array( $this, 'subscribe_user' ), 10, 4 );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scritps
     *
     * @return array
     */
    public function enqueue_mixin( $scripts ) {
        $scripts['weforms-int-convertkit'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'convertkit_api_key' );

        return $api_key;
    }

    /**
     * Fetch convertkits saved list from server
     *
     * @return array
     */
    public function fetch_lists() {
        $lists = get_option( 'wpuf_ck_lists', array() );

        wp_send_json_success( $lists );
    }

    /**
     * Require the convertkit class if not exists
     *
     * @return void
     */
    public function require_convertkit() {
        if ( ! class_exists( 'ConvertKit' ) ) {
            require_once dirname( __FILE__ ) . '/class-convertkit.php';
        }
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['convertkit'] = array(
            'label' => __( 'ConvertKit', 'weforms-pro' ),
            'icon'  => WEFORMS_ASSET_URI . '/images/integrations/convertkit2.svg',
        );

        return $tabs;
    }

    /**
     * Fetch the udpated list from convertkit, save and return it
     *
     * @return array
     */
    public function update_lists() {
        $this->require_convertkit();

        $lists      = array();
        $convertkit = new ConvertKit( $this->get_api_key(),$version = 'v3' );
        $response   = $convertkit->getForms();

        if ( $response ) {
            $response = json_decode( $response,true );

            foreach ($response['forms'] as $list) {
                $lists[] = array(
                    'id'     => $list['id'],
                    'name'   => $list['name']
                );
            }

            update_option( 'wpuf_ck_lists', $lists );
        }

        wp_send_json_success( $lists );
    }

    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function subscribe_user( $entry_id, $form_id, $page_id, $form_settings ) {
        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->list ) || empty( $integration->fields->email ) ) {
            return;
        }

        $email = WeForms_Notification::replace_field_tags( $integration->fields->email, $entry_id );

        if ( empty( $email ) ) {
            return;
        }

        $first_name = WeForms_Notification::replace_name_tag( $integration->fields->first_name, $entry_id );
        $last_name  = WeForms_Notification::replace_name_tag( $integration->fields->last_name, $entry_id );
        $name       = $first_name . ' ' . $last_name;

        $this->require_convertkit();

        $convertkit = new ConvertKit( $this->get_api_key(),$version = 'v3' );

        $convertkit->subscribeToAForm( $integration->list, $email, $name, $integration->double );
    }
}

