<?php

/**
 * Google Analytics Tracker CLass
 */
class WeDevs_Google_Analytics_Tracker {

    /**
     * API endpoint
     *
     * @var string
     */
    private $api_endpoint   = 'http://www.google-analytics.com';

    /**
     * Google Analytics Tracking ID
     *
     * @var string
     */
    private $tracking_id    = null;

    /**
     * Configuration / Event data
     *
     * @var array
     */
    private $form_values    = array();

    /**
     * Create a new instance
     *
     * @param string $tracking_id Google Analytics Tracking ID
     *
     * @param array $form_values
     */
    function __construct( $tracking_id, $form_values = array() ) {

        $this->tracking_id = $tracking_id;
        $this->form_values = $form_values;
    }

    /**
     * Call an API method.
     *
     * @return array Associative array of json decoded API response.
     */
    public function call() {
        return $this->makeRequest('collect');
    }

    /**
     * Performs the underlying HTTP request.
     *
     * @param  string $method The API method to be called
     *
     * @return array | bool
     */
    private function makeRequest($method) {

        $post = array();

        $post['v']   = 1; // protocol version
        $post['t']   = 'event'; // account id
        $post['tid'] = $this->tracking_id; // account id
        $post['cid'] = $this->createClientID(); // client id

        $post['uip'] = weforms_get_client_ip();
        $post['ua']  = $this->get_user_agent();

        $post['ec']  = $this->form_values['event_category'];
        $post['ea']  = $this->form_values['event_action'];
        $post['el']  = $this->form_values['event_label'];
        $post['ev']  = $this->form_values['event_value'];

        $url = $this->api_endpoint.'/'.$method;

        $result  = wp_remote_post( $url, array('body' => $post) );

        if( is_wp_error( $result ) ){

            return false;
        }

        return isset($result['body']) ? json_decode($result['body'], true) : false;
    }

    /**
     * Create a GUID on Client specific values
     *
     * @return string
     */
    private function createClientID() {

        // collect user specific data
        if ( isset( $_COOKIE['_ga'] ) ) {

            $gaCookie = explode('.', $_COOKIE['_ga']);

            if( isset($gaCookie[2] ) ) {

                // check if uuid
                if( $this->checkUUID( $gaCookie[2] ) ) {

                    // uuid set in cookie
                    return $gaCookie[2];
                }
                elseif( isset($gaCookie[2]) && isset($gaCookie[3]) ) {

                    // google default client id
                    return $gaCookie[2] . '.' . $gaCookie[3];
                }
            }
        }

        // nothing found - return random uuid client id
        return $this->generateUUID();
    }

    /**
     * Check if is a valid UUID v4
     *
     * @param $uuid
     *
     * @return int
     */
    private function checkUUID( $uuid ) {

        return preg_match('#^[0-9A-F]{8}-[0-9A-F]{4}-4[0-9A-F]{3}-[89AB][0-9A-F]{3}-[0-9A-F]{12}$#i', $uuid );
    }

    /**
     * Generate UUID v4 function - needed to generate a CID when one isn't available
     *
     * @author Andrew Moore http://www.php.net/manual/en/function.uniqid.php#94959
     *
     * @return string
     */
    private function generateUUID() {
        return sprintf( '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            // 32 bits for "time_low"
            mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff ),

            // 16 bits for "time_mid"
            mt_rand( 0, 0xffff ),

            // 16 bits for "time_hi_and_version",
            // four most significant bits holds version number 4
            mt_rand( 0, 0x0fff ) | 0x4000,

            // 16 bits, 8 bits for "clk_seq_hi_res",
            // 8 bits for "clk_seq_low",
            // two most significant bits holds zero and one for variant DCE1.1
            mt_rand( 0, 0x3fff ) | 0x8000,

            // 48 bits for "node"
            mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff ), mt_rand( 0, 0xffff )
        );
    }

    /**
     * Get user_agent
     *
     * @return string
     */
    function get_user_agent() {

        return $_SERVER['HTTP_USER_AGENT'];
    }

}
