<?php

/**
 * Google Analytics Integration
 */
class WeForms_Integration_Google_Analytics extends WeForms_Pro_Integration {

    function __construct() {

        $this->id              = 'google-analytics';
        $this->title           = 'Google Analytics';
        $this->icon            = WEFORMS_ASSET_URI . '/images/icon-google-analytics.svg';
        $this->template        = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled' => false,
            'fields'  => array(
                'tracking_id'    => $this->get_tracking_id(),
                'event_category' => __( 'Form', 'weforms-pro' ),
                'event_action'   => __( 'Submission', 'weforms-pro' ),
                'event_label'    => '',
                'event_value'    => '',
            ),
        );

        $this->load_settings();

        add_filter( 'admin_footer', array( $this, 'load_template' ) );

        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );

        add_action( 'weforms_entry_submission', array( $this, 'send_ga_event' ), 10, 4 );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );

    }

    /**
     * Enqueue the mixin
     *
     * @param $scritps
     *
     * @return array
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-google-analytics'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['google-analytics'] = array(
            'label' => __( 'Google Analytics', 'weforms-pro' ),
            'icon'  => WEFORMS_PRO_ASSETS . '/images/google-analytics.svg',
        );

        return $tabs;
    }


    /**
     * Require the GATracking class if not exists
     *
     * @return void
     */
    public function require_google_analytics() {

        if ( ! class_exists( 'WeDevs_Google_Analytics_Tracker' ) ) {

            require_once dirname( __FILE__ ) . '/class-google-analytics-tracker.php';
        }
    }

    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     *
     * @param  int $form_id
     *
     * @param  int $page_id
     *
     * @param  array $form_settings
     *
     * @return void
     */
    public function send_ga_event( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->fields->tracking_id ) ) {
            return;
        }

        foreach ($integration->fields as $key => $value) {

            // default
            if( empty( $value ) ) {

                switch ( $key ) {
                    case 'event_category':
                        $value = 'From';
                        break;

                    case 'event_action':
                        $value = 'Submission';
                        break;

                    case 'event_label':
                        $value = 'Form_' . $form_id;
                        break;

                    case 'event_value':
                        $value = $form_id;
                        break;
                }
            }

            $value = WeForms_Notification::replace_field_tags( $value, $entry_id );

            $form_values[$key] = $value;
        }


        $form_values = wp_parse_args( $form_values, $this->settings_fields['fields'] );

        $this->require_google_analytics();

        $ga = new WeDevs_Google_Analytics_Tracker( $form_values['tracking_id'], $form_values );

        $response = $ga->call();
    }

    /**
     * Get the Global tracking_id
     *
     * @return string
     */
    private function get_tracking_id() {
        $tracking_id = weforms_get_settings( 'google_analytics_ua_tracking_id' );

        return $tracking_id;
    }
}

