<?php

/**
 * Google Sheets Integration
 */
class WeForms_Integration_Google_Sheets extends WeForms_Pro_Integration {

    /**
     * Google Client ID
     *
     * @var string
     */
    const CLIENT_ID = '346492796244-bnc5c6eskht74a2f13acdllf6aokm685.apps.googleusercontent.com';

    /**
     * Google Client Secret
     *
     * @var string
     */
    const CLIENT_SECRET = 'To6UWLetWWy3VslhFfgxA5lL';

    /**
     * Redirect URL after google authentication
     *
     * @var string
     */
    const REDIRECT_URL = 'urn:ietf:wg:oauth:2.0:oob';


    /**
     * Authentication mode. Currently we have only 'code'. Later will add redirect
     *
     * @var string
     */
    const AUTH_METHOD = 'code';

    /**
     * The google client object
     *
     * @var object
     */
    private $client = NULL;


    /**
     * Fields we want to use for sending data
     *
     * @var array
     */
    private $valid_fields  = array();

    function __construct() {

        $this->id              = 'google-sheets';
        $this->title           = __( 'Google Sheets', 'weforms-pro' );
        $this->icon            = WEFORMS_ASSET_URI . '/images/icon-google-sheets.svg';
        $this->template        = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled'      => false,
            'spread_sheet' => '',
            'sheet'        => '',
            'fields'       => array(),
            'meta_data'    => array('created'),
        );

        $this->load_settings();

        $this->valid_fields = array(
            'name_field',
            'text_field',
            'textarea_field',
            'dropdown_field',
            'radio_field',
            'checkbox_field',
            'website_url',
            'email_address',
            'image_upload',
            'date_field',
            'file_upload',
            'country_list_field',
            'numeric_text_field',
            'address_field',
            'ratings',
            'repeat_field',
            'signature_field',
            // 'custom_hidden_field',
            // 'section_break',
            // 'custom_html',
            // 'shortcode',
            // 'action_hook',
            // 'google_map',
            // 'recaptcha',
            // 'toc',
        );

        add_filter( 'admin_footer', array( $this, 'load_template' ) );
        add_action( 'wp_ajax_wefroms_google_sheets_fetch_lists', array( $this, 'fetch_lists' ) );
        add_action( 'wp_ajax_wefroms_google_sheets_update_lists', array( $this, 'update_lists' ) );
        add_action( 'wp_ajax_wefroms_google_sheets_update_sheets', array( $this, 'update_sheets' ) );
        add_action( 'wp_ajax_wefroms_google_sheets_fetch_sheets', array( $this, 'fetech_sheets' ) );
        add_action( 'wp_ajax_wefroms_google_sheets_map_header', array( $this, 'map_header' ) );
        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        add_action( 'weforms_entry_submission', array( $this, 'add_sheets_row' ), 10, 4 );
        add_action( 'admin_post_weforms_gs_auth', array( $this, 'auth') );
        add_action( 'wp_ajax_weforms_gs_auth_verify', array( $this, 'verify') );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scripts
     *
     * @return void
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-google-sheets'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Enqueue scripts
     *
     * @return void
     */
    public function enqueue_scripts() {
        if ( !$this->is_weforms_page() ) {
            return;
        }

        $data =array(
            'auth'   => add_query_arg( 'action' , 'weforms_gs_auth', admin_url( 'admin-post.php' )),
        );

        wp_localize_script( 'jquery', 'weforms_gs', $data);
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['google-sheets'] = array(
            'label' => __( 'Google Sheets', 'weforms-pro' ),
            'icon'  => WEFORMS_ASSET_URI . '/images/integrations/google-sheets.svg',
        );

        return $tabs;
    }


    /**
     * Google Client
     *
     * @return object
     */
    public function client() {

        require_once __DIR__ . '/vendor/autoload.php';

        $client = new Google_Client();

        $client->setClientId( self::CLIENT_ID );
        $client->setClientSecret( self::CLIENT_SECRET );
        $client->setRedirectUri( self::REDIRECT_URL );
        $client->setAccessType("offline");
        $client->setIncludeGrantedScopes(true);

        $this->client = $client;

        return $client;
    }

    /**
     * Setup and refresh authentication token and save to db
     *
     * @return bool
     */
    public function setup_token() {

        $token  = get_option( 'weforms_google_sheets_access_token' );
        $auth   = weforms_get_settings( 'google_sheets_auth' );


        if ( ! $auth  || empty( $token ) || !isset( $token['access_token'] ) ) {

            return;
        }

        $this->client->setAccessToken( $token );

        if ( $this->client->isAccessTokenExpired() ) {

            $this->client->refreshToken( $token['refresh_token'] );

            update_option( 'weforms_google_sheets_access_token', $this->client->getAccessToken() );
        }

        return true;
    }


    /**
     * Check if google sheets is autheticated
     *
     * @return bool
     */
    public function is_google_sheets_authenticated() {

        $token  = get_option( 'weforms_google_sheets_access_token' );
        $auth   = weforms_get_settings( 'google_sheets_auth' );


        if ( ! $auth  || empty( $token ) || !isset( $token['access_token'] ) ) {

            return;
        }

        return true;
    }

    /**
     * Redirect to authentication URL
     *
     * @return void
     */
    public function auth() {

        $client = $this->client();

        $scopes = array(
            Google_Service_Sheets::SPREADSHEETS,
            Google_Service_Drive::DRIVE_METADATA_READONLY,
        );

        $client->addScope( $scopes );

        $auth_url = filter_var( $client->createAuthUrl(), FILTER_SANITIZE_URL );

        wp_redirect($auth_url);

        die();
    }

    /**
     * Verify if the retuned code from google is valid
     *
     * @return void
     */
    public function verify() {

        $code = ! empty( $_REQUEST['code'] ) ? $_REQUEST['code'] : '';

        if ( ! $code ) {
             wp_send_json_error( 'Something went wrong!' );
        }

        $client = $this->client();

        $access_token = $client->authenticate($code);

        if ( isset( $access_token['error_description'] ) ) {
            wp_send_json_error( $access_token['error_description'] );
        }

        if ( isset( $access_token['access_token'] ) ) {

            update_option( 'weforms_google_sheets_access_token', $access_token );
            wp_send_json_success( 'Access token updated' );
        }

        wp_send_json_error( 'Something went wrong!' );

    }

    /**
     * Fetch spreadsheets list from db
     *
     * @return void
     */
    public function fetch_lists() {

        $list = get_option( 'wefroms_google_sheets_list' , array() );

        wp_send_json_success( array( 'list' => $list, 'is_authenticated' => $this->is_google_sheets_authenticated() ) );
    }


    /**
     * Fetch spreadsheets sheets from db
     *
     * @return void
     */
    public function fetech_sheets() {

        $list = get_option( 'wefroms_google_sheets_list_sheets' , array() );

        wp_send_json_success( $list );
    }

    /**
     * Get all spreadsheets from autheticated users drive account
     *
     * @return void
     */
    public function update_lists() {

        $client = $this->client();

        if ( ! $this->setup_token() ) {

            wp_send_json_error( 'Google Sheets account is not authenticated properly.' );
        }

        try {

            $service = new Google_Service_Drive( $client );

            $optParams = array(
              'pageSize' => 30,
              'q'        => "mimeType='application/vnd.google-apps.spreadsheet'",
              'fields'   => 'nextPageToken, files(id, name)'
            );

            $results = $service->files->listFiles( $optParams );

            $files = array();

            if ( empty( $results['files'] ) ) {
                wp_send_json_error( 'No spreadsheets found in your google dirve' );
            }

            foreach ( $results['files'] as $key => $file ) {

                $files[] = array(
                    'id'     => $file['id'],
                    'name'   => $file['name']
                );
            }

            update_option( 'wefroms_google_sheets_list' , $files );

            wp_send_json_success( $files );

        }
        catch( Exception $e ) {
             wp_send_json_error( $e->getMessage() );
        }

    }

    /**
     * Get all spreadsheets sheets by spreadsheetId
     *
     * @return void
     */
    public function update_sheets() {

        $spreadsheetId = !empty( $_REQUEST['spread_sheet'] ) ? $_REQUEST['spread_sheet'] : '';

        $client = $this->client();

        if ( ! $this->setup_token() ) {

            wp_send_json_error( 'Google Sheets account is not authenticated properly.' );
        }

        try {

            $service = new Google_Service_Sheets($client);

            // Print the names and IDs for up to 10 files.
            $response = $service->spreadsheets->get( $spreadsheetId );

            $_sheets = array();

            $sheets = get_option( 'wefroms_google_sheets_list_sheets' , array() );

            foreach ( $response['sheets'] as $key => $sheet) {

                $_sheets[] = array(
                    'id'     => $sheet->properties->sheetId,
                    'name'   => $sheet->properties->title,
                );
            }

            $sheets[$spreadsheetId] = $_sheets;

            update_option( 'wefroms_google_sheets_list_sheets' , $sheets );

            wp_send_json_success( $sheets );

        }
        catch( Exception $e ) {
             wp_send_json_error( $e->getMessage() );
        }

    }


    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function add_sheets_row( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->spread_sheet ) ) {
            return;
        }


        if ( empty( $integration->sheet ) && $integration->sheet !== 0 ) {
            return;
        }

        $meta_data = array();

        if ( isset( $integration->meta_data ) ) {

            if ( ! empty( $integration->meta_data) && is_array( $integration->meta_data ) ) {
                $meta_data = $integration->meta_data;
            }

        } else {

            $meta_data = ['created'];
        }


        $client = $this->client();

        if ( ! $this->setup_token() ) {

            return;
        }

        $form       = weforms()->form->get( $form_id );
        $entry      = $form->entries()->get( $entry_id );
        $fields     = $entry->get_fields();
        $metadata   = $entry->get_metadata();

        $row_data   = array();
        $row_data[] = $entry_id;

        foreach ($fields as $key => $field) {

            if ( in_array( $field['type'], $this->valid_fields) ) {

                if ( $field['type'] == 'signature_field' ) {
               
                  $filename = sanitize_file_name( $form_id . "-" . time() . ".png" );
                  $url = wp_upload_dir()['baseurl'] . '/weforms-signatures/' . $filename;
                  $row_data[] = $url;

                }
                else{
                    $row_data[] = $field['value'];
                }
            }
        }

        foreach ( $meta_data as $key => $meta_key ) {

            if ( isset( $metadata[ $meta_key ] ) && !empty( $metadata[ $meta_key ] ) ) {

                $row_data[]    = $metadata[$meta_key];

            } else {
                $row_data[]    = '';
            }

        }


        $sheet_service = new Google_Service_Sheets( $client );

        $fileId        = $integration->spread_sheet ;

        $values = array();
        foreach( $row_data as $field_value ) {

            $cellData = new Google_Service_Sheets_CellData();
            $value    = new Google_Service_Sheets_ExtendedValue();

            if ( is_numeric( $field_value ) ) {

                $value->setNumberValue( $field_value );

            } elseif ( is_array($field_value) ) {

                $value->setStringValue( implode(", ", $field_value) );

            } elseif ( is_serialized( $field_value ) ) {

                $field_value = unserialize( $field_value );

                $serialized_value = array();

                if ( is_array( $field_value ) ) {

                    foreach ( $field_value as $key => $single_value ) {

                        $single_value = str_replace( array( "_", "-" ), " ", $key) . ': ' . $single_value;
                        $single_value = ucwords( $single_value );
                        $serialized_value[] = $single_value;
                    }

                    $value->setStringValue( implode(", ", $serialized_value) );
                }


            } else {

                $field_value = trim( strip_tags( $field_value ) );
                $value->setStringValue( $field_value );
            }


            $cellData->setUserEnteredValue( $value );
            $values[] = $cellData;
        }

        $rowData = new Google_Service_Sheets_RowData();
        $rowData->setValues( $values );

        $append_request = new Google_Service_Sheets_AppendCellsRequest();

        $append_request->setSheetId( $integration->sheet );

        $append_request->setRows( $rowData );
        $append_request->setFields( 'userEnteredValue' );

        $request = new Google_Service_Sheets_Request();

        $request->setAppendCells( $append_request );

        $requests = array();
        $requests[] = $request;

        $batchUpdateRequest = new Google_Service_Sheets_BatchUpdateSpreadsheetRequest(array(
            'requests' => $requests
        ));

        try {

            $response = $sheet_service->spreadsheets->batchUpdate($fileId, $batchUpdateRequest);

            if ( $response->valid() ) {

                return true;
            }


        } catch ( Exception $e ) {

            weforms()->log('error', $e->getMessage());
        }

        return false;

    }

    /**
     * Put the header row at selected sheet
     *
     * @return void
     */
    public function map_header( ) {

        $client = $this->client();

        if ( ! $this->setup_token() ) {

             wp_send_json_error( 'Google Sheets account is not authenticated properly.' );

        }

        $form_id    = !empty($_REQUEST['form_id']) ? $_REQUEST['form_id'] : '';
        $sheet_id   = !empty($_REQUEST['spread_sheet']) ? $_REQUEST['spread_sheet'] : '';
        $sheet      = !empty($_REQUEST['sheet']) ? $_REQUEST['sheet'] : '';
        $meta_data  = !empty($_REQUEST['meta_data']) ? $_REQUEST['meta_data'] : '';
        $meta_data  = is_array( $meta_data ) ? $meta_data : array() ;


        $form       = weforms()->form->get( $form_id );
        $fields     = $form->get_fields();

        $sheets     = get_option( 'wefroms_google_sheets_list_sheets' , array() );
        $key        = array_search($sheet, array_map( function( $element ){ return $element['id']; }, $sheets[$sheet_id] ) );

        $sheet_name = $sheets[$sheet_id][$key]['name'];


        try {

            $service    = new Google_Service_Sheets( $client );

            $range      = $sheet_name . "!A1:DZ1";

            $result     = $service->spreadsheets_values->get( $sheet_id, $range );

            $values[] = 'ID';

            foreach ( $fields as $field ) {

                if ( in_array( $field['template'], $this->valid_fields ) ) {

                    $label = !empty( $field['label'] ) ? $field['label'] : $field['name'];

                    $values[] = $label;
                }
            }

            if ( $meta_data ) {
                foreach ( $meta_data as $key => $meta ) {
                   $values[] = ucwords( str_replace("_", " ", $meta ) );
                }
            }

            $valueRange = new Google_Service_Sheets_ValueRange();
            $clear      = new Google_Service_Sheets_ClearValuesRequest();

            $valueRange->setValues( ["values" => $values] );

            $conf = ["valueInputOption" => "RAW"];

            $response = $service->spreadsheets_values->clear( $sheet_id, $range, $clear );
            $response = $service->spreadsheets_values->update( $sheet_id, $range, $valueRange, $conf );

            wp_send_json_success( 'Sheet header created' );

        }
        catch( Exception $e ) {
            wp_send_json_error( $e->getMessage() );
        }

    }
}

