<?php

/**
 * Hubspot API  manager class
 *
 * @since 1.4.2
 */

class WeForms_Hubspot_Controller extends Weforms_REST_Controller {

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'hubspot';

    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'hubspot_fetch_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'hubspot_update_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );
    }

    /**
     * get Collection of Hubspot Lists
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function hubspot_fetch_lists( $request ) {
        $lists    = get_option( 'weforms_hubspot_forms', array( ) );
        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function hubspot_update_lists( $request ) {
        $response = $this->hubspot()->get_forms( $this->get_api_key() );

        if ( isset( $response['error'] ) ) {
            return new WP_Error( 'rest_weforms_hubspot_error', __( $response['error'], 'weforms-pro') , array( 'status' => 404 ) );
        }

        $forms = isset( $response['forms'] ) ? $response['forms'] : array();
        update_option( 'weforms_hubspot_forms', $forms );

        $response = $this->prepare_response_for_collection( $forms );
        $response = rest_ensure_response( $response );

        return $response;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'hubspot_api_key' );

        return $api_key;
    }

    /**
     * Require the hubspot class if not exists
     *
     * @return void
     */
    public function hubspot() {
        if ( ! class_exists( 'weForms_HubSpot_API' ) ) {
            require_once dirname( __FILE__ ) . '/../class-hubspot.php';
        }

        return new weForms_HubSpot_API();
    }
}
