<?php

/**
 * HubSpot Integration
 */
class WeForms_Integration_HubSpot extends WeForms_Pro_Integration {

    function __construct() {

        $this->id              = 'hubspot';
        $this->title           = __( 'HubSpot', 'weforms-pro' );
        $this->icon            = WEFORMS_ASSET_URI . '/images/icon-hubspot.svg';
        $this->template        = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled' => false,
            'guid'    => '',
            'fields'  => array(
                'email' => ''
            ),
        );

        $this->load_settings();

        add_filter( 'admin_footer', array( $this, 'load_template' ) );

        add_action( 'wp_ajax_wpuf_hubspot_fetch_lists', array( $this, 'fetch_lists' ) );
        add_action( 'wp_ajax_wpuf_hubspot_update_lists', array( $this, 'update_lists' ) );

        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );
        add_action( 'weforms_entry_submission', array( $this, 'subscribe_user' ), 10, 4 );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scritps
     *
     * @return array
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-hubspot'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'hubspot_api_key' );

        return $api_key;
    }

    /**
     * Fetch hubspots saved list from server
     *
     * @return array
     */
    public function fetch_lists() {
        $lists = get_option( 'weforms_hubspot_forms', array() );

        wp_send_json_success( $lists );
    }

    /**
     * Require the hubspot class if not exists
     *
     * @return void
     */
    public function hubspot() {
        if ( ! class_exists( 'weForms_HubSpot_API' ) ) {
            require_once dirname( __FILE__ ) . '/class-hubspot.php';
        }

        return new weForms_HubSpot_API();
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['hubspot'] = array(
            'label' => __( 'HubSpot', 'weforms-pro' ),
            'icon'  => WEFORMS_ASSET_URI . '/images/integrations/hubspot.svg',
        );

        return $tabs;
    }

    /**
     * Fetch the udpated list from hubspot, save and return it
     *
     * @return array
     */
    public function update_lists() {

        $response = $this->hubspot()->get_forms( $this->get_api_key() );

        if ( isset( $response['error'] ) ) {
            wp_send_json_error(  $response['error'] );
        }

        $forms = isset( $response['forms'] ) ? $response['forms'] : array();

        update_option( 'weforms_hubspot_forms', $forms );

        wp_send_json_success( $forms );
    }

    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function subscribe_user( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->fields ) || empty( $integration->guid ) ) {
            return;
        }

        $lists    = get_option( 'weforms_hubspot_forms', array() );
        $list     = isset( $lists[$integration->guid] ) ? $lists[$integration->guid] : array();
        $portalId = isset( $list['id'])  ? $list['id'] : '';

        if ( empty( $portalId ) ) {
            weforms()->log( 'error', 'HubSpot Error: Something is wrong with hubspot on form ' . $form_id );
        }

        $notification = new WeForms_Notification( array( 'form_id' => $form_id, 'entry_id' => $entry_id ) );
        $notification->set_merge_tags();

        $data = array();

        foreach ( $integration->fields as $key => $value ) {
            $value        = $notification->replace_tags($value);
            $value        = WeForms_Notification::replace_name_tag( $value , $entry_id );
            $data[ $key ] = $value;
        }

        $response = $this->hubspot()->submit_form( $portalId, $integration->guid , $data );
    }
}

