<?php

/**
 * MailChimp API  manager class
 *
 * @since 1.4.2
 */

class WeForms_Mailchimp_Controller extends Weforms_REST_Controller {

    private $mailchimp = null;

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'mailchimp';

    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'mailchimp_fetch_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'mailchimp_update_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );
    }

    /**
     * get Collection of Mailchimp Lists
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function mailchimp_fetch_lists( $request ) {
        $type       = !empty( $request['type'] ) ? $request['type'] : 'lists';
        $list_id    = !empty( $request['list_id'] ) ? $request['list_id'] : '';
        $option_key = 'categories' === $type ? 'weforms_mailchimp_categories_' . $list_id : 'weforms_mailchimp_lists' ;
        $lists      = get_option( $option_key , array() );

        if ( 'categories' === $type  ) {
            $lists = $this->format_categories( $lists );
        }

        $response = $this->prepare_response_for_collection( $lists );
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function mailchimp_update_lists( $request ) {
        $this->mailchimp();

        $api_key = $this->get_api_key();
        $type    = !empty( $request['type'] ) ? $request['type'] : 'lists';
        $list_id = !empty( $request['list_id'] ) ? $request['list_id'] : '';
        $url     = 'lists?count=100';

        if ( empty( $api_key ) ) {
            return new WP_Error( 'rest_weforms_mailchimp_invalid_api_key', __( 'MailChimp API Key is missing. Please update your API key from weForms settings page.', 'weforms-pro') , array( 'status' => 404 ) );
        }

        if ( 'categories' === $type && empty( $list_id ) ) {
            return new WP_Error( 'rest_weforms_mailchimp_invalid_list_id', __( 'Select a list first', 'weforms-pro') , array( 'status' => 404 ) );
        }

        if ( 'categories' === $type ) {
            $url .= '/' . $list_id . '/interest-categories' . '/';
        }

        $response  = $this->mailchimp()->get( $url );

        if ( $this->is_error_response( $response ) ) {
            return new WP_Error( 'rest_weforms_mailchimp_invalid_api_key', $this->get_response_error_message( $response ) , array( 'status' => 404 ) );
        }

        if ( 'categories' === $type ) {
            $lists = $this->update_mailchimp_categories( $list_id, $response );
            $lists = $this->format_categories( $lists );
        } else {
            $lists = $this->update_mailchimp_lists( $response );
        }

        $response = $this->prepare_response_for_collection( $lists );
        $response = rest_ensure_response( $response );

        return $response;
    }


    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'mailchimp_api_key' );

        return $api_key;
    }

    /**
     * Require the mailchimp class if not exists
     *
     * @return void
     */
    public function mailchimp() {
        if ( ! class_exists( 'MailChimp' ) ) {
            require_once dirname( __FILE__ ) . '/../class-mailchimp.php';
        }

        if ( ! $this->mailchimp ) {
            $this->mailchimp = new MailChimp( $this->get_api_key() );
        }

        return $this->mailchimp;
    }


    /**
     * format_categories
     *
     * @return
     **/
    function format_categories( $lists ) {
        return array_map( function($category) {
            $category['selected'] = ($category['type'] == 'checkboxes' || $category['type'] === 'hidden') ? array() : '';

            return $category;
        }, $lists);
    }

    /**
     * Update mailchimps database
     *
     * @return array
     **/
    function update_mailchimp_categories( $list_id, $response ) {
        $lists = array();

        if ( isset( $response['categories'] ) && is_array( $response['categories'] ) ) {

            foreach ( $response['categories'] as $category ) {
                $lists[] = array(
                    'id'        => $category['id'],
                    'name'      => $category['title'],
                    'type'      => $category['type'],
                    'interests' => $this->get_interests( $list_id, $category['id'] ),
                );
            }

            update_option( 'weforms_mailchimp_categories_' . $list_id, $lists );
        }

        return $lists;
    }

    /**
     * get_interests
     *
     * @return
     **/
    function get_interests( $list_id, $category_id ) {
        $lists    = array();
        $url      = 'lists/' . $list_id . '/interest-categories' . '/' . $category_id . '/interests';
        $response = $this->mailchimp()->get( $url );

        if ( ! $this->is_error_response( $response ) ) {
            if ( isset( $response['interests'] ) && is_array( $response['interests'] ) ) {
                foreach ( $response['interests'] as $interests ) {
                    $lists[] = array(
                        'id'        => $interests['id'],
                        'name'      => $interests['name'],
                    );
                }

                return $lists;
            }
        }

        return array();
    }

    /**
     * Update mailchimps database
     *
     * @return array
     **/
    function update_mailchimp_lists( $response ) {
        $lists = array();

        if ( isset( $response['lists'] ) && is_array( $response['lists'] ) ) {
            foreach ( $response['lists'] as $list ) {
                $lists[] = array(
                    'id'     => $list['id'],
                    'name'   => !empty( $list['name'] ) ? $list['name'] : '',
                );
            }

            update_option( 'weforms_mailchimp_lists', $lists );
        }

        return $lists;
    }

    /**
     * is error response
     *
     * @param $response array
     *
     * @return boolen
     **/
    function is_error_response( $response ) {
        return isset( $response['status'] ) && $response['status'] != 200;
    }
}
