<?php

/**
 * Mailchimp Integration
 */
class WeForms_Integration_MailChimp extends WeForms_Pro_Integration {

    private $mailchimp = null;

    function __construct() {

        $this->id                = 'mailchimp';
        $this->title             = __( 'MailChimp', 'weforms-pro' );
        $this->icon              = WEFORMS_ASSET_URI . '/images/icon-mailchimp.svg';
        $this->template          = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled'   => false,
            'list'      => '',
            'interests' => array(),
            'double'    => false,
            'fields'   => array(
                'email'      => '',
                'first_name' => '',
                'last_name'  => ''
            )
        );

        $this->load_settings();

        add_filter( 'weforms_integrations', array( $this, 'register_integration_settings' ) );

        add_filter( 'admin_footer', array( $this, 'load_template' ) );

        add_action( 'wp_ajax_weforms_mailchimp_fetch_lists', array( $this, 'fetch_lists' ) );
        add_action( 'wp_ajax_weforms_mailchimp_update_lists', array( $this, 'update_lists' ) );

        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );

        add_action( 'weforms_entry_submission', array( $this, 'subscribe_user' ), 10, 4 );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scritps
     *
     * @return array
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-mailchimp'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        $api_key = weforms_get_settings( 'mailchimp_api_key' );

        return $api_key;
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['mailchimp'] = array(
            'label' => __( 'MailChimp', 'weforms-pro' ),
            'icon'  => WEFORMS_ASSET_URI . '/images/integrations/mailchmp2.svg',
        );

        return $tabs;
    }

    /**
     * Fetch mailchimps saved list from server
     *
     * @return array
     */
    public function fetch_lists() {
        $type       = !empty( $_REQUEST['type'] ) ? $_REQUEST['type'] : 'lists';
        $list_id    = !empty( $_REQUEST['list_id'] ) ? $_REQUEST['list_id'] : '';

        $option_key = 'categories' === $type ? 'weforms_mailchimp_categories_' . $list_id : 'weforms_mailchimp_lists' ;
        $lists      = get_option( $option_key , array() );

        if ( 'categories' === $type  ) {
            $lists = $this->format_categories( $lists );
        }

        wp_send_json_success( $lists );
    }

    /**
     * format_categories
     *
     * @return
     **/
    function format_categories( $lists ) {
        return array_map( function($category) {
            $category['selected'] = ($category['type'] == 'checkboxes' || $category['type'] === 'hidden') ? array() : '';
            return $category;
        }, $lists);
    }

    /**
     * Require the mailchimp class if not exists
     *
     * @return void
     */
    public function mailchimp() {
        if ( ! class_exists( 'MailChimp' ) ) {
            require_once dirname( __FILE__ ) . '/class-mailchimp.php';
        }

        if ( ! $this->mailchimp ) {
            $this->mailchimp = new MailChimp( $this->get_api_key() );
        }

        return $this->mailchimp;
    }

    /**
     * Fetch the udpated list from mailchimp, save and return it
     *
     * @return array
     */
    public function update_lists() {

        $this->mailchimp();

        $api_key = $this->get_api_key();
        $type    = !empty( $_REQUEST['type'] ) ? $_REQUEST['type'] : 'lists';
        $list_id = !empty( $_REQUEST['list_id'] ) ? $_REQUEST['list_id'] : '';
        $url     = 'lists?count=100';

        if ( empty( $api_key ) ) {
            wp_send_json_error( __( 'MailChimp API Key is missing. Please update your API key from weForms settings page.', 'weforms-pro' ) );
        }

        if ( 'categories' === $type && empty( $list_id ) ) {
            wp_send_json_error( __( 'Select a list first', 'weforms-pro' ) );
        }

        if ( 'categories' === $type ) {
            $url = 'lists/' . $list_id . '/interest-categories' . '/';
        }

        $response  = $this->mailchimp()->get( $url );

        if ( $this->is_error_response( $response ) ) {
            wp_send_json_error( $this->get_response_error_message( $response ) );
        }

        if ( 'categories' === $type ) {
            $lists = $this->update_mailchimp_categories( $list_id, $response );
            $lists = $this->format_categories( $lists );
        } else {
            $lists = $this->update_mailchimp_lists( $response );
        }

        wp_send_json_success( $lists );
    }

    /**
     * is error response
     *
     * @param $response array
     *
     * @return boolen
     **/
    function is_error_response( $response ) {
        return isset( $response['status'] ) && $response['status'] != 200;
    }

    /**
     * get response error message
     *
     * @param $response array
     *
     * @return string
     **/
    function get_response_error_message( $response ) {
        if ( isset( $response['status'] ) && $response['status'] != 200 ) {

            $error = !empty( $response['title'] ) ? $response['title'] : '';
            $error .= !empty( $response['detail'] ) ? ': ' . $response['detail'] : '';

            if ( empty( $error ) ) {
                $error = __( 'Something went wrong.', 'weforms-pro' );
            }

            return $error;
        }
    }

    /**
     * get_interests
     *
     * @return
     **/
    function get_interests( $list_id, $category_id ) {

        $url      = 'lists/' . $list_id . '/interest-categories' . '/' . $category_id . '/interests';
        $response = $this->mailchimp()->get( $url );

        $lists = array();

        if ( ! $this->is_error_response( $response ) ) {
            if ( isset( $response['interests'] ) && is_array( $response['interests'] ) ) {
                foreach ( $response['interests'] as $interests ) {
                    $lists[] = array(
                        'id'        => $interests['id'],
                        'name'      => $interests['name'],
                    );
                }

                return $lists;
            }
        }

        return array();
    }

    /**
     * Update mailchimps database
     *
     * @return array
     **/
    function update_mailchimp_categories( $list_id, $response ) {

        $lists = array();

        if ( isset( $response['categories'] ) && is_array( $response['categories'] ) ) {

            foreach ( $response['categories'] as $category ) {
                $lists[] = array(
                    'id'        => $category['id'],
                    'name'      => $category['title'],
                    'type'      => $category['type'],
                    'interests' => $this->get_interests( $list_id, $category['id'] ),
                );
            }

            update_option( 'weforms_mailchimp_categories_' . $list_id, $lists );
        }

        return $lists;
    }

    /**
     * Update mailchimps database
     *
     * @return array
     **/
    function update_mailchimp_lists( $response ) {

        $lists = array();

        if ( isset( $response['lists'] ) && is_array( $response['lists'] ) ) {

            foreach ( $response['lists'] as $list ) {
                $lists[] = array(
                    'id'     => $list['id'],
                    'name'   => !empty( $list['name'] ) ? $list['name'] : '',
                );
            }

            update_option( 'weforms_mailchimp_lists', $lists );
        }

        return $lists;
    }

    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function subscribe_user( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->list ) || empty( $integration->fields->email ) ) {
            return;
        }


        $email = WeForms_Notification::replace_field_tags( $integration->fields->email, $entry_id );

        if ( empty( $email ) ) {
            return;
        }

        $first_name = WeForms_Notification::replace_name_tag( $integration->fields->first_name, $entry_id );
        $last_name  = WeForms_Notification::replace_name_tag( $integration->fields->last_name, $entry_id );

        $interests_obj = array();

        foreach ( $integration->interests as $key => $interest ) {

            if ( isset( $interest->selected ) && $interests = $interest->selected ) {

                if ( is_array( $interests ) ) {

                    foreach ( $interests as $key => $interest_id ) {
                        $interests_obj[ $interest_id ] = true;
                    }

                } elseif ( is_string( $interests ) ) {

                    $interests_obj[ $interests ] = true;
                }
            }
        }

        $mailchimp_args = array(
            'email_address' => $email,
            'status'        => 'subscribed',
            'merge_fields'  => array(
                'FNAME' => $first_name,
                'LNAME' => $last_name,
            ),
            'double_optin' => $integration->double,
        );

        if ( !empty( $interests_obj ) ) {
            $mailchimp_args['interests']   = $interests_obj;
        }

        $response = $this->mailchimp()->post( 'lists/' . $integration->list . '/members', $mailchimp_args );

        if ( ! isset( $response['status'] ) || $response['status']  != 'subscribed' ) {
            weforms()->log('MailChimp: ' . $response);
        }
    }
}

