<?php

/**
 * Transaction API Manager
 *
 * @since 1.4.2
 */

class Weforms_Transaction_Controller extends Weforms_REST_Controller {

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $rest_base = 'transaction';


    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->rest_base . '/(?P<form_id>[\d]+)' ,
            array(
                'args'   => array(
                    'form_id' => array(
                        'description'       => __( 'Unique identifier for the object', 'weforms' ),
                        'type'              => 'integer',
                        'sanitize_callback' => 'absint',
                        'validate_callback' => array( $this, 'is_form_exists' ),
                        'required'          => true,
                    )
                ),
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'get_items' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'                => $this->get_collection_params()
                ),
            )
        );


        register_rest_route(
            $this->namespace,
            '/'. $this->rest_base,
            array(
                array(
                    'methods'             => WP_REST_Server::DELETABLE,
                    'callback'            => array( $this, 'bulk_delete_items' ),
                    'permission_callback' => array( $this, 'delete_item_permissions_check' ),
                    'args'                => array(
                        'entry_id'  => array(
                            'required'    => true,
                            'type'        => 'object',
                            'description' => __( 'Entry id', 'weforms' ),
                            'validate_callback' => array( $this, 'is_entry_exists' ),
                        )
                    )
                ),
            )
        );
    }

    /**
     * Retrieves a collection of Transaction.
     *
     * @since  1.3.9
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function get_items( $request ) {
        $form_id      = $request['form_id'];
        $current_page = isset( $request['page'] ) ? intval( $request['page'] ) : 1;
        $per_page     = isset( $request['posts_per_page'] ) ? intval( $request['posts_per_page'] ) : 20;
        $offset       = ( $current_page - 1 ) * $per_page;

        $entries = weforms_get_form_payments( $form_id, array(
            'number' => $per_page,
            'offset' => $offset
        ) );

        $total_entries   = weforms_count_form_payments( $form_id );

        $columns         = array(
            'total'          => 'Amount',
            'transaction_id' => 'Transaction ID',
            'created_at'     => 'Created',
        );

        array_map( function( $entry ) use ($columns) {
            $entry->id = $entry->entry_id;
            $entry->fields = array();

            foreach ($columns as $meta_key => $label) {
                switch ($meta_key) {
                    case 'transaction_id':
                        if ( 'paypal' === $entry->gateway ) {
                            $value = $entry->{$meta_key};

                        } elseif ( 'stripe' === $entry->gateway ){
                            $value = sprintf(
                                "<a href='https://dashboard.stripe.com/payments/%s' target='_blank'>%s</a>",
                                $entry->$meta_key,
                                $entry->$meta_key
                            );
                        }
                        break;

                    case 'total':
                        $value = weforms_format_price( $entry->{$meta_key} );
                        break;

                    case 'status':
                        $value = ucfirst($entry->{$meta_key});
                        break;

                    default:
                        $value = $entry->{$meta_key};
                        break;
                }

                $entry->fields[$meta_key] = $value;
            }

        }, $entries );

        $entries = apply_filters('weforms_get_payments', $entries, $form_id );

        $transaction = array(
            'id'            => $form_id,
            'columns'       => $columns,
            'form_entries'  => $entries,
            'form_title'    => get_post_field( 'post_title', $form_id ),
            'pagination'    => array(
                'total'    => $total_entries,
                'per_page' => $per_page,
                'pages'    => ceil( $total_entries / $per_page ),
                'current'  => $current_page
            )
        );

        $transaction = $this->prepare_item_for_response( (array) $transaction, $request );
        $transaction = $this->prepare_response_for_collection( $transaction );
        $response    = rest_ensure_response( $transaction );

        $response->header( 'X-WP-TotalPages', (int) ceil( $total_entries / $per_page ) );
        $response->header( 'X-WP-Total', (int) $total_entries );

        return $response;
    }

    /**
     * Bulk Delete Transaction
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     *
     **/
    public function bulk_delete_items( $request ) {
        global $wpdb;

        $entry_ids = isset( $request['entry_id'] ) ? array_map( 'absint', $request['entry_id'] ) : array();

        if ( ! $entry_ids ) {
            return new WP_Error('rest_weforms_invalid_data',__( 'No entry ids provided!', 'weforms-pro' ));
        }

        $response = array();

        foreach ( $entry_ids as $entry_id ) {
            $status = $wpdb->delete( $wpdb->prefix . 'weforms_payments' , array( 'entry_id' => $entry_id ), array( '%d' ) );

            if( $status ) {
                $response['message'][$entry_id] = __( 'Transaction Deleted successfully ', 'weforms' );
            } else {
                $response['message'][$entry_id] = __( 'Transaction Not found ', 'weforms' );
            }
        }

        $response = $this->prepare_response_for_collection( $response );
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function get_collection_params() {
        $query_params = parent::get_collection_params();
        $query_params['context']['default'] = 'view';

        return $query_params;
    }
}
