<?php
/**
 * Plugin Name: Payment
 * Description: Receive payments with weFroms
 * Plugin URI: https://wedevs.com/weforms/
 * Thumbnail URL: payment.svg
 * Author: weDevs
 * Version: 1.0
 * Author URI: https://wedevs.com
 */

/**
 * Payment CLass
 */
class WeForms_Payment {

    public $id, $title, $icon;

    function __construct() {

        $this->id              = 'payment';
        $this->title           = __( 'Payment', 'weforms-pro' );
        $this->icon            = WEFORMS_PRO_ASSETS . '/images/payment.svg';

        $this->includes();

        // define payment fields
        add_filter( 'weforms_form_fields', array( $this, 'weform_payment_fields' ) );

        // scripts
        add_filter( 'weforms_frontend_scripts', array( $this, 'frontend_payment_scripts' ) );
        add_filter( 'weforms_frontend_styles', array( $this, 'frontend_payment_styles' ) );
        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );

        // routes
        add_action( 'admin_menu', array( $this, 'register_admin_menu' ), 12 );
        add_filter( 'weforms_vue_routes', array( $this, 'payment_routes' ) );

        // payment settings
        add_filter( 'weforms_field_groups', array( $this, 'add_payment_section' ) );
        add_filter( 'wpuf_contact_form_editor_tabs', array( $this, 'weform_editor_payment_tab' ) );
        add_action( 'wpuf-form-builder-tab-contents-contact_form', array( $this, 'weform_editor_payment_tab_content' )  );
        add_action( 'wpuf-form-builder-payment-tabs-contact_form', array( $this, 'add_payment_settings_tabs' ) );
        add_action( 'wpuf-form-builder-payment-tab-contents-contact_form', array( $this, 'add_payment_tab_contents' ) );

        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ) , 12 );
        add_action( 'weforms_settings_tab_content_' . $this->id, array( $this, 'settings_panel' ) );

        // calcualte total
        add_filter( 'weforms_before_entry_submission', array( $this, 'save_total_value' ), 10, 4 );

        // auto polulate total and payment method if not used
        add_action( 'weforms_form_fields_before_submit_button', array( $this, 'maybe_populate_payment_fields'), 10, 3 );

        // set default value settings
        add_filter( 'weforms_get_settings', array( $this, 'set_default_settings' ) );

        // check if needed payment
        add_filter( 'weforms_entry_submission_response', array( $this, 'send_payment_data' ));

        // make payments and save entry
        add_action( 'wp_ajax_weforms-paypal-ipn', array( $this, 'paypal_ipn_listener') );
        add_action( 'wp_ajax_nopriv_weforms-paypal-ipn', array( $this, 'paypal_ipn_listener') );
        add_action( 'wp_ajax_weforms-stripe-payment', array( $this, 'stripe_payment') );
        add_action( 'wp_ajax_nopriv_weforms-stripe-payment', array( $this, 'stripe_payment') );
        add_action( 'template_redirect', array( $this, 'stripe_payment_entry') );

        // load payment contents
        add_action( 'wp_ajax_weforms_form_payments', array( $this, 'get_payments' ) );
        add_action( 'wp_ajax_weforms_form_payments_trash_bulk', array( $this, 'bulk_delete_payments') );
        add_action( 'weforms_ajax_get_contact_forms', array( $this, 'filter_contact_forms' ), 12 );
        add_action( 'weforms_ajax_get_contact_forms_args', array( $this, 'filter_contact_forms_args' ), 12 );

        add_filter( 'weforms_rest_api_class_map', array( $this, 'register_payment_rest_api_integration' ), 10, 1 );
    }

    /**
     * Administrator validation
     *
     * @return void
     */
    public function check_admin() {
        if ( !current_user_can( weforms_form_access_capability() ) ) {
            wp_send_json_error( __( 'You do not have sufficient permission.', 'weforms-pro' ) );
        }
    }

    /**
     * Filter the query args to get all forms, so we can filter all
     *
     * @return void
     */
    public function filter_contact_forms_args( $args ) {

        if ( isset($_REQUEST['filter'] ) && $_REQUEST['filter'] == 'transactions' ) {
            $args['posts_per_page'] = -1;
        }

        return $args;
    }

    /**
     * Filter only forms with payment entry
     *
     * @return void
     */
    public function filter_contact_forms( $contact_forms ) {

        if ( isset($_REQUEST['filter'] ) && $_REQUEST['filter'] == 'transactions' ) {
            foreach ($contact_forms['forms'] as $key => &$form) {
                if ( isset($form->payments) && ! $form->payments ) {
                    unset($contact_forms['forms'][$key]);
                }
            }

            $contact_forms['meta']['total'] = count($contact_forms['forms']);
        }

        return $contact_forms;
    }

    /**
     * Frontend payment scripts
     *
     * @param  array $scripts
     *
     * @return array
     */
    public function frontend_payment_scripts( $scripts ) {

        $scripts['weforms-payment-script'] = array(
            'src'       => WEFORMS_PRO_URL . '/modules/payment/assets/js/payment.js',
            'deps'      => array('jquery', 'wpuf-form','weforms-stripe-v3'),
            'in_footer' => true
        );

        $scripts['weforms-stripe-v3'] = array(
            'src'       => 'https://js.stripe.com/v3/',
            'deps'      => array('jquery', 'wpuf-form'),
            'in_footer' => true
        );

        wp_localize_script( 'weforms-payment-script', 'weFormsPaymentText', array(
            'error'   => __( 'Something went wrong while processing your payment.', 'weforms-pro' ),
            'loading' => __( 'Please wait while we are making the payment..', 'weforms-pro' ),
        ) );

        return $scripts;
    }

    /**
     * Frontend payment styles
     *
     * @param  array $styles
     *
     * @return array
     */
    public function frontend_payment_styles( $styles ) {

        $styles['weforms-payment-script'] = array(
            'src'       => WEFORMS_PRO_URL . '/modules/payment/assets/front-end.css',
            'deps'      => array('weforms-css'),
        );

        return $styles;
    }

    /**
     * Enqueue the mixin
     *
     * @param $scripts
     *
     * @return void
     */
    public function enqueue_mixin( $scripts ) {

        $prefix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

        $scripts['weforms-int-payment-settings'] = array(
            'src' => plugins_url( 'component/index' . $prefix . '.js', __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Enqueue the mixin
     *
     * @param $scripts
     *
     * @return void
     */
    public function admin_scripts() {

        global $pagenow;

        if ( $pagenow == 'admin.php' && isset( $_GET['page'] ) && $_GET['page'] == 'weforms' ) {

            wp_enqueue_media();
        }
    }

    /**
     * Include
     *
     * @return void
     */
    function includes() {
        require_once WEFORMS_PRO_MODULES . '/payment/includes/functions.php';
    }

    /**
     * Add Transactions route
     *
     * @param array $routes
     *
     * @return array
     */
    function payment_routes( $routes ) {
        foreach ( $routes as $key => &$value ) {
            if ( $value['path'] == '/form/:id' ) {

                $routes[$key]['children'][] = array(
                    'path'      => 'payments',
                    'name'      => 'formPayments',
                    'component' => 'FormPayments',
                    'props'     => true,
                );
            }
        }

        $routes[] = array(
            'path'      => '/transactions',
            'name'      => 'transactions',
            'component' => 'Transactions'
        );

        return $routes;
    }


    /**
     * Register the admin menu
     *
     * @return void
     */
    public function register_admin_menu() {
        global $submenu;

        $capability = weforms_form_access_capability();

        if ( current_user_can( $capability ) ) {
            $transactions = array( array( __( 'Transactions', 'weforms-pro' ), $capability, 'admin.php?page=weforms#/transactions' ));

            array_splice( $submenu['weforms'], 2, 0, $transactions );
        }
    }


    /**
     * Add payment Fields panel sections
     *
     * @since 1.1
     *
     * @return array
     */
    public function add_payment_section( $groups ) {

        $fields = apply_filters( 'weforms_field_groups_payment', array(
            'single_product', 'multiple_product', 'total', 'payment_method'
        ) );

        $groups[] = array(
            'title'     => __( 'Payment Fields', 'weforms-pro' ),
            'id'        => 'payment-fields',
            'fields'    => $fields
        );
        return $groups;
    }

    /**
     * Register payment fields
     *
     * @param array $fields
     *
     * @return array
     */
    function weform_payment_fields( $fields ) {

        require_once dirname( __FILE__ ) . '/fields/class-field-single-product.php';
        require_once dirname( __FILE__ ) . '/fields/class-field-multiple-product.php';
        require_once dirname( __FILE__ ) . '/fields/class-field-total.php';
        require_once dirname( __FILE__ ) . '/fields/class-field-payment-method.php';

        $fields['single_product']   = new WeForms_Form_Field_Single_Product();
        $fields['multiple_product'] = new WeForms_Form_Field_Multiple_Product();
        $fields['total']            = new WeForms_Form_Field_Total();
        $fields['payment_method']   = new WeForms_Form_Field_Payment_Method();

        return $fields;
    }

    /**
     * Add form editor tab
     *
     * @since 1.1
     *
     * @return array
     */
    public function weform_editor_payment_tab( $tab ){
        $tab['payment']  = __( 'Payment', 'weforms-pro' );
        return $tab;
    }

    /**
     * Add form editor tab
     *
     * @since 1.1
     *
     * @return array
     */
    public function weform_editor_payment_tab_content( $tab ){
        ?>
        <div id="wpuf-form-builder-payment" class="clearfix" v-show="isActiveTab('payment')">
            <fieldset>
                <h2 id="wpuf-form-builder-payment-tabs" class="nav-tab-wrapper">
                    <?php do_action( "wpuf-form-builder-payment-tabs-contact_form" ); ?>
                </h2><!-- #wpuf-form-builder-payment-tabs -->

                <div id="wpuf-form-builder-payment-contents" class="tab-contents">
                    <?php do_action( "wpuf-form-builder-payment-tab-contents-contact_form" ); ?>
                </div><!-- #wpuf-form-builder-payment-contents -->
            </fieldset>
        </div><!-- #wpuf-form-builder-payment -->
        <?php
    }

    /**
     * Add payment tabs
     *
     * @return void
     */
    public function add_payment_settings_tabs() {
        ?>
            <a href="#" :class="['nav-tab', isActivePaymentTab( 'paypal' ) ? 'nav-tab-active' : '']" v-on:click.prevent="makeActivePaymentTab( 'paypal' )" class="nav-tab"><?php _e( 'Paypal', 'weforms' ); ?></a>
            <a href="#" :class="['nav-tab', isActivePaymentTab( 'stripe' ) ? 'nav-tab-active' : '']" v-on:click.prevent="makeActivePaymentTab( 'stripe' )" class="nav-tab"><?php _e( 'Stripe', 'weforms' ); ?></a>

            <?php do_action( 'wpuf_contact_form_payment_tab' ); ?>

        <?php
    }

    /**
     * Add payment tabs
     *
     * @return void
     */
    public function add_payment_tab_contents() {
        ?>
            <div id="wpuf-metabox-payment" class="tab-content" v-show="isActivePaymentTab('paypal')">
                <?php include_once dirname( __FILE__ ) . '/views/paypal-settings.php'; ?>
            </div>

            <div id="wpuf-metabox-payment-stripe" class="tab-content" v-show="isActivePaymentTab('stripe')">
                <?php include_once dirname( __FILE__ ) . '/views/stripe-settings.php'; ?>
            </div>

            <?php do_action( 'wpuf_contact_form_payment_tab_content' ); ?>

        <?php
    }


    /**
     * Paypal IPN listen and save Transactions
     *
     * @return void
     **/
    public function paypal_ipn_listener() {
        weforms()->log( 'paypal_ipn_request_start', print_r( $_REQUEST, true ) );
        require_once __DIR__ . '/includes/library/paypal-ipn/PaypalIPN.php';

        $form_id       = !empty($_REQUEST['form_id']) ? $_REQUEST['form_id'] : '';
        $page_id       = !empty($_REQUEST['page_id']) ? $_REQUEST['page_id'] : '';
        $entry_fields  = !empty($_REQUEST['entry_fields']) ? $_REQUEST['entry_fields'] : '';

        $form          = weforms()->form->get( $form_id );
        $form_settings = $form->get_settings();

        $sandbox       = $form_settings['paypal_mode'] == 'sandbox' ? true : false;

        $ipn = new PaypalIPN();

        if ( $sandbox ) {
            $ipn->useSandbox();
        }

        $verified = $ipn->verifyIPN();

        if ( $verified ) {

            $entry_id        = $this->insert_entry( $entry_fields, $form_id, $page_id, $form_settings );

            $payment_id      = $this->insert_payment( array(
                'form_id'        => $form_id,
                'entry_id'       => $entry_id,
                'total'          => $_REQUEST['mc_gross'],
                'gateway'        => 'paypal',
                'transaction_id' => $_REQUEST['txn_id'],
                'status'         => 'completed',
                'payment_data'   => serialize( $_REQUEST ),
            ));

        } else {

            error_log( 'Invalid PayPal IPN request: ' . json_encode($_REQUEST) );
        }

        header("HTTP/1.1 200 OK");

        return;
    }

    /**
     * Make stripe payment
     *
     * @return json
     **/
    public function stripe_payment() {
        require_once __DIR__ . '/includes/library/stripe-php/init.php';

        $form_id         = !empty( $_REQUEST['form_id'] ) ? $_REQUEST['form_id'] : '';
        $entry_fields    = !empty( $_REQUEST['entry_fields'] ) ? $_REQUEST['entry_fields'] : '';
        $page_id         = !empty( $_REQUEST['page_id'] ) ? intval($_REQUEST['page_id']) : '';
        $name            = !empty( $_REQUEST['name'] ) ? $_REQUEST['name'] : '';
        $amount          = !empty( $_REQUEST['amount'] ) ? $_REQUEST['amount'] : '';
        $currency        = !empty( $_REQUEST['currency'] ) ? $_REQUEST['currency'] : '';
        $stripe_url      = !empty( $_REQUEST['stripe_url'] ) ? $_REQUEST['stripe_url'] : '';
        $success_message = !empty( $_REQUEST['success_message'] ) ? $_REQUEST['success_message'] : '';

        $form          = weforms()->form->get( $form_id );
        $form_settings = $form->get_settings();
        $fields        = json_encode( $entry_fields );

        if( $stripe_url ) {
        	$success_url = $stripe_url;
        } else {
        	$success_url = get_permalink( $page_id );
        }

        $cancel_url = get_permalink( $page_id );

        $key           = $form_settings['stripe_mode'] == 'test' ? 'stripe_secret_key_test' : 'stripe_secret_key';
        $stripe_key    = $form_settings['stripe_override_keys'] ? $form_settings[$key] : weforms_get_settings( $key );

        // Set Transient for entry fields.
        $transient_key = wp_create_nonce( 'weforms_transient_' . $form_id );
        set_transient( $transient_key, $fields, HOUR_IN_SECONDS );

        \Stripe\Stripe::setApiKey( $stripe_key );

        $session = \Stripe\Checkout\Session::create([
            'payment_method_types' => ['card'],
            'line_items' => [[
              'price_data' => [
                'currency' => $currency,
                'product_data' => [
                  'name' => $name,
                ],
                'unit_amount' => $amount,
              ],
              'quantity' => 1,
            ]],
            'mode' => 'payment',
            'success_url' => add_query_arg( array (
                                'session_id'      => '{CHECKOUT_SESSION_ID}',
                                'key'             => $stripe_key,
                                'transient_key'   => $transient_key,
                                'form_id'         => $form_id,
                                'form_page'       => $page_id,
                                'success_message' => $success_message,
                            ), $success_url ),
            'cancel_url' => add_query_arg( 'payment_failed', __( 'Error : Payment is not completed', 'weforms-pro' ), $cancel_url ),
        ]);

        wp_send_json_success( $session );

    }

    public function stripe_payment_entry() {
        if( isset( $_GET['success_message'] ) )  {
            require_once __DIR__ . '/includes/library/stripe-php/init.php';

            $session_id      = !empty( $_GET['session_id'] ) ? $_GET['session_id'] : '';
            $form_id         = !empty( $_GET['form_id'] ) ? $_GET['form_id'] : '';
            $page_id         = !empty( $_GET['form_page'] ) ? $_GET['form_page'] : '';
            $success_message = !empty( $_GET['success_message'] ) ? $_GET['success_message'] : '';
            $transient_key   = !empty( $_GET['transient_key'] ) ? $_GET['transient_key'] : '';
            $form            = weforms()->form->get( $form_id );
            $form_settings   = $form->get_settings();

            // Get entry fields transient.
            $entry_fields = json_decode( get_transient( $transient_key ), true );

            \Stripe\Stripe::setApiKey( $_GET['key'] );

            $session = \Stripe\Checkout\Session::retrieve( $session_id, [] );

            $data = \Stripe\PaymentIntent::retrieve( $session->payment_intent );
            $charge = $data->charges->data[0];
            $amount = floatval( $charge->amount / 100 );

            if( $charge->paid ) {

                $entry_id        = $this->insert_entry( $entry_fields, $form_id, $page_id, $form_settings );

                $payment_data = array(
                    'livemode'             => $charge->livemode,
                    'amount'               => $amount,
                    'currency'             => $charge->currency,
                    'paid'                 => $charge->paid,
                    'status'               => $charge->status,
                    'receipt_email'        => $charge->receipt_email,
                    'receipt_number'       => $charge->receipt_number,
                    'refunded'             => $charge->refunded,
                    'amount_refunded'      => $charge->amount_refunded,
                    'application_fee'      => $charge->application_fee,
                    'balance_transaction'  => $charge->balance_transaction,
                    'captured'             => $charge->captured,
                    'created'              => $charge->created,
                    'customer'             => $charge->customer,
                    'description'          => $charge->description,
                    'destination'          => $charge->destination,
                    'dispute'              => $charge->dispute,
                    'failure_code'         => $charge->failure_code,
                    'failure_message'      => $charge->failure_message,
                    'fraud_details'        => $charge->fraud_details,
                    'invoice'              => $charge->invoice,
                    'order'                => $charge->order,
                    'shipping'             => $charge->shipping,
                    'source_transfer'      => $charge->source_transfer,
                    'statement_descriptor' => $charge->statement_descriptor,
                );

                 $payment_id = $this->insert_payment( array(
                    'form_id'        => $form_id,
                    'entry_id'       => $entry_id,
                    'total'          => $amount,
                    'gateway'        => 'stripe',
                    'transaction_id' => $charge->id,
                    'status'         => 'completed',
                    'payment_data'   => serialize( $payment_data ),
                ));

                echo '<script type="text/javascript">alert("'.$success_message.'");</script>';
            }
        } elseif( isset( $_GET['payment_failed'] ) ) {
            $payment_failed = !empty( $_GET['payment_failed'] ) ? $_GET['payment_failed'] : '';

            echo '<script type="text/javascript">alert("'.$payment_failed.'");</script>';
        }
    }

    /**
     * Modify response and attach payment data
     *
     * @param array $response
     *
     * @return array
     **/
    function send_payment_data( $response ) {

        $form            = weforms()->form->get( $response['form_id'] );
        $settings        = $form->get_settings();
        $global_settings = weforms_get_settings();
        $entry           = $form->entries()->get( $response['entry_id'] );

        if ( $settings['payment_paypal_deactivate'] && $settings['payment_stripe_deactivate'] ) {
            return $response;
        }

        $methods        = array();

        if ( $settings['payment_paypal_deactivate'] != 1 ) {
            $methods[] = 'paypal';
        }

        if ( $settings['payment_stripe_deactivate'] != 1 ) {
            $methods[] = 'stripe';
        }

        $entry_fields     = $form->prepare_entries();
        $default_message  = __( "You are being redirected to payment.", 'weforms' );
        $products         = array();
        $products         = $this->preapre_products( $form, $entry_fields );
        $default_redirect = !empty( $global_settings['payment_redirect_to'] ) ? get_permalink( $global_settings['payment_redirect_to'] ) : '';
        $default_success_message = !empty( $global_settings['payment_success_message'] ) ? $global_settings['payment_success_message'] : __( 'Thanks for your payment', 'weforms-pro' );

        if ( count($products['products']) ) {

            if ( $payment_method = $form->search_field( 'payment_method' ) ) {

               $payment_method = $entry_fields[$payment_method['name']];

            } elseif( !empty( $_REQUEST['weforms_default_payment_method'] ) ){
                $payment_method = $_REQUEST['weforms_default_payment_method'];
            }

           if ( ! in_array( $payment_method, $methods ) ) {
               $payment_method = $methods[0];
           }

            switch ($payment_method) {

                case 'paypal':

                    if ( '_donations' === $settings['paypal_type'] ) {

                        $products['products'] = array(); //make empty

                        $products['products']['item_name'] = $form->name . ' - ' . get_bloginfo( 'name' );
                        $products['products']['amount']    = floatval($products['total']);
                    }

                    $paypal_email = $settings['paypal_override'] ? $settings['paypal_email'] : weforms_get_settings('paypal_email');

                    $products['products']['no_shipping']     = intval($global_settings['paypal_ask_shipping']);

                    $email_key  = !empty($settings['paypal_user_email_field']) ? $settings['paypal_user_email_field'] : '';

                    if ( !empty( $settings['paypal_prefill_email'] ) && $email_key && !empty( $_REQUEST[$email_key] ) ) {
                        $products['products']['email'] = $_REQUEST[$email_key];
                    }

                    if ( !empty( $global_settings['paypal_logo'] ) ) {
                        $products['products']['image_url']       = $global_settings['paypal_logo'];
                    }

                    $response['products']       = $products['products'];
                    $response['paypal_type']    = $settings['paypal_type'];
                    $response['sandbox']        = $settings['paypal_mode'] == 'sandbox' ? 'sandbox.' : '';
                    $response['business']       = $paypal_email;
                    $response['paypal_message'] = ! empty( $global_settings['paypal_message'] ) ? $global_settings['paypal_message'] : $default_message;


                    $paypal_return = !empty( $settings['paypal_page_id'] ) ? get_permalink( $settings['paypal_page_id'] ) : '';

                    if ( ! $paypal_return ) {
                        $paypal_return = !empty( $default_redirect ) ? $default_redirect : $entry->referer;
                    }

                    $response['paypal_return']  = add_query_arg('payment_success','paypal', $paypal_return);
                    $response['cancel_return']  = add_query_arg('payment_failed','paypal', $entry->referer);

                    $response['notify_url']     = add_query_arg(array(
                        'action'   => 'weforms-paypal-ipn',
                        'form_id'  => $response['form_id'],
                        'page_id'  => $response['data']['page_id'],
                    ),admin_url( 'admin-ajax.php' ));

                    $response['currency_code']     = weforms_get_settings('currency','USD');

                break;

                case 'stripe':

                    $key        = $settings['stripe_mode'] == 'test' ? 'stripe_key_test' : 'stripe_key';
                    $stripe_key = $settings['stripe_override_keys'] ? $settings[$key] : weforms_get_settings($key);

                    $default_image = 'https://stripe.com/img/documentation/checkout/marketplace.png';

                    $response['config']         = array(

                            'image'       => !empty( $global_settings['stripe_logo'] ) ? $global_settings['stripe_logo'] : $default_image,
                            'name'        => !empty( $global_settings['stripe_company_name'] ) ? $global_settings['stripe_company_name'] : get_bloginfo( 'name' ),
                            'description' => !empty( $global_settings['stripe_company_desc'] ) ? $global_settings['stripe_company_desc'] : get_bloginfo( 'description' ),
                            'key'         => $stripe_key,
                            'amount'      => floatval( $products['total'] ) * 100,
                            'currency'    => weforms_get_settings( 'currency','USD' ),
                    );

                    $email_key  = !empty( $settings['stripe_user_email_field'] ) ? $settings['stripe_user_email_field'] : '';


                    if ( !empty( $settings['stripe_prefill_email'] ) && $email_key && !empty( $_REQUEST[$email_key] ) ) {
                        $response['config']['email'] = $_REQUEST[$email_key];
                    }

                    if ( !empty( $global_settings['stripe_ask_billing'] ) ) {
                        $response['config']['billingAddress'] = true;
                    }

                    if ( !empty( $global_settings['stripe_ask_shipping'] ) ) {
                        $response['config']['shippingAddress'] = true;
                    }

                    $response['stripe_message'] = !empty($global_settings['stripe_message']) ? $global_settings['stripe_message'] : $default_message;

                    $stripe_return = !empty( $settings['stripe_page_id'] ) ? get_permalink( $settings['stripe_page_id'] ) : '';

                    if (  $stripe_return ) {
                        $response['stripe_return']  = add_query_arg('payment_success','stripe', $stripe_return);

                    } elseif( $default_redirect ) {
                        $response['stripe_return'] = $default_redirect ;
                    }

                    $response['stripe_success_message'] = $default_success_message;

                break;
            }


            $response['payment_method'] = $payment_method;

            $response['redirect_to']    = false;
            $response['show_message']   = true;
        }

        weforms()->log( 'payment_response', print_r( $response, true ) );

        return $response;
    }

    /**
     * Prepare products to send with paypal
     *
     * @param object $form
     * @param array $entry_fields
     *
     * @return type
     */
    function preapre_products( $form, $entry_fields ) {

        $products = array();
        $total = 0;

        $product_id = 1;

         if( $form->has_field( 'column_field' ) ) {
            foreach ( $form->search_fields( 'column_field' ) as $key => $field ) {
                foreach ( $field['inner_fields'] as $key2 => $inner_fields ) {
                    foreach ($inner_fields as $key3 => $innerfield) {
                        if (
                            isset( $entry_fields[$innerfield['name']] ) && $entry_field =  $entry_fields[$innerfield['name']] &&
                            $innerfield['template'] == 'multiple_product'
                        ) {
                           foreach ( $entry_fields[$innerfield['name']] as $entry ) {
                                $products['item_name_' . $product_id] = $field['label'] . ' - ' . $entry['product'];
                                $products['quantity_' . $product_id]  = $entry['quantity'];
                                $products['amount_' . $product_id]    = $entry['price'];
                                $total += intval($entry['quantity']) * floatval($entry['price']);
                                $product_id++;
                           }
                        }

                        if (
                            isset( $entry_fields[$innerfield['name']] ) && $entry_field =  $entry_fields[$innerfield['name']] &&
                            $innerfield['template'] == 'single_product'
                        ) {
                           $entry = $entry_fields[$innerfield['name']];

                            $price    = isset( $innerfield['price'] ) ? $innerfield['price'] : array();
                            $quantity = isset( $innerfield['quantity'] ) ? $innerfield['quantity'] : array();

                            $_price    = isset($price['is_flexible']) && $price['is_flexible'] ? $entry['price'] : $price['price'];
                            $_quantity = isset($quantity['status']) && $quantity['status'] ? $entry['quantity'] : 1;

                            $products['item_name_' . $product_id] = $field['label'];
                            $products['quantity_' . $product_id]  = $_quantity;
                            $products['amount_' . $product_id]    = $_price;

                            $total += intval($_quantity) * floatval($_price);

                            $product_id++;
                        }
                    }
                }
            }
        }

        if ( $form->has_field( 'multiple_product' ) ) {

            foreach ( $form->search_fields( 'multiple_product' ) as $key => $field) {

                if ( isset( $entry_fields[$field['name']] ) && $entry_field =  $entry_fields[$field['name']] ) {

                    $default_products = $field['options']; // 'name' => price

                    foreach ( $entry_field as $k => $product ) {

                        if ( $product['quantity'] < 1 ) {
                            continue;
                        }

                        // manupulate the price from default product, to avoid user price overriding
                        $price = isset( $default_products[$product['product']] ) ? $default_products[$product['product']] : $product['price'];

                        $products['item_name_' . $product_id] = $field['label'] . ' - ' . $product['product'];
                        $products['quantity_' . $product_id]  = $product['quantity'];
                        $products['amount_' . $product_id]    = $price;

                        $total += intval($product['quantity']) * floatval($price);

                        $product_id++;
                    }

                }
            }
        }

        if ( $form->has_field( 'single_product' ) ) {

            foreach ( $form->search_fields( 'single_product' ) as $key => $field) {

                if ( isset( $entry_fields[$field['name']] ) && $entry_field =  $entry_fields[$field['name']] ) {

                    if ( $entry_field['quantity'] < 1 ) {
                        continue;
                    }

                    $price    = isset( $field['price'] ) ? $field['price'] : array();
                    $quantity = isset( $field['quantity'] ) ? $field['quantity'] : array();

                    $_price    = isset($price['is_flexible']) && $price['is_flexible'] ? $entry_field['price'] : $price['price'];
                    $_quantity = isset($quantity['status']) && $quantity['status'] ? $entry_field['quantity'] : 1;

                    $products['item_name_' . $product_id] = $field['label'];
                    $products['quantity_' . $product_id]  = $_quantity;
                    $products['amount_' . $product_id]    = $_price;

                    $total += intval($_quantity) * floatval($_price);

                    $product_id++;

                }

            }
        }

        return array( 'products' => $products, 'total' => $total );
    }

    /**
     * Calculate total ammount
     *
     * @param object  $form
     * @param array $entry_fields
     *
     * @return int
     */
    function calculate_total( $form, $entry_fields ) {

        $total = 0;

        if ( $form->has_field( 'multiple_product' ) ) {

            foreach ( $form->search_fields( 'multiple_product' ) as $key => $field) {

                if ( isset( $entry_fields[$field['name']]['value'] ) && $entry_field =  unserialize($entry_fields[$field['name']]['value']) ) {

                    foreach ( $entry_field as $k => $product ) {

                        if ( $product['quantity'] < 1 ) {
                            continue;
                        }

                        $total += intval($product['quantity']) * floatval($product['price']);
                    }

                }
            }
        }

        if ( $form->has_field( 'single_product' ) ) {

            foreach ( $form->search_fields( 'single_product' ) as $key => $field) {


                if ( isset( $entry_fields[$field['name']]['value'] ) && $entry_field =  unserialize($entry_fields[$field['name']]['value']) ) {

                    $price    = isset( $field['price'] ) ? $field['price'] : array();
                    $quantity = isset( $field['quantity'] ) ? $field['quantity'] : array();

                    $_price    = isset($price['is_flexible']) && $price['is_flexible'] ? $entry_field['price'] : $price['price'];
                    $_quantity = isset($quantity['status']) && $quantity['status'] ? $entry_field['quantity'] : 1;

                    $total += intval($_quantity) * floatval($_price);
                }

            }
        }

        return $total;
    }

    /**
     * Insert a new entry
     */
    function insert_entry( $entry_fields, $form_id, $page_id, $form_settings ) {

        if ( !$entry_fields ) {
            return new WP_Error( 'no-entry-fields', __( 'No entry fields was found.', 'weforms-pro' ) );
        }

        if ( !$form_id ) {
            return new WP_Error( 'no-form-id', __( 'No form ID was found.', 'weforms-pro' ) );
        }

        if ( !$page_id ) {
            return new WP_Error( 'no-page-id', __( 'No page ID was found.', 'weforms-pro' ) );
        }

        if ( !$form_settings) {
            return new WP_Error( 'no-form-setting', __( 'No for setting was found.', 'weforms-pro' ) );
        }

        $entry_id = weforms_insert_entry( [
            'form_id' => $form_id,
        ], $entry_fields );

        do_action( 'weforms_entry_submission', $entry_id, $form_id, $page_id, $form_settings );

        $notification = new WeForms_Notification( [
            'form_id'  => $form_id,
            'page_id'  => $page_id,
            'entry_id' => $entry_id,
        ] );
        $notification->send_notifications();

        return $entry_id;
    }


    /**
     * Insert a new payment
     *
     * @param  array $args
     *
     * @return WP_Error|integer
     */
    function insert_payment( $args ) {
        global $wpdb;

        $defaults = array(
            'form_id'        => 0,
            'entry_id'       => 0,
            'user_id'        => get_current_user_id(),
            'total'          => 0,
            'gateway'        => 'paypal',
            'transaction_id' => NULL,
            'status'         => 'unpaid',
            'created_at'     => current_time( 'mysql' )
        );

        $data = wp_parse_args( $args, $defaults );

        if ( !$data['form_id'] ) {
            return new WP_Error( 'no-form-id', __( 'No form ID was found.', 'weforms-pro' ) );
        }

        if ( !$data['entry_id'] ) {
            return new WP_Error( 'no-entry-id', __( 'No entry ID was found.', 'weforms-pro' ) );
        }

        if ( !$data['gateway'] ) {
            return new WP_Error( 'no-gateway', __( 'No payment method gateway was found.', 'weforms-pro' ) );
        }

        $success = $wpdb->insert( $wpdb->prefix . 'weforms_payments' , $data );

        if ( is_wp_error( $success ) || ! $success ) {
            return new WP_Error( 'could-not-create', __( 'Could not create an payment', 'weforms-pro' ) );
        }

        $payment_id = $wpdb->insert_id;

        if ( is_wp_error( $payment_id ) ) {

            error_log('Error inserting weForms payment data. ' . $payment_id->get_error_message() . ' ' . json_encode($data) );
        }

        return $payment_id;
    }

    /**
     * Change an entry status
     *
     * @param  int $entry_id
     * @param  string $status
     *
     * @return int|boolean
     */
    function weforms_change_entry_status( $entry_id, $status ) {
        global $wpdb;

        return $wpdb->update( $wpdb->weforms_payments,
            array( 'status'     => $status ),
            array( 'entry_id'   => $entry_id ),
            array( '%s' ),
            array( '%d' )
        );
    }

    /**
     * Delete an entry
     *
     * @param  int $entry_id
     *
     * @return int|boolean
     */
    function weforms_delete_entry( $entry_id ) {
        global $wpdb;

        return $wpdb->delete( $wpdb->weforms_payments, array( 'entry_id' => $entry_id ), array( '%d' ) );
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_panel( ) {
        include_once dirname( __FILE__ ) . '/component/template.php';
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {

        $tabs[ $this->id ] = array(
            'label' => $this->title,
            'icon' => $this->icon,
        );

        return $tabs;
    }


    /**
     * Set Default Settings
     *
     * @param array $settings
     *
     * @return array
     */
    function set_default_settings( $settings ) {

        $default_settings    = array(
            'currency'               => 'USD',
            'currency_position'      => 'left',
            'price_thousand_sep'     => ',',
            'price_decimal_sep'      => '.',
            'price_num_decimals'     => '2',
            'payment_redirect_to'    => '',
            'payment_default_fields' => true,

            'paypal_email'           => '',
            'paypal_logo'            => '',
            'paypal_ask_shipping'    => 0,
            'paypal_message'         => __( 'You are being redirected to a PayPal page for the payment.', 'weforms-pro' ),

            'stripe_key'             => '',
            'stripe_secret_key'      => '',

            'stripe_key_test'        => '',
            'stripe_secret_key_test' => '',
            'stripe_company_name'    => get_bloginfo( 'name' ),
            'stripe_company_desc'    => get_bloginfo( 'description' ),
            'stripe_ask_billing'     => '',
            'stripe_ask_shipping'    => '',
            'stripe_logo'            => '',

            'stripe_message'         => __( "Please wait while the payment popup is loading.", 'weforms-pro' ),
        );

        return array_merge($default_settings, $settings);
    }


    /**
     * Update the total filed value
     *
     * @param array $entry_fields
     * @param object $form
     * @param array $form_settings
     * @param array $form_fields
     *
     * @return array
     */
    public function save_total_value( $entry_fields, $form, $form_settings, $form_fields ) {

        if ( $form->has_field('total') ) {

            $products = $this->preapre_products( $form, $entry_fields );

            foreach ( $form->search_fields('total')  as $key => $field ) {
                $entry_fields[$field['name']] = $products['total'];
            }
        }

        return $entry_fields;
    }

    /**
     * Get all payments
     *
     * @return void
     */
    public function get_payments() {
        check_ajax_referer( 'weforms' );

        $this->check_admin();

        $form_id      = isset( $_REQUEST['id'] ) ? intval( $_REQUEST['id'] ) : 0;
        $current_page = isset( $_REQUEST['page'] ) ? intval( $_REQUEST['page'] ) : 1;
        $per_page     = 20;
        $offset       = ( $current_page - 1 ) * $per_page;

        if ( ! $form_id ) {
            wp_send_json_error( __( 'No form id provided!', 'weforms-pro' ) );
        }

        $entries = weforms_get_form_payments( $form_id, array(
            'number' => $per_page,
            'offset' => $offset
        ) );

        $total_entries   = weforms_count_form_payments( $form_id );

        $columns         = array(
            'total'          => 'Amount',
            'transaction_id' => 'Transaction ID',
            'created_at'     => 'Created',
        );


        array_map( function( $entry ) use ($columns) {
            $entry->id = $entry->entry_id;
            $entry->fields = array();

            foreach ($columns as $meta_key => $label) {

                switch ($meta_key) {
                    case 'transaction_id':

                        if ( 'paypal' === $entry->gateway ) {
                            $value = $entry->{$meta_key};

                        } elseif ( 'stripe' === $entry->gateway ){
                            $value = sprintf(
                                "<a href='https://dashboard.stripe.com/payments/%s' target='_blank'>%s</a>",
                                $entry->$meta_key,
                                $entry->$meta_key
                            );
                        }
                        break;

                    case 'total':

                        $value = weforms_format_price( $entry->{$meta_key} );
                        break;

                    case 'status':
                        $value = ucfirst($entry->{$meta_key});
                        break;

                    default:
                        $value = $entry->{$meta_key};
                        break;
                }

                $entry->fields[$meta_key] = $value;
            }

        }, $entries );

        $entries         = apply_filters('weforms_get_payments', $entries, $form_id );

        $response = array(
            'columns'    => $columns,
            'entries'    => $entries,
            'form_title' => get_post_field( 'post_title', $form_id ),
            'pagination' => array(
                'total'    => $total_entries,
                'per_page' => $per_page,
                'pages'    => ceil( $total_entries / $per_page ),
                'current'  => $current_page
            )
        );

        wp_send_json_success( $response );
    }



    /**
     * Bulk trash entries
     *
     * @return void
     */
    public function bulk_delete_payments() {
        check_ajax_referer( 'weforms' );

        $this->check_admin();

        $entry_ids = isset( $_POST['ids'] ) ? array_map( 'absint', $_POST['ids'] ) : array();

        if ( ! $entry_ids ) {
            wp_send_json_error( __( 'No entry ids provided!', 'weforms-pro' ) );
        }

        global $wpdb;

        foreach ($entry_ids as $entry_id) {

            $wpdb->delete( $wpdb->prefix . 'weforms_payments' , array( 'entry_id' => $entry_id ), array( '%d' ) );
        }

        wp_send_json_success();
    }

    /**
     * Render default fields required for payments
     *
     * @param object $form
     * @param array $form_fields
     * @param array $form_settings
     *
     * @return void
     */
    public function maybe_populate_payment_fields( $form, $form_fields, $form_settings ) {

        if ( ! weforms_get_settings('payment_default_fields', true) ) {
            return;
        }

        if ( ($form->has_field('multiple_product') || $form->has_field('single_product') ) && ! $form->has_field('payment_method') ) {

            if ( empty($form_settings['payment_paypal_deactivate'] ) && empty($form_settings['payment_stripe_deactivate'] ) ) {

                if ( $field_object = weforms()->fields->field_exists( 'payment_method' ) ) {

                    $field_object->render(
                        array_merge(
                            $field_object->get_field_props(),
                            array(
                                'name' => 'weforms_default_payment_method' )
                            ),
                        $form->id
                    );
                }
            }
        }

        if ( $form->has_field('multiple_product') || count( $form->search_fields('single_product') ) > 1 ) {

            if ( ! $form->has_field('total') ) {
                 if ( $field_object = weforms()->fields->field_exists( 'total' ) ) {
                    $field_object->render( $field_object->get_field_props() , $form->id);
                 }
            }

        }
    }

    public function register_payment_rest_api_integration( $class_map ) {
        $class_map[dirname( __FILE__ )  .'/api/class-weforms-transaction-controller.php'] = 'Weforms_Transaction_Controller';

        return $class_map;
    }

}

new WeForms_Payment();
