<?php
/**
 * Plugin Name: PDF Form Submission
 * Description: Download PDF of Form submissions
 * Plugin URI: https://wedevs.com/weforms/
 * Thumbnail URL: pdf.svg
 * Author: weDevs
 * Version: 1.0
 * Author URI: https://wedevs.com
 *
 */

/**
 * PDF CLass
 */
class WeForms_PDF {

    /**
     * Default Settings
     *
     * @var arrray
     */
    private $default_settings = array();


    /**
     * Fields we want to use for building PDF
     *
     * @var array
     */
    private $valid_fields = array();

    /**
     * Init
     */
    function __construct() {

        $this->default_settings = array(
            'pdf_header_type'    => 'title',
            'pdf_custom_html'    => '',
            'pdf_logo'           => '',
            'pdf_show_form_name' => true,
            'pdf_show'           => true,
        );


        $this->valid_fields = array(
            'name_field',
            'text_field',
            'textarea_field',
            'dropdown_field',
            'radio_field',
            'checkbox_field',
            'multiple_select',
            'website_url',
            'email_address',
            'image_upload',
            'date_field',
            'file_upload',
            'country_list_field',
            'numeric_text_field',
            'address_field',
            'ratings',
            'repeat_field',
            'signature_field'
            // 'custom_hidden_field',
            // 'section_break',
            // 'custom_html',
            // 'shortcode',
            // 'action_hook',
            // 'google_map',
            // 'recaptcha',
            // 'toc',
        );

        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
        add_action( 'weforms_settings_tab_content_pdf', array( $this, 'settings_panel' ) );

        add_action( 'admin_post_weforms_pdf_download', array( $this, 'download_pdf' ) );

        add_filter( 'weforms_get_entry_columns', array( $this, 'pdf_column' ), 10, 2 );

        add_filter( 'weforms_get_entries', array( $this, 'pdf_link_entry' ), 10, 2 );

        add_filter( 'weforms_get_settings', array( $this, 'set_default_settings' ) );

        add_filter( 'weforms_rest_api_class_map', array( $this, 'register_pdf_rest_api_integration' ), 10, 1 );
    }

    /**
     * Add PDF Colum to entry table
     *
     * @param array $column
     *
     * @param int $form_id
     *
     * @return array
     */
    function pdf_column( $column, $form_id ) {

        $column['_pdf'] = 'PDF';

        return $column;
    }


    /**
     * Set Default Settings
     *
     * @param array $settings
     *
     * @return array
     */
    function set_default_settings( $settings ) {

        foreach ( $this->default_settings as $key => $default_setting ) {

            if ( ! isset( $settings[ $key ] ) ) {

                $settings[ $key ] = $default_setting;
            }
        }

        return $settings;
    }

    /**
     * Add PDF Download to entry row
     *
     * @param array $entries
     *
     * @param int $form_id
     *
     * @return void
     */
    function pdf_link_entry( $entries, $form_id ) {

        $target = weforms_get_settings( 'pdf_show', true ) ? ' target="_blank" ' : '';

        foreach ( $entries as $key => &$entry ) {

            $entry->fields['_pdf'] = '<a ' . $target . 'href="' . admin_url( 'admin-post.php?action=weforms_pdf_download&entry_id=' . $entry->id ) . '">Download</a>';
        }

        return $entries;
    }

    /**
     * Download PDF
     *
     * @return void
     */
    function download_pdf() {

        $entry_id = ! empty( $_REQUEST['entry_id'] ) ? $_REQUEST['entry_id'] : null;
        $form_id  = ! empty( $_REQUEST['form_id'] ) ? $_REQUEST['form_id'] : null;

        if ( ! $entry_id ) {

            wp_die( 'Entry Not Found!' );
        }

        if ( ! $form_id ) {

            global $wpdb;

            $query = 'SELECT form_id FROM ' . $wpdb->weforms_entries . ' WHERE id = %d';

            $form_id = $wpdb->get_var( $wpdb->prepare( $query, $entry_id ) );
        }

        $html      = $this->generate_html( $form_id, $entry_id );
        $file_name = get_the_title( $form_id ) . '_' . $entry_id . '.pdf';

        $this->render_pdf( $html, $file_name );

        die();
    }


    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_panel() {
        include_once dirname( __FILE__ ) . '/views/settings.php';
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {

        $tabs['pdf'] = array(
            'label' => 'PDF Submission',
            'icon'  => WEFORMS_ASSET_URI . '/images/integrations/pdf.svg',
        );

        return $tabs;
    }

    /**
     * Render PDF
     *
     * @return void
     */
    public function render_pdf( $html, $file_name ) {

        if ( ! defined( 'DOMPDF_ENABLE_REMOTE' ) ) {
            define( 'DOMPDF_ENABLE_REMOTE', true );
        }

        require_once __DIR__ . '/vendor/autoload.php';

        // instantiate and use the dompdf class
        $dompdf = new Dompdf\Dompdf();

        $dompdf->loadHtml( $html );

        // (Optional) Setup the paper size and orientation
        $dompdf->setPaper( 'A4', 'portrait' );

        $dompdf->set_option( 'defaultFont', '"DejaVu Sans",Helvetica,Arial,sans-serif' );

        // Render the HTML as PDF
        $dompdf->render();

        // Output the generated PDF to Browser
        $dompdf->stream(
            sanitize_file_name( $file_name ),
            array(
                'Attachment' => ! weforms_get_settings( 'pdf_show', true )
            ) );
    }


    /**
     * Generate HTML of entry, later wil use this to generate PDF
     *
     * @param int $form_id
     *
     * @param int $entry_id
     *
     * @return string
     */
    public function generate_html( $form_id, $entry_id ) {


        $form           = weforms()->form->get( $form_id );
        $entry          = $form->entries()->get( $entry_id );
        $fields         = $entry->get_fields();
        $form_settings  = $form->get_settings();
        $quizForm       = $form_settings['quiz_form'];

        if ( empty( $fields ) || empty( $entry ) ) {
            return;
        }

        $form_name   = get_the_title( $form_id );
        $table       = '';
        $show_empty  = weforms_get_settings( 'pdf_show_empty', false );
        $header_type = weforms_get_settings( 'pdf_header_type', 'title' );

        $table .= '<div class="weforms-pdf">';

        switch ( $header_type ) {
            case 'html':

                $table .= '<div class="header row">';
                $table .= weforms_get_settings( 'pdf_custom_html' );
                $table .= '</div>';

                break;

            case 'title':
                $table .= '<h1 class="row">';
                $table .= get_bloginfo( 'name' );
                $table .= '</h1>';

                break;

            case 'logo':

                if ( $pdf_logo = weforms_get_settings( 'pdf_logo' ) ) {
                    $table .= '<div class="header row">';
                    $table .= '<img src="' . str_replace( home_url( '/' ), ABSPATH, $pdf_logo ) . '" height="60px">';
                    $table .= '</div>';
                }

                break;

        }

        if ( weforms_get_settings( 'pdf_show_form_name', true ) ) {

            $table .= '<h3>' . $form_name . '</h3>';
        }

        if ( $quizForm == 'yes' ) {
            $respondentPoints   = isset($form_settings['total_points']) ? floatval( $form_settings['total_points'] ) : 0 ;
        }
        foreach ( $fields as $field ) {
            if ( ! in_array( $field['type'], $this->valid_fields ) ) {

                continue;
            }

            $field_value = $field['value'];

            if ( empty( $field_value ) && ! $show_empty ) {

                continue;
            }

            if ( is_array( $field_value ) ) {
                $value = implode( ", ", $field_value );

            } elseif ( is_serialized( $field_value ) ) {
                $field_value = unserialize( $field_value );

                $serialized_value = array();

                if ( is_array( $field_value ) ) {

                    foreach ( $field_value as $key => $single_value ) {

                        $single_value       = str_replace( array( "_", "-" ), " ", $key ) . ': ' . $single_value;
                        $single_value       = ucwords( $single_value );
                        $serialized_value[] = $single_value;
                    }

                    $value = implode( "<br> ", $serialized_value );
                }

            } elseif ( $field['type'] == 'signature_field' ) {
				$value = preg_replace( '/(<img src=")(http.*\.png)(">)/', '$2', $field_value );
			} else {
                $field_value = trim( strip_tags( $field_value ) );
                $value       = $field_value;
            }

            if ( $field['type'] == 'country_list_field' ) {
                $value = get_country_name( $field_value );
            }

            if ($field['type'] == 'address_field') {
                $tot = preg_match('/Country Select: (.+)/', $value, $country_code);

                if (isset($country_code[1]) && !empty($country_code[1])) {
                    $country_name = get_country_name( $country_code[1] );
                    $value = str_replace($country_code[1], $country_name, $value);
                }
            }

            if ( $field['type'] == 'signature_field' ) {
                $value  = str_replace( content_url(), WP_CONTENT_DIR, $value );
                $value  = str_replace('\\', '/', $value);
                $data   = file_get_contents( $value );
                $base64 = 'data:image/' . '.png' . ';base64,' . base64_encode( $data );
                $value  = sprintf( '<img src="%s">', $base64 );
            }

            if ( $quizForm == 'yes' ) {
                $selectedAnswers    = isset($field['selected_answers']) ? $field['selected_answers'] : '';
                $givenAnswer        = isset($field['value']) ? $field['value'] : '';
                $options            = isset($field['options']) ? $field['options'] : '';
                $template           = $field['template'];
                $fieldPoints        = isset($field['points']) ? floatval( $field['points'] ) : 0;

                if ( $template == 'radio_field' || $template == 'dropdown_field' ) {
                    $isAnswerRight = true;

                    if ( empty($givenAnswer) ) {
                        $isAnswerRight = false;
                        $respondentPoints  -= $fieldPoints;
                    }else {
                        foreach ($options as $key => $fieldvalue) {
                            if ( $givenAnswer == $fieldvalue ) {
                                if ( $key != $selectedAnswers ) {
                                    $isAnswerRight = false;
                                    $respondentPoints  -= $fieldPoints;
                                }
                            }
                        }
                    }
                } elseif ( $template == 'checkbox_field' || $template == 'multiple_select' ) {
                    $isAnswerRight = true;
                    $userAnswer = [];

                    foreach ($options as $key => $fieldvalue) {
                        foreach ($givenAnswer as $answer) {
                            if ($fieldvalue == $answer) {
                                $userAnswer[] = $key;
                            }
                        }
                    }

                    $userAnswer   = implode('|', $userAnswer);
                    $rightAnswers = implode('|', $selectedAnswers);

                    if ( $userAnswer != $rightAnswers || empty($userAnswer) ) {
                        $isAnswerRight = false;
                        $respondentPoints  -= $fieldPoints;
                    }
                }
            }

            $table .= '<div class="row">';
            $table .= '<div class="label">';
            $table .= '<strong>' . $field['label'] . '</strong>';

            if ( $quizForm == 'yes' ) {
                $table .= '<span class="field-points">';
                $table .= $isAnswerRight ? $field['points'] : 0;
                $table .= '/';
                $table .= $field['points'];
                $table .= '</span>';
            }

            $table .= '</div>';
            $table .= '<div class="data">';
            $table .= $value;
            $table .= '</div>';
            $table .= '</div>';
        }

        if ( $quizForm == 'yes' ) {
            $table .= '<div class="weforms-total-points">';
            $table .= '<strong>';
            $table .= __( 'Total Points:', 'weforms-pro');
            $table .= ' ' . $form_settings['total_points'];
            $table .= '</strong><br>';
            $table .= '<strong>';
            $table .= __( 'Points Earned:', 'weforms-pro');
            $table .= ' ' .$respondentPoints;
            $table .= '</strong>';
            $table .= '</div>';
        }

        $table .= '</div>';

        ob_start();

        ?>

        <style>

            .weforms-pdf {
                font-size: 10pt;
                font-family: 'DejaVu Sans', 'Roboto', sans-serif;
            }

            .weforms-total-points,
            .field-points{
                float: right;
            }

            .weforms-total-points{
                text-align: right;
            }

            .field-points{
                padding-top: 3px;
            }

            .row {
                clear: both;
                padding: 1.25mm 0;
            }

            h3 {
                margin: 1.5mm 0 0.5mm;
                padding: 0;
            }

            ul, ol {
                margin: 0;
                padding-left: 1mm;
                padding-right: 1mm;
            }

            li {
                margin: 0;
                padding: 0;
                width: 100% !important;
                float: none !important;
                list-style-position: inside;
            }

            .alignleft {
                float: left;
            }

            .alignright {
                float: right;
            }

            .aligncenter {
                text-align: center;
            }

            p.alignleft {
                text-align: left;
                float: none;
            }

            p.alignright {
                text-align: right;
                float: none;
            }
        </style>

        <?php echo $table; ?>

        <?php

        return ob_get_clean();
    }

    public function register_pdf_rest_api_integration( $class_map ) {
        $class_map[dirname( __FILE__ )  .'/api/class-weforms-pdf-controller.php'] = 'Weforms_Pdf_Controller';

        return $class_map;
    }
}

new WeForms_PDF();
