<?php

/**
 * Campaign Monitor API  manager class
 *
 * @since 1.3.9
 */

class WeForms_Salesforce_Controller extends Weforms_REST_Controller {

    /**
     * @var object
     */
    private $salesforce;

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'salesforce';

    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->base, array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'salesforce_fetch_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base,
            array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'salesforce_update_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );
    }

    /**
     * get Collection of Salesforce Lists
     *
     * @since 1.3.9
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function salesforce_fetch_lists( $request ) {
        $object   = !empty( $request['type'] ) ? $request['type'] : 'Contact';
        $lists    = get_option( 'weforms_salesforce_sobject_fields_' . $object , array() );
        $data     = array( 'lists' => $lists, 'is_authenticated' => weforms_salesforce_get_access_token() );
        $response = $this->prepare_response_for_collection( $data );
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function salesforce_update_lists( $request ) {
        $object   = !empty( $request['type'] ) ? $request['type'] : 'Contact';
        $lists    = array();
        $response = $this->salesforce()->get_object_fields( $object );

        if ( $response ) {
            foreach ($response['fields'] as $list) {
                if ( empty( $list['createable'] ) || empty( $list['updateable'] ) ) {
                    continue;
                }

                $lists[] = array(
                    'id'       => $list['name'],
                    'name'     => $list['label'],
                    'required' => $list['nillable'] ? false : true,
                );
           }

            update_option( 'weforms_salesforce_sobject_fields_' . $object, $lists );
        }

        $response = $this->prepare_response_for_collection( $lists );
        $response = rest_ensure_response( $response );

        return $response;
    }

     /**
     * Require Salesforece Classes
     *
     * @return
     **/
    function salesforce() {
        if ( $this->salesforce ) {
            return $this->salesforce;
        }

        require_once __DIR__ . '/../includes/class-http-client.php';
        require_once __DIR__ . '/../includes/class-salesforce.php';

        $instance_url  = weforms_salesforce_get_instance_url();
        $access_token  = weforms_salesforce_get_access_token();
        $refresh_token = weforms_salesforce_get_refresh_token();

        $this->salesforce = new \WeDevs\WeForms\Salesforce\Salesforce( $instance_url, $access_token, $refresh_token );

        return $this->salesforce;
    }
}
