<?php

// don't call the file directly
if ( !defined( 'ABSPATH' ) ) exit;

class WeForms_Integration_Salesforce extends WeForms_Pro_Integration {

    /**
     * @var object
     */
    private $salesforce;


    /**
     * Class constructor.
     */
    public function __construct() {

        $this->id                = 'salesforce';
        $this->title             = __( 'Salesforce', 'weforms-pro' );
        $this->icon              = WEFORMS_ASSET_URI . '/images/icon-salesforce.svg';
        $this->template          = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled' => false,
            'list'    => '',
            'type'    => 'Contact',
            'fields'  => array(),
            'formFields' => array(
                array(
                    'salesForceField'  => '',
                    'formField'        => '',
                    'salesForceCustom' => '',
                    'formCustom'       => '',
                )
            ),
        );

        $this->plugin_init();

        $this->load_settings();

        add_filter( 'admin_footer', array( $this, 'load_template' ) );
        add_action( 'admin_post_wefomrs_sales_force_auth', array( $this, 'sales_force_auth') );
        add_filter( 'weforms_get_settings', array( $this, 'set_default_settings' ) );
        add_filter( 'wp_ajax_weforms_salesforce_disconnect', array( $this, 'disconnect' ) );
        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );

        add_action( 'wp_ajax_weforms_salesforce_fetch_lists', array( $this, 'fetch_lists' ) );
        add_action( 'wp_ajax_weforms_salesforce_update_lists', array( $this, 'update_lists' ) );

        add_action( 'weforms_entry_submission', array( $this, 'subscribe_user' ), 10, 4 );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }


    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function subscribe_user( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        $type = !empty( $integration->type ) ? $integration->type : 'Contacts';

        $notification = new WeForms_Notification( array( 'form_id' => $form_id, 'entry_id' => $entry_id ) );
        $notification->set_merge_tags();

        $contactData = array();

        foreach ( $integration->formFields as $key => $field ) {

            $salesForceField = $field->salesForceField;
            $formField       = $field->formField;

            if ( $salesForceField === '_custom_' ) {
                $salesForceField = $field->salesForceCustom;
            }

            if ( $formField === '_custom_' ) {

                $formField = $field->formCustom;
                $value     = $notification->replace_tags($formField);
                $value     = WeForms_Notification::replace_name_tag( $value , $entry_id );

            } else {

                $value = weforms_get_entry_meta( $entry_id, $formField , true );

                if ( is_array( $value ) ) {
                    $value = implode(WeForms::$field_separator, $value);
                }
            }

            $salesForceField = str_replace(" ", "", $salesForceField);

            $contactData[$salesForceField] = $value;
        }

        try {
            $this->salesforce()->create_contact( $type, $contactData );
        } catch( Exception $e ) {
            weforms()->log('error', $e->getMessage() );
        }
    }


    /**
     * Require Salesforece Classes
     *
     * @return
     **/
    function salesforce() {

        if ( $this->salesforce ) {
            return $this->salesforce;
        }

        require_once __DIR__ . '/includes/class-http-client.php';
        require_once __DIR__ . '/includes/class-salesforce.php';

        $instance_url  = weforms_salesforce_get_instance_url();
        $access_token  = weforms_salesforce_get_access_token();
        $refresh_token = weforms_salesforce_get_refresh_token();

        $this->salesforce = new \WeDevs\WeForms\Salesforce\Salesforce( $instance_url, $access_token, $refresh_token );

        return $this->salesforce;
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['salesforce'] = array(
            'label' => __( 'Salesforce', 'weforms-pro' ),
            'icon'  => WEFORMS_PRO_ASSETS . '/images/salesforce.png'
        );

        return $tabs;
    }

    /**
     * Fetch the udpated list from hubspot, save and return it
     *
     * @return array
     */
    public function update_lists() {

        $object = !empty( $_REQUEST['type'] ) ? $_REQUEST['type'] : 'Contact';

        $lists    = array();

        $response = $this->salesforce()->get_object_fields( $object );

        if ( $response ) {

            foreach ($response['fields'] as $list) {

                if ( empty( $list['createable'] ) || empty( $list['updateable'] ) ) {
                    continue;
                }

                $lists[] = array(
                    'id'       => $list['name'],
                    'name'     => $list['label'],
                    'required' => $list['nillable'] ? false : true,
                );
           }

            update_option( 'weforms_salesforce_sobject_fields_' . $object, $lists );
        }

        wp_send_json_success( $lists );
    }


    /**
     * Fetch hubspots saved list from server
     *
     * @return array
     */
    public function fetch_lists() {

        $object = !empty( $_REQUEST['type'] ) ? $_REQUEST['type'] : 'Contact';

        $lists = get_option( 'weforms_salesforce_sobject_fields_' . $object , array() );

        wp_send_json_success( array( 'lists' => $lists, 'is_authenticated' => weforms_salesforce_get_access_token() ) );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scripts
     *
     * @return void
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-salesforce'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Disconnect
     *
     * @param array $settings
     *
     * @return array
     */
    function disconnect() {

        delete_option( 'weforms_settings_salesforce-integration' );

        wp_send_json_success();
    }

    /**
     * Set Default Settings
     *
     * @param array $settings
     *
     * @return array
     */
    function set_default_settings( $settings ) {

        $settings['salesforce_status'] = weforms_salesforce_get_access_token() ? true : false;

        return $settings;
    }

    /**
     * sales_force_auth
     *
     * @return
     **/
    function sales_force_auth( ) {

        // $token = $this->salesforce()->get_token( $_REQUEST['code'] );
        $token = $_REQUEST;

        if ( ! empty( $token['access_token'] ) && ! empty( $token['refresh_token'] ) ) {

            $valid = array( 'access_token', 'refresh_token', 'instance_url', 'scope');

            $options = array();

            foreach ( $token as $key => $value ) {
                if ( in_array( $key, $valid ) ) {
                    $options[$key] = $value;
                }
            }

            weforms_salesforce_update_options( $options );
        }

        wp_redirect( admin_url( '/admin.php?page=weforms#/settings' ) );

        die();
    }

    /**
     * Initialize the class.
     */
    public static function init() {
        static $instance = false;

        if ( ! $instance ) {
            $instance = new self();
        }

        return $instance;
    }

    /**
     * Init the plugin.
     *
     * @return void
     */
    public function plugin_init() {
	    include_once dirname( __FILE__ ) . '/includes/functions.php';

    }

}
