<?php

/**
 * Trello API  manager class
 *
 * @since 1.4.2
 */

class Weforms_Trello_Controller extends Weforms_REST_Controller {

    /**
     * Trello API Key
     *
     * @var string
     */
    const API_KEY = '1cba8600bacc3c5b8e4c6de02eedfc22';

    /**
     * Endpoint namespace
     *
     * @var string
     */
    protected $namespace = 'weforms/v1';

    /**
     * Route name
     *
     * @var string
     */
    protected $base = 'trello';

    /**
     * Register all routes releated with forms
     *
     * @return void
     */
    public function register_routes() {

        register_rest_route( $this->namespace, '/'. $this->base . '/lists', array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'trello_fetch_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base . '/boards', array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'trello_fetch_boards' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base . '/members', array(
                array(
                    'methods'             => WP_REST_Server::READABLE,
                    'callback'            => array( $this, 'trello_fetch_members' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                    'args'     => array(
                        'context' => $this->get_context_param( [ 'default' => 'view' ] )
                    ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base . '/lists', array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'trello_update_lists' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base . '/boards', array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'trello_update_boards' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );

        register_rest_route( $this->namespace, '/'. $this->base . '/members', array(
                array(
                    'methods'             => WP_REST_Server::EDITABLE,
                    'callback'            => array( $this, 'trello_update_members' ),
                    'permission_callback' => array( $this, 'get_items_permissions_check' ),
                ),
            )
        );
    }

    /**
     * get Collection of Trello Lists
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function trello_fetch_lists( $request ) {
        $lists    = get_option( 'weforms_trello_lists', array( ) );
        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;

    }

        /**
     * get Collection of Trello Boards
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function trello_fetch_boards( $request ) {
        $lists    = get_option( 'weforms_trello_boards', array( ) );
        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    /**
     * get Collection of Trello members
     *
     * @since 1.4.2
     *
     * @param WP_REST_Request $request Full details about the request.
     *
     * @return WP_REST_Response|WP_Error Response object on success, or WP_Error object on failure.
     **/
    public function trello_fetch_members( $request ) {
        $lists    = get_option( 'weforms_trello_members', array( ) );
        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;

    }

    public function trello_update_lists( $request ) {
        $lists = get_option( 'weforms_trello_members', array( ) );
        $this->require_trllo();

        $_lists         = get_option( 'weforms_trello_lists', array() );
        $board          = !empty( $request['board'] ) ? $request['board']: '';
        $lists          = array();

        if( ! is_array( $_lists ) ) {
            $_lists = [];
        }

        $trello     = new WeForms_Trello_API( self::API_KEY, $this->get_token() );
        $response   = $trello->getLists( $board );

        if ( $response ) {
            foreach ( $response as $list ) {
                $lists[] = array(
                    'id'     => $list['id'],
                    'name'   => $list['name']
                );
            }

            $_lists[ $board ] = $lists;

            update_option( 'weforms_trello_lists', $_lists );
        }

        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function trello_update_boards( $request ) {
        $this->require_trllo();

        $lists      = array();
        $trello     = new WeForms_Trello_API( self::API_KEY, $this->get_token() );
        $response   = $trello->getBoards( $this->get_user_id() );

        if ( $response ) {
            foreach ( $response as $list ) {
                $lists[] = array(
                    'id'     => $list['id'],
                    'name'   => $list['name']
                );
            }

            update_option( 'weforms_trello_boards', $lists );
        }

        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    public function trello_update_members( $request ) {
        $this->require_trllo();

        $_lists         = get_option( 'weforms_trello_members', array() );
        $board          = !empty( $request['board'] ) ? $request['board']: '';
        $lists          = array();

        if( ! is_array( $_lists ) ) {
            $_lists = [];
        }

        $trello     = new WeForms_Trello_API( self::API_KEY, $this->get_token() );
        $response   = $trello->getBoardMembers( $board );

        if ( $response ) {
            foreach ( $response as $list ) {
                $lists[] = array(
                    'id'     => $list['id'],
                    'name'   => $list['fullName']
                );
            }

            $_lists[ $board ] = $lists;

            update_option( 'weforms_trello_members', $_lists );
        }

        $response = $this->prepare_response_for_collection( $lists);
        $response = rest_ensure_response( $response );

        return $response;
    }

    /**
     * Require the WeForms_Trello_API class if not exists
     *
     * @return void
     */
    public function require_trllo() {
        if ( ! class_exists( 'WeForms_Trello_API' ) ) {
            require_once dirname( __FILE__ ) . '/../class-trello-api.php';
        }
    }

    /**
     * Get the trello token generated from user
     *
     * @return string
     */
    private function get_token() {
        return weforms_get_settings( 'trello_token' );
    }

    /**
     * Get the trello user id / hash
     *
     * @return string
     */
    private function get_user_id() {
        return weforms_get_settings( 'trello_user_id' );
    }
}
