<?php

/**
 * Trello Integration
 */
class WeForms_Integration_Trello extends WeForms_Pro_Integration {

    /**
     * Trello API Key
     *
     * @var string
     */
    const API_KEY = '1cba8600bacc3c5b8e4c6de02eedfc22';

    /**
     * Init
     */
    function __construct() {

        $this->id                = 'trello';
        $this->title             = 'Trello';
        $this->icon              = WEFORMS_ASSET_URI . '/images/icon-trello.svg';
        $this->template          = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled'       => false,
            'board'         => '',
            'list'          => '',
            'member'        => '',
            'fields'  => array(
                'title'         => '',
                'description'   => '',
            ),
        );

        $this->load_settings();

        add_filter( 'admin_footer', array( $this, 'load_template' ) );

        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );

        add_action( 'weforms_entry_submission', array( $this, 'send_event' ), 10, 4 );

        add_action( 'wp_ajax_weforms_trello_verify_token', array( $this, 'verify_token' ) );

        // @TODO: Refactor these and make two function. fetch & update
        add_action( 'wp_ajax_weforms_trello_fetch_lists', array( $this, 'fetch_lists' ) );
        add_action( 'wp_ajax_weforms_trello_update_lists', array( $this, 'update_lists' ) );
        add_action( 'wp_ajax_weforms_trello_fetch_boards', array( $this, 'fetch_boards' ) );
        add_action( 'wp_ajax_weforms_trello_update_boards', array( $this, 'update_boards' ) );
        add_action( 'wp_ajax_weforms_trello_fetch_members', array( $this, 'fetch_members' ) );
        add_action( 'wp_ajax_weforms_trello_update_members', array( $this, 'update_members' ) );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scripts
     *
     * @return void
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-trello'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }


    /**
     * Require the WeForms_Trello_API class if not exists
     *
     * @return void
     */
    public function require_trllo() {
        if ( ! class_exists( 'WeForms_Trello_API' ) ) {
            require_once dirname( __FILE__ ) . '/class-trello-api.php';
        }
    }

    /**
     * Fetch saved list from server
     *
     * @return array
     */
    public function fetch_lists() {
        $lists = get_option( 'weforms_trello_lists', array() );

        wp_send_json_success( $lists );
    }

    /**
     * Fetch saved boards from server
     *
     * @return array
     */
    public function fetch_boards() {
        $lists = get_option( 'weforms_trello_boards', array() );

        wp_send_json_success( $lists );
    }

    /**
     * Fetch saved members from server
     *
     * @return array
     */
    public function fetch_members() {
        $lists = get_option( 'weforms_trello_members', array() );

        wp_send_json_success( $lists );
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['trello'] = array(
            'label' => __( 'Trello', 'weforms-pro' ),
            'icon'  => WEFORMS_PRO_ASSETS . '/images/trello.png'
        );

        return $tabs;
    }

    /**
     * Fetch boards from trello and save to our server
     *
     * @return array
     */
    public function update_boards() {

        $this->require_trllo();

        $lists      = array();

        $trello     = new WeForms_Trello_API( self::API_KEY, $this->get_token() );

        $response   = $trello->getBoards( $this->get_user_id() );

        if ( $response ) {

            foreach ( $response as $list ) {

                $lists[] = array(
                    'id'     => $list['id'],
                    'name'   => $list['name']
                );
            }

            update_option( 'weforms_trello_boards', $lists );
        }

        wp_send_json_success( $lists );
    }

    /**
     * Fetch lists from trello and save to our server
     *
     * @return array
     */
    public function update_lists() {


        $this->require_trllo();

        $_lists         = get_option( 'weforms_trello_lists', array() );
        $board          = !empty( $_REQUEST['board'] ) ? $_REQUEST['board']: '';
        $lists          = array();

        if( ! is_array( $_lists ) ) {
            $_lists = [];
        }

        $trello     = new WeForms_Trello_API( self::API_KEY, $this->get_token() );

        $response   = $trello->getLists( $board );

        if ( $response ) {

            foreach ( $response as $list ) {

                $lists[] = array(

                    'id'     => $list['id'],
                    'name'   => $list['name']
                );
            }

            $_lists[ $board ] = $lists;

            update_option( 'weforms_trello_lists', $_lists );
        }

        wp_send_json_success( $_lists );
    }

    /**
     * Fetch members from trello and save to our server
     *
     * @return array
     */
    public function update_members() {

        $this->require_trllo();

        $_lists         = get_option( 'weforms_trello_members', array() );
        $board          = !empty( $_REQUEST['board'] ) ? $_REQUEST['board']: '';
        $lists          = array();

        if( ! is_array( $_lists ) ) {
            $_lists = [];
        }

        $trello     = new WeForms_Trello_API( self::API_KEY, $this->get_token() );

        $response   = $trello->getBoardMembers( $board );

        // wp_send_json_success( $response );

        if ( $response ) {

            foreach ( $response as $list ) {

                $lists[] = array(

                    'id'     => $list['id'],
                    'name'   => $list['fullName']
                );
            }

            $_lists[ $board ] = $lists;

            update_option( 'weforms_trello_members', $_lists );
        }

        wp_send_json_success( $_lists );
    }


   /**
     * Verify token and
     *
     * @return array
     */
    public function verify_token() {

        $this->require_trllo();

        $trello     = new WeForms_Trello_API( self::API_KEY, $this->get_token() );
        $response   = $trello->getMember();

        wp_send_json_success( $response );
    }


    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function send_event( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->list ) ) {
            return;
        }

        $this->require_trllo();

        $trello         = new WeForms_Trello_API( self::API_KEY, $this->get_token() );
        $form_values    = array();

        foreach ($integration->fields as $key => $value) {

            if ( is_array( $value ) ) {
               $value = implode(WeForms::$field_separator, $value);
            }

            $notification = new WeForms_Notification( array( 'form_id'  => $form_id, 'entry_id' => $entry_id ) );
            $notification->set_merge_tags();

            $value = $notification->replace_tags($value);

            $value = WeForms_Notification::replace_name_tag( $value, $entry_id );

            $form_values[$key] = $value;
        }

        $data = array(
            'name'      => $form_values['title'],
            'desc'      => $form_values['description'],
            'idMembers' => trim( $integration->member, ',' ),
        );

        $trello->createCard( $integration->list, $data );

    }

    /**
     * Get the trello token generated from user
     *
     * @return string
     */
    private function get_token() {

        return weforms_get_settings( 'trello_token' );
    }

    /**
     * Get the trello user id / hash
     *
     * @return string
     */
    private function get_user_id() {

        return weforms_get_settings( 'trello_user_id' );
    }

}

