<?php

/**
 * Zapier Integration
 */
class WeForms_Integration_Zapier extends WeForms_Pro_Integration {

    function __construct() {

        $this->id              = 'zapier';
        $this->title           = __( 'Zapier', 'weforms-pro' );
        $this->icon            = WEFORMS_ASSET_URI . '/images/icon-zapier.svg';
        $this->template        = dirname( __FILE__ ) . '/component/template.php';

        $this->settings_fields = array(
            'enabled' => false,
            'url'    => '',
            'fields'  => array()
        );

        add_filter( 'admin_footer', array( $this, 'load_template' ) );

        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );

        add_action( 'weforms_entry_submission', array( $this, 'subscribe_user' ), 10, 4 );

        add_action( 'weforms_entry_submission', array( $this, 'send_data_to_connected_apps' ), 10, 2 );
    }

    /**
     * Enqueue the mixin
     *
     * @param $scritps
     *
     * @return void
     */
    public function enqueue_mixin( $scritps ) {

        $scritps['weforms-int-zapier'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scritps;
    }

    /**
     * Require the zapier class if not exists
     *
     * @return void
     */
    public function require_zapier() {
        if ( ! class_exists( 'Zapier' ) ) {
            require_once dirname( __FILE__ ) . '/class-zapier.php';
        }
    }

    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function subscribe_user( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        if ( empty( $integration->url ) ) {
            return;
        }

        $data = $this->get_entry_data($entry_id);

        if ( empty( $data ) ) {
            return;
        }

        $this->require_zapier();

        $zapier = new Zapier( $integration->url );

        $response = $zapier->call( json_encode( $data ) );

        return $response;
    }

    /**
     * Send data to zapier connected apps when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function send_data_to_connected_apps( $entry_id, $form_id ) {
        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        // Only run if this form has connections for this provider and entry has fields
        $zaps = get_post_meta( $form_id, 'weforms_zapier', true );

        if ( empty( $zaps ) ) {
            return;
        }

        // Fire for each Zap
        foreach ( $zaps as $zap_id => $zap ) {
            // Only process this Zap if it is enabled
            if ( empty( $zap['hook'] ) ) {
                continue;
            }

            $zapUrl = $zap['hook'];
            $data   = $this->get_entry_data($entry_id);

            if ( empty( $data ) ) {
                return;
            }

            $this->require_zapier();

            $zapier = new Zapier( $zapUrl );

            $response = $zapier->call( json_encode( $data ) );
        }
    }

    /**
     * Get key=>value pair of entry data
     *
     * @param type $entry_id
     *
     * @return array
     */
    function get_entry_data( $entry_id ) {

        $entry      = weforms_get_entry( $entry_id );
        $data       = weforms_get_entry_data( $entry_id );

        if ( !$entry || ! $data ) {

            return array();
        }

        $response   = array(
            'form_title' => get_post_field( 'post_title', $entry->form_id ),
            'created'    => date_i18n( 'F j, Y g:i a', strtotime( $entry->created_at ) ),
            'ip'         => $entry->ip_address,
            'user'       => $entry->user_id ? get_user_by( 'id', $entry->user_id )->display_name : false,
            'referer'    => $entry->referer,
            'device'     => $entry->user_device
        );

        foreach ($data['fields'] as $key => $value ) {

            $response[$key] = $data['data'][$key];
        }

        return $response;
    }
}

