<?php
/**
 * Plugin Name: Zapier Integration
 * Description: Subscribe a contact to Zapier when a form is submited
 * Plugin URI: https://wedevs.com/weforms/
 * Thumbnail URL: zapier-logomark.png
 * Author: weDevs
 * Version: 1.0
 * Author URI: https://wedevs.com
 */
require_once dirname( __FILE__ ) . '/class-integration-zapier.php';
/**
 * Zapier CLass
 */
class WeForms_Zapier {

    function __construct() {
        add_action( 'init', array( $this, 'zapier_callback' ) );
        add_filter( 'weforms_integrations', array( $this, 'register_integration' ) );
    }

    /**
     * Handle API request and connect with Zapier
     * Callback to provide Zapier with specific information for forms and fields
     *
     * @since 1.3.1
     */
    public function zapier_callback() {
        $apikey = $this->get_apikey();
        $data   = array();

        // weForms Zapier API key is required
        if ( empty( $_GET['weforms_zapier'] ) ) {
            return;
        }

        // Callback action is required
        if ( empty( $_GET['weforms_action'] ) ) {
            return;
        }

        // Validate provided API Key
        if ( empty( $apikey ) || trim( $_GET['weforms_zapier'] ) !== $apikey ) {
            // Key is incorrect or missing
            nocache_headers();
            header( 'HTTP/1.1 401 Unauthorized' );
            echo wp_json_encode(
                array(
                    'error' => __( 'Invalid weForms Zapier API key', 'weforms-pro' ),
                )
            );
            exit;
        }

        // Provide available forms
        if ( 'forms' === $_GET['weforms_action'] ) {
            $args  = array(
                'post_type'   => 'wpuf_contact_form',
                'post_status' => array( 'publish', 'draft', 'pending' ),
                'posts_per_page' => -1
            );

            $forms = weforms()->form->get_forms( $args );

            if ( ! empty( $forms['forms'] ) ) {
                $forms_data = array();
                foreach ( $forms['forms'] as $form ) {
                    $forms_data[$form->id] = sanitize_text_field( $form->name );
                }

                $data['forms']['key']       = 'form';
                $data['forms']['label']     = 'Select a Form';
                $data['forms']['required']  = true;
                $data['forms']['choices']   = $forms_data;

                nocache_headers();
                header( 'Content-Type: application/json; charset=utf-8' );
                echo wp_json_encode( $data['forms'] );
                exit;
            }
        }

        // Provide available fields from a recent form entry
        if ( 'entries' === $_GET['weforms_action'] && ! empty( $_GET['weforms_form'] ) ) {
            $entries = weforms_get_form_entries( absint( $_GET['weforms_form'] ) );

            if ( !empty( $entries ) ) {
                $last_entry_id   = $entries[0]->id;
                $last_entry      = weforms_get_entry_data( $last_entry_id );
                $last_entry_data = $last_entry['data'];
            }

            if ( !empty( $last_entry_data ) ) {
                $last_entry_data['id'] = $last_entry_id;
                $data['entries'][] = $last_entry_data;

                nocache_headers();
                header( 'Content-Type: application/json; charset=utf-8' );
                echo wp_json_encode( $data['entries'] );
                exit;
            }
        }

        // Subscribe/Add Zap
        if ( 'subscribe' === $_GET['weforms_action'] ) {
            $form_id = absint( $_GET['weforms_form'] );
            $hook    = ! empty( $_GET['hook_url'] ) ? esc_url_raw( $_GET['hook_url'] ) : '';
            $id      = uniqid();

            $zaps = get_post_meta( $form_id, 'weforms_zapier', true );

            if ( empty( $zaps ) ) {
                $zaps = array();
            }

            $zaps[ $id ] = array(
                'hook' => $hook,
                'date' => time(),
            );

            update_post_meta( $form_id, 'weforms_zapier', $zaps );

            $data = array(
                'status' => 'subscribed',
            );

            nocache_headers();
            header( 'Content-Type: application/json; charset=utf-8' );
            echo wp_json_encode( $data['status'] );
            exit;
        }

        // Unsubscribe/Delete Zap
        if ( 'unsubscribe' === $_GET['weforms_action'] ) {

            $form_id = absint( $_GET['weforms_form'] );
            $url     = ! empty( $_GET['hook_url'] ) ? esc_url_raw( $_GET['hook_url'] ) : '';

            $zaps = get_post_meta( $form_id, 'weforms_zapier', true );

            if ( ! empty( $zaps ) ) {
                foreach ( $zaps as $zap_id => $zap ) {
                    if ( $url === $zap['hook'] ) {
                        unset( $zaps[ $zap_id ] );
                    }
                }
                if ( empty( $zaps ) ) {
                    delete_post_meta( $form_id, 'weforms_zapier' );
                } else {
                    update_post_meta( $form_id, 'weforms_zapier', $zaps );
                }
            }

            $data = array(
                'status' => 'unsubscribed',
            );

            nocache_headers();
            header( 'Content-Type: application/json; charset=utf-8' );
            echo wp_json_encode( $data['status'] );
            exit;
        }

        // If data is empty something went wrong, so we stop
        if ( empty( $data ) ) {
            $data = array(
                'error' => __( 'No data', 'weforms-pro' ),
            );
        }
    }

    /**
     * Returns weForms Zapier API key.
     *
     * If one hasn't been generated yet then we create one and save it.
     *
     * @return string
     *
     * @since 1.3.1
     */
    public function get_apikey() {

        $key = get_option( 'weforms_zapier_apikey' );

        if ( empty( $key ) ) {

            $chars = array_merge( range( 0, 9 ), range( 'a', 'z' ) );
            $key   = '';
            for ( $i = 0; $i < 20; $i ++ ) {
                $key .= $chars[ mt_rand( 0, count( $chars ) - 1 ) ];
            }
            update_option( 'weforms_zapier_apikey', $key );
        }

        return $key;
    }

    /**
     * Register default integrations
     *
     * @param  array $integrations
     *
     * @return array
     */
    public function register_integration( $integrations ) {

        $integrations = array_merge( $integrations, array( 'WeForms_Integration_Zapier' ) );

        return $integrations;
    }
}

new WeForms_Zapier();
