<?php

/**
 * Integration class
 */

class WeForms_Integration_Zoho extends WeForms_Pro_Integration {

    private $zohoApi;

    function __construct() {

        $this->id                = 'zoho';
        $this->title             = __( 'Zoho', 'weforms-pro' );
        $this->icon              = WEFORMS_ASSET_URI . '/images/icon-zoho.svg';
        $this->template          = dirname( __FILE__ ) . '/component/template.php';
        $this->settings_template = dirname( __FILE__ ) . '/views/settings.php';

        $this->settings_fields = array(
            'enabled' => false,
            'type'    => 'Contacts',
            'update'  => 1,
            'fields'  => array(),
            'formFields' => array(
                array(
                    'zohoField'  => '',
                    'formField'  => '',
                    'zohoCustom' => '',
                    'formCustom' => '',
                )
            ),
        );

        $this->load_settings();

        add_action( 'admin_footer', array( $this, 'load_template' ) );
        add_filter( 'weforms_builder_scripts', array( $this, 'enqueue_mixin' ) );
        add_action( 'weforms_entry_submission', array( $this, 'subscribe_user' ), 10, 4 );
        add_action( 'wp_ajax_weforms_update_zoho_login', array( $this, 'getApiToken' ) );
        add_action( 'wp_ajax_weforms_zoho_disconnect', array( $this, 'disconnect' ) );
        add_action( 'wp_ajax_weforms_zoho_check_token', array( $this, 'check' ) );

        // add_filter( 'weforms_after_save_settings', array( $this, 'getApiToken' ) );

        add_filter( 'weforms_get_settings', array( $this, 'set_default_settings' ) );
        add_action( 'weforms_settings_tabs', array( $this, 'settings_tabs' ), 11 );
    }


    /**
     * check
     *
     * @return
     **/
    function check() {
        wp_send_json_success( weforms_get_settings( 'zoho_token' ) );
    }

    /**
     * disconnect
     *
     * @return
     **/
    function disconnect() {

        weforms_update_settings( array('zoho_token' => false ) );

        wp_send_json_success();
    }

    /**
     * Render the settings panel
     *
     * @return void
     */
    public function settings_tabs( $tabs ) {
        $tabs['zoho'] = array(
            'label' => __( 'Zoho', 'weforms-pro' ),
            'icon'  => WEFORMS_ASSET_URI . '/images/integrations/zoho-crm.svg'
        );

        return $tabs;
    }


    /**
     * Get Api Token
     *
     * @return void
     **/
    function getApiToken() {

        $login = !empty( $_REQUEST['zoho_login'] ) ? $_REQUEST['zoho_login'] : '';
        $pass  = !empty( $_REQUEST['zoho_password'] ) ? $_REQUEST['zoho_password'] : '';

        $token = false;
        $error = false;

        if ( $login && $pass) {
            $token = $this->zohoApi()->getApiToken( $login, $pass );
        }

        if ( empty( $token ) ) {
             $error = __( 'Please ensure that your email/username and password is correct.', 'weforms-pro' );
        }

        if ( is_wp_error( $token ) ) {
            $error = $token->get_error_message();
            $token = false;
        }

        weforms_update_settings( array( 'zoho_token' => $token, 'zoho_login' => $login ) );

        if ( $error ) {
            wp_send_json_error( $error );
        } else {
            wp_send_json_success( __( 'Zoho CRM account successfully connected.', 'weforms-pro' ) );
        }
    }

    /**
     * Set Default Settings
     *
     * @param array $settings
     *
     * @return array
     */
    function set_default_settings( $settings ) {

        $default_settings    = array(
            'zoho_login'    => '',
            'zoho_password' => '',
            'zoho_token'    => '',
        );

        return array_merge( $default_settings, $settings );
    }
    /**
     * Enqueue the mixin
     *
     * @param $scritps
     *
     * @return array
     */
    public function enqueue_mixin( $scripts ) {

        $scripts['weforms-int-zoho'] = array(
            'src' => $this->module_component_file( __FILE__ ),
            'deps' => array( 'weforms-form-builder-components' )
        );

        return $scripts;
    }

    /**
     * Get the API key
     *
     * @return string
     */
    private function get_api_key() {
        return weforms_get_settings('zoho_token');
    }

    /**
     * Require the constant contact class if not exists
     *
     * @return void
     */
    public function zohoApi() {
        if ( ! class_exists( 'WeForms_Zoho_API' ) ) {
            require_once dirname( __FILE__ ) . '/includes/class-zoho-api.php';
        }

        if ( ! $this->zohoApi ) {
            $this->zohoApi = new WeForms_Zoho_API( $this->get_api_key() );
        }

        return $this->zohoApi;
    }


    /**
     * Subscribe a user when a form is submitted
     *
     * @param  int $entry_id
     * @param  int $form_id
     * @param  int $page_id
     * @param  array $form_settings
     *
     * @return void
     */
    public function subscribe_user( $entry_id, $form_id, $page_id, $form_settings ) {

        $integration = weforms_is_integration_active( $form_id, $this->id );

        if ( false === $integration ) {
            return;
        }

        $type = !empty( $integration->type ) ? $integration->type : 'Contacts';
        $updateExisting = !empty( $integration->update ) ? true : false;

        $notification = new WeForms_Notification( array( 'form_id' => $form_id, 'entry_id' => $entry_id ) );

        $notification->set_merge_tags();

        $contactData = array();

        foreach ( $integration->formFields as $key => $field ) {

            $zohoField = $field->zohoField;
            $formField = $field->formField;

            if ( $zohoField === '_custom_' ) {
                $zohoField = $field->zohoCustom;
            }

            if ( $formField === '_custom_' ) {

                $formField = $field->formCustom;

                $value = $notification->replace_tags($formField);
                $value = WeForms_Notification::replace_name_tag( $value , $entry_id );

            } else {

                $value = weforms_get_entry_meta( $entry_id, $formField , true );

                if ( is_array( $value ) ) {
                    $value = implode(WeForms::$field_separator, $value);
                }
            }

            $contactData[$zohoField] = $value;
        }

        try{

            if ( $type == 'Leads' ) {
                $this->zohoApi()->addLead( $contactData, $updateExisting );
            } else {
                $this->zohoApi()->addContact( $contactData, $updateExisting );
            }

        } catch ( Exception $e ) {
            weforms()->log('error', $e->getMessage() );
        }
    }
}
