<?php
/**
 * Plugin Name: weForms Pro - Business
 * Plugin URI: https://weformspro.com/
 * Description: The premium version of weForms
 * Version: 1.3.17
 * Author: weForms
 * Author URI: https://weformspro.com/
 * License: GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: weforms-pro
 * Domain Path: /languages
 *
 * @package weFormsPro
 */

/**
 * Copyright (c) 2020 weForms (email: support@weformspro.com). All rights reserved.
 *
 * Released under the GPL license
 * http://www.opensource.org/licenses/gpl-license.php
 *
 * This is an add-on for WordPress
 * http://wordpress.org/
 *
 * **********************************************************************
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 * **********************************************************************
 */

if ( !defined( 'ABSPATH' ) ) exit;

/**
 * WeForms_Pro class
 *
 * @class WeForms_Pro The class that holds the entire WeForms_Pro plugin
 */
class WeForms_Pro {

    /**
     * Premium plan name
     *
     * @var string
     */
    public $pro_plan = 'weforms-business';

    /**
     * Plugin version
     *
     * @var string
     */
    public $version = '1.3.17';


    /**
     * weForms version Required to run weForms Pro
     *
     * @var string
     */
    public $min_core = '1.6.7';

    /**
     * Constructor for the WeForms_Pro class
     *
     * Sets up all the appropriate hooks and actions
     * within our plugin.
     *
     * @uses register_activation_hook()
     * @uses register_deactivation_hook()
     * @uses is_admin()
     * @uses add_action()
     */
    public function __construct() {

        $this->define_constants();

        register_activation_hook( __FILE__, array( $this, 'activate' ) );
        register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );

        add_action( 'init', array( $this, 'initiate_plugin_notice' ) );

        add_action( 'plugins_loaded', array( $this, 'plugin_upgrades') );
        add_action( 'weforms_loaded', array( $this, 'init_plugin' ) );
        add_filter( 'wpuf_free_loader', '__return_false' );

    }

    /**
     * Define the constants
     *
     * @return void
     */
    public function define_constants() {
        define( 'WEFORMS_PRO_VERSION', $this->version );
        define( 'WEFORMS_PRO_FILE', __FILE__ );
        define( 'WEFORMS_PRO_PATH', dirname( WEFORMS_PRO_FILE ) );
        define( 'WEFORMS_PRO_INCLUDES', WEFORMS_PRO_PATH . '/includes' );
        define( 'WEFORMS_PRO_MODULES', WEFORMS_PRO_PATH . '/modules' );
        define( 'WEFORMS_PRO_URL', plugins_url( '', WEFORMS_PRO_FILE ) );
        define( 'WEFORMS_PRO_ASSETS', WEFORMS_PRO_URL . '/assets' );
    }

    /**
     * Load the plugin after weForms is loaded
     *
     * @return void
     */
    public function init_plugin() {

        if ( ! $this->is_core_supported() ) {
            add_filter( 'wpuf_free_loader', '__return_true' );
            return;
        }

        $this->includes();
        $this->init_hooks();
    }

    /**
     * Initializes the WeForms_Pro() class
     *
     * Checks for an existing WeForms_Pro() instance
     * and if it doesn't find one, creates it.
     */
    public static function init() {
        static $instance = false;

        if ( ! $instance ) {
            $instance = new WeForms_Pro();
        }

        return $instance;
    }

    /**
     * Placeholder for activation function
     *
     * Nothing being called here yet.
     */
    public function activate() {

        $this->create_tables();
        $this->maybe_activate_modules();
        $this->maybe_set_default_settings();

        update_option( 'weforms_pro_version', WEFORMS_PRO_VERSION );
    }

    /**
     * Create the table schema
     *
     * @return void
     */
    public function create_tables() {

        global $wpdb;

        $collate = '';

        if ( $wpdb->has_cap( 'collation' ) ) {
            if ( ! empty($wpdb->charset ) ) {
                $collate .= "DEFAULT CHARACTER SET $wpdb->charset";
            }

            if ( ! empty($wpdb->collate ) ) {
                $collate .= " COLLATE $wpdb->collate";
            }
        }

        $table_schema = array(
            "CREATE TABLE `{$wpdb->prefix}weforms_payments` (
                `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                `form_id` bigint(20) unsigned DEFAULT NULL,
                `entry_id` bigint(20) unsigned DEFAULT NULL,
                `user_id` bigint(20) unsigned DEFAULT NULL,
                `total` float(20) unsigned DEFAULT NULL,
                `gateway` varchar(50) DEFAULT NULL,
                `transaction_id` varchar(50) DEFAULT NULL,
                `status` varchar(10) DEFAULT 'completed',
                `payment_data` longtext,
                `created_at` datetime DEFAULT NULL,
                PRIMARY KEY (`id`),
                KEY `form_id` (`form_id`),
                KEY `entry_id` (`entry_id`),
                KEY `user_id` (`user_id`)
            ) $collate;",
        );

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        foreach ( $table_schema as $table ) {
            dbDelta( $table );
        }
    }

    /**
     * Do plugin upgrades
     *
     * @since 1.1.1
     *
     * @return void
     */
    function plugin_upgrades() {

        if ( ! is_admin() && ! current_user_can( 'manage_options' ) ) {
            return;
        }

        require_once WEFORMS_PRO_INCLUDES . '/class-upgrades.php';

        $upgrader = new WeForms_Upgrades_Pro();

        if ( $upgrader->needs_update() ) {
            $upgrader->perform_updates();
        }
    }

    /**
     * Placeholder for deactivation function
     *
     * Nothing being called here yet.
     */
    public function deactivate() {

    }

    /**
     * Initialize the hooks
     *
     * @return void
     */
    public function init_hooks() {

        // Localize our plugin
        add_action( 'init', array( $this, 'localization_setup' ) );

        // init the classes
        add_action( 'init', array( $this, 'init_classes' ), 15 );

        // email gateways
        add_filter( 'weforms_email_gateways', array( $this, 'register_email_gateways' ) );
        add_filter( 'weforms_rest_api_class_map', array( $this, 'register_rest_api_integration' ) );
    }

    /**
     * [register_rest_api_integration description]
     * @param  [type] $class_map [description]
     * @return [type]            [description]
     */
    public function register_rest_api_integration( $class_map ) {
        add_filter( 'weforms_form_access_capability', array( $this, 'grant_admin_permission' ) );

        $class_map[ WEFORMS_PRO_INCLUDES . '/api/class-form-report-controller.php'] = 'WeForms_Form_Report_Controller';
        $class_map[ WEFORMS_PRO_INCLUDES . '/api/class-modules-controller.php']     = 'Weforms_Modules_Controller';

        return $class_map;
    }

    /**
     * Grants admin access capability for other roles
     *
     * @param  string $capability
     *
     * @return string
     */
    public function grant_admin_permission( $capability ) {
        $capability = weforms_get_settings( 'permission', 'manage_options' );

        return $capability;
    }

    /**
     * Include the required files
     *
     * @return void
     */
    public function includes() {

        require_once WEFORMS_PRO_INCLUDES . '/functions.php';
        require_once WEFORMS_PRO_INCLUDES . '/functions-modules.php';
        require_once WEFORMS_PRO_INCLUDES . '/class-integration-pro.php';
        require_once WEFORMS_PRO_INCLUDES . '/class-modules.php';
        require_once WEFORMS_PRO_INCLUDES . '/class-fields-manager.php';
        require_once WEFORMS_PRO_INCLUDES . '/class-updater.php';
        require_once WEFORMS_PRO_INCLUDES . '/class-form-report.php';
        require_once WEFORMS_PRO_INCLUDES . '/class-resend-notifications.php';

        require_once WEFORMS_PRO_INCLUDES . '/class-scripts-styles.php';

        if ( is_admin() ) {
            require_once WEFORMS_PRO_INCLUDES . '/class-admin.php';
        }

        // load all the active modules
        $modules = weforms_pro_get_active_modules();

        if ( $modules ) {
            foreach ($modules as $module_file) {
                $module_path = WEFORMS_PRO_MODULES . '/' . $module_file;

                if ( file_exists( $module_path ) ) {
                    include_once $module_path;
                }
            }
        }
    }

    /**
     * Instantiate the classes
     *
     * @return void
     */
    public function init_classes() {
        new WeForms_Pro_Modules();
        // new WeForms_Pro_Fields();
        new WeForms_Pro_Fields_Manager();
        new WeForms_Pro_Script_Styles();

        if ( current_user_can( 'manage_options' )  ) {
            new WeForms_Pro_Updater( $this->pro_plan );
        }

        if ( is_admin() ) {
            new WeForms_Pro_Admin();
        }
    }

    /**
     * Initialize plugin for localization
     *
     * @uses load_plugin_textdomain()
     */
    public function localization_setup() {
        load_plugin_textdomain( 'weforms-pro', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
    }

    /**
     * Activate all the modules for the first time
     *
     * @return void
     */
    public function maybe_activate_modules() {
        global $wpdb;

        $has_installed = $wpdb->get_row( "SELECT option_id FROM {$wpdb->options} WHERE option_name = 'weforms_pro_active_modules'" );

        if ( $has_installed ) {
            return;
        }

        if ( ! function_exists( 'weforms_pro_get_modules' ) ) {
            require_once WEFORMS_PRO_INCLUDES . '/functions-modules.php';
        }

        $modules = weforms_pro_get_modules();

        if ( $modules ) {
            foreach ($modules as $module_file => $data) {
                weforms_pro_activate_module( $module_file );
            }
        }
    }

    /**
     * Set the required default settings key if not present
     *
     * This is required for setting up the component settings data
     *
     * @return void
     */
    public function maybe_set_default_settings() {
        $requires_update = false;
        $settings        = get_option( 'weforms_settings', array() );
        $additional_keys = array(
            'mailchimp_api_key', 'campaign_monitor_key', 'constant_contact_key'
        );

        foreach ($additional_keys as $key) {
            if ( ! isset( $settings[ $key ] ) ) {
                $settings[ $key ] = '';

                $requires_update = true;
            }
        }

        if ( $requires_update ) {
            update_option( 'weforms_settings', $settings );
        }
    }

    /**
     * Register email gateways
     *
     * @param  array $gateways
     *
     * @return array
     */
    public function register_email_gateways( $gateways ) {
        require_once WEFORMS_PRO_INCLUDES . '/emailers/class-emailer-sparkpost.php';
        require_once WEFORMS_PRO_INCLUDES . '/emailers/class-emailer-mailgun.php';
        require_once WEFORMS_PRO_INCLUDES . '/emailers/class-emailer-sendgrid.php';

        $gateways['sparkpost'] = new WeForms_Emailer_SparkPost();
        $gateways['mailgun']   = new WeForms_Emailer_Mailgun();
        $gateways['sendgrid']  = new WeForms_Emailer_SendGrid();

        return $gateways;
    }

    /**
    * Initiate free version prompt notice
    *
    * @since 1.0.1
    *
    * @return void
    **/
    public function initiate_plugin_notice() {

        // Check already weForms free version is available or not
        if ( ! class_exists( 'WeForms' ) ) {

            if ( ! current_user_can( 'manage_options' ) ) {
                return;
            }

            add_action( 'admin_notices', array( $this, 'activation_notice' ) );
            add_action( 'wp_ajax_weforms_free_version_install', array( $this, 'install_core_plugins' ) );
            return;
        }

        if ( ! $this->is_core_supported() ) {
            add_action( 'admin_notices', array( $this, 'weforms_version_notice' ) );
        }
    }

    /**
    * weForms free plugin activation notice
    *
    * @since 1.0.1
    *
    * @return void
    **/
    public function activation_notice() {
        $plugin_file = basename( dirname( __FILE__ ) ) . '/weforms-pro.php';
        ?>
        <div class="updated" id="weforms-free-installer-notice" style="padding: 1em; position: relative;">
            <h2><?php _e( 'Your weForms Pro is almost ready!', 'weforms-pro' ); ?></h2>
            <a href="<?php echo wp_nonce_url( 'plugins.php?action=deactivate&amp;plugin=' . $plugin_file . '&amp;plugin_status=all&amp;paged=1&amp;s=', 'deactivate-plugin_' . $plugin_file ); ?>" class="notice-dismiss" style="text-decoration: none;" title="<?php _e( 'Dismiss this notice', 'weforms-pro' ); ?>"></a>
            <p><?php echo sprintf( __( "You just need to install the %sCore Plugin%s to make it functional.", "weforms-pro" ), '<a target="_blank" href="https://wordpress.org/plugins/weforms/">', '</a>' ); ?></p>
            <p>
                <button id="weforms-free-installer" class="button"><?php _e( 'Install Now', 'weforms-pro' ); ?></button>
            </p>
        </div>

        <script type="text/javascript">
            (function ($) {
                $('#weforms-free-installer-notice #weforms-free-installer').click( function (e) {
                    e.preventDefault();

                    $(this).addClass('install-now updating-message');
                    $(this).text('<?php echo esc_js( 'Installing...', 'weforms-pro' ); ?>');

                    var data = {
                        action: 'weforms_free_version_install',
                        _wpnonce: '<?php echo wp_create_nonce('weforms-free-installer-nonce'); ?>'
                    };

                    $.post(ajaxurl, data, function (response) {
                        if (response.success) {
                            var activateButton = $('#weforms-free-installer-notice #weforms-free-installer');
                            activateButton.attr('disabled', 'disabled');
                            activateButton.removeClass('install-now updating-message');
                            activateButton.text('<?php echo esc_js( 'Installed', 'weforms-pro' ); ?>');
                            window.location.reload();
                        }
                    });
                });
            })(jQuery);
        </script>
        <?php
    }

    /**
    * Install weForms free version
    *
    * @since 1.0.1
    *
    * @return void
    **/
    public function install_core_plugins() {
        if ( ! isset( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'weforms-free-installer-nonce' ) ) {
            wp_send_json_error( __( 'Error: Nonce verification failed', 'weforms-pro' ) );
        }

        include_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        include_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';

        $weforms_plugin_basename = 'weforms/weforms.php';

        $upgrader = new Plugin_Upgrader( new WP_Ajax_Upgrader_Skin() );

        // First check if weforms exsit or not
        if ( file_exists( WP_PLUGIN_DIR . '/' . $weforms_plugin_basename ) ) {
            // if exist then just activate
            activate_plugin( $weforms_plugin_basename );
        } else {
            // Plugin not found so download it form wp.org server
            $api    = plugins_api( 'plugin_information', array( 'slug' => 'weforms', 'fields' => array( 'sections' => false ) ) );
            $result = $upgrader->install( $api->download_link );
            activate_plugin( $weforms_plugin_basename );
        }

        wp_send_json_success();
    }

    /**
     * Check if weForms core is competable with pro
     *
     * @return void
     */
    function is_core_supported() {

        if ( version_compare( WEFORMS_VERSION, $this->min_core , '<' ) ) {
            return false;
        }

        return true;
    }


    /**
     * Show notice about PHP version
     *
     * @return void
     */
    function weforms_version_notice() {

        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $error = __( 'Your installed weForms Version is: ', 'weforms' ) . WEFORMS_VERSION . '. ';
        $error .= __( 'The <strong>weForms Pro</strong> plugin requires version <strong>', 'weforms' ) . $this->min_core . __( '</strong> or greater.', 'weforms' );
        ?>
        <div class="error">
            <p><?php printf( $error ); ?></p>
        </div>
        <?php
    }


} // WeForms_Pro


/**
 * Initialize the plugin
 *
 * @return \WeForms_Pro
 */
function weforms_pro() {
    return WeForms_Pro::init();
}

// kick-off.
weforms_pro();
