<?php

// phpcs:ignore Generic.Commenting.DocComment.MissingShort
/** @noinspection AutoloadingIssuesInspection */

namespace WPFormsSignatures\Fields;

use WPForms\Pro\Helpers\Upload;
use WPForms_Field;
use WPForms\Integrations\Divi\Divi;

/**
 * Signature field.
 *
 * @since 1.0.0
 */
class Signature extends WPForms_Field {

	/**
	 * Handle name for wp_register_styles handle.
	 *
	 * @since 1.5.0
	 *
	 * @var string
	 */
	const HANDLE = 'wpforms-signatures';

	/**
	 * Init class.
	 *
	 * @since 1.0.0
	 */
	public function init() {

		// Define field type information.
		$this->name     = esc_html__( 'Signature', 'wpforms-signatures' );
		$this->keywords = esc_html__( 'user, e-signature', 'wpforms-signatures' );
		$this->type     = 'signature';
		$this->icon     = 'fa-pencil';
		$this->order    = 310;
		$this->group    = 'fancy';

		$this->hooks();
	}

	/**
	 * Add hooks.
	 *
	 * @since 1.8.0
	 */
	private function hooks() {

		// Form frontend javascript.
		add_action( 'wpforms_frontend_js', [ $this, 'frontend_js' ] );

		// Form frontend CSS.
		add_action( 'wpforms_frontend_css', [ $this, 'frontend_css' ] );

		// Field styles for Gutenberg.
		add_action( 'enqueue_block_editor_assets', [ $this, 'gutenberg_enqueues' ] );

		// Admin form builder enqueues.
		add_action( 'wpforms_builder_enqueues', [ $this, 'admin_builder_enqueues' ] );

		// Set editor style for block type editor. Must run at 20 in add-ons.
		add_filter( 'register_block_type_args', [ $this, 'register_block_type_args' ], 20, 2 );

		// Admin form builder default field settings.
		add_filter( 'wpforms_field_new_default', [ $this, 'admin_builder_defaults' ] );

		// Customize HTML field values.
		add_filter( 'wpforms_html_field_value', [ $this, 'field_html_value' ], 10, 4 );

		// Define additional field properties.
		add_filter( 'wpforms_field_properties_signature', [ $this, 'field_properties' ], 5, 3 );

		// Delete uploaded files from entry.
		add_action( 'wpforms_pre_delete_entries', [ $this, 'delete_uploaded_files' ], 10, 2 );
	}

	/**
	 * Enqueue frontend field js.
	 *
	 * @since 1.0.0
	 *
	 * @param array $forms Forms on the current page.
	 */
	public function frontend_js( $forms ) {

		if (
			wpforms()->get( 'frontend' )->assets_global() ||
			wpforms_has_field_type( 'signature', $forms, true )
		) {

			$min = wpforms_get_min_suffix();

			wp_enqueue_script(
				'wpforms-signature_pad',
				WPFORMS_SIGNATURES_URL . 'assets/js/signature_pad.min.js',
				[],
				'2.3.2',
				true
			);

			wp_enqueue_script(
				'wpforms-signature',
				WPFORMS_SIGNATURES_URL . "assets/js/wpforms-signatures{$min}.js",
				[ 'jquery', 'wpforms-signature_pad' ],
				WPFORMS_SIGNATURES_VERSION,
				true
			);
		}
	}

	/**
	 * Enqueue frontend field CSS.
	 *
	 * @since 1.0.0
	 *
	 * @param array $forms Forms on the current page.
	 */
	public function frontend_css( $forms ) {

		if (
			wpforms()->get( 'frontend' )->assets_global() ||
			wpforms_has_field_type( 'signature', $forms, true )
		) {

			$min = wpforms_get_min_suffix();

			wp_enqueue_style(
				self::HANDLE,
				WPFORMS_SIGNATURES_URL . "assets/css/wpforms-signatures{$min}.css",
				[],
				WPFORMS_SIGNATURES_VERSION
			);

			$this->enqueue_integrations( $min );
		}
	}

	/**
	 * Admin form builder enqueues.
	 *
	 * @since 1.0.0
	 */
	public function admin_builder_enqueues() {

		$min = wpforms_get_min_suffix();

		wp_enqueue_style(
			'wpforms-builder-signatures',
			WPFORMS_SIGNATURES_URL . "assets/css/admin-builder-signatures{$min}.css",
			[],
			WPFORMS_SIGNATURES_VERSION
		);
	}

	/**
	 * Load enqueues for the Gutenberg editor.
	 *
	 * @since 1.1.3
	 */
	public function gutenberg_enqueues() {

		if ( version_compare( get_bloginfo( 'version' ), '5.5', '>=' ) ) {
			return;
		}

		$min = wpforms_get_min_suffix();

		wp_enqueue_style(
			self::HANDLE,
			WPFORMS_SIGNATURES_URL . "assets/css/wpforms-signatures{$min}.css",
			[],
			WPFORMS_SIGNATURES_VERSION
		);
	}

	/**
	 * Set editor style for block type editor.
	 *
	 * @see WPForms_Field_File_Upload::register_block_type_args
	 *
	 * @since 1.5.0
	 *
	 * @param array  $args       Array of arguments for registering a block type.
	 * @param string $block_type Block type name including namespace.
	 */
	public function register_block_type_args( $args, $block_type ) {

		if ( $block_type !== 'wpforms/form-selector' ) {
			return $args;
		}

		$min = wpforms_get_min_suffix();

		// CSS.
		wp_register_style(
			self::HANDLE,
			WPFORMS_SIGNATURES_URL . "assets/css/wpforms-signatures{$min}.css",
			[ $args['editor_style'] ],
			WPFORMS_SIGNATURES_VERSION
		);

		$args['editor_style'] = self::HANDLE;

		return $args;
	}

	/**
	 * Field defaults when creating new field.
	 *
	 * Default size to large.
	 *
	 * @since 1.0.0
	 *
	 * @param array $field Current field settings.
	 *
	 * @return array
	 */
	public function admin_builder_defaults( $field ) {

		if ( $field['type'] === 'signature' && empty( $field['size'] ) ) {
			$field['size'] = 'large';
		}

		return $field;
	}

	/**
	 * Return signature link/image for HTML supported values.
	 *
	 * @since 1.0.0
	 *
	 * @param string $value     Field value.
	 * @param array  $field     Field settings.
	 * @param array  $form_data Form data and settings.
	 * @param string $context   Value display context.
	 *
	 * @return string
	 */
	public function field_html_value( $value, $field, $form_data = [], $context = '' ) {

		if ( ! empty( $field['value'] ) && $field['type'] === 'signature' ) {

			$value = sanitize_text_field( $field['value'] );

			return sprintf(
				'<a href="%s" rel="noopener noreferrer" target="_blank" style="max-width:500px;display:block;margin:0;"><img src="%s" style="max-width:100%%;display:block;margin:0;" alt=""></a>',
				$value,
				$value
			);
		}

		return $value;
	}

	/**
	 * Define additional field properties.
	 *
	 * @since 1.1.0
	 *
	 * @param array $properties Field properties.
	 * @param array $field      Field settings.
	 * @param array $form_data  Form data and settings.
	 *
	 * @return array
	 */
	public function field_properties( $properties, $field, $form_data ) {

		// Disable autocomplete for the hidden field.
		$properties['inputs']['primary']['attr']['autocomplete'] = 'off';

		$properties['inputs']['primary']['class'] = [ 'wpforms-signature-input', 'wpforms-screen-reader-element' ];

		if ( ! empty( $properties['inputs']['primary']['required'] ) ) {
			$properties['inputs']['primary']['class'][] = 'wpforms-field-required';
		}

		$properties['inputs']['primary']['data']['is-wrapped-field'] = true;

		return $properties;
	}

	/**
	 * Whether current field can be populated dynamically.
	 *
	 * @since 1.5.0
	 *
	 * @param array $properties Field properties.
	 * @param array $field      Current field specific data.
	 *
	 * @return bool
	 */
	public function is_dynamic_population_allowed( $properties, $field ) {

		return false;
	}

	/**
	 * Whether current field can be populated dynamically.
	 *
	 * @since 1.5.0
	 *
	 * @param array $properties Field properties.
	 * @param array $field      Current field specific data.
	 *
	 * @return bool
	 */
	public function is_fallback_population_allowed( $properties, $field ) {

		return false;
	}

	/**
	 * Field options panel inside the builder.
	 *
	 * @since 1.0.0
	 *
	 * @param array $field Field settings.
	 */
	public function field_options( $field ) {
		/*
		 * Basic field options.
		 */
		// Options open markup.
		$this->field_option(
			'basic-options',
			$field,
			[
				'markup' => 'open',
			]
		);

		// Label.
		$this->field_option( 'label', $field );

		// Description.
		$this->field_option( 'description', $field );

		// Required toggle.
		$this->field_option( 'required', $field );

		// Options close markup.
		$this->field_option(
			'basic-options',
			$field,
			[
				'markup' => 'close',
			]
		);

		/*
		 * Advanced field options.
		 */

		// Options open markup.
		$this->field_option(
			'advanced-options',
			$field,
			[
				'markup' => 'open',
			]
		);

		// Ink color picker.
		$lbl = $this->field_element(
			'label',
			$field,
			[
				'slug'    => 'ink_color',
				'value'   => esc_html__( 'Ink Color', 'wpforms-signatures' ),
				'tooltip' => esc_html__( 'Select the color for the signature ink.', 'wpforms-signatures' ),
			],
			false
		);

		$ink_color = isset( $field['ink_color'] ) ? wpforms_sanitize_hex_color( $field['ink_color'] ) : '';
		$ink_color = empty( $ink_color ) ? '#000000' : $ink_color;

		$fld = $this->field_element(
			'color',
			$field,
			[
				'slug'  => 'ink_color',
				'value' => $ink_color,
				'data'  => [
					'fallback-color' => $ink_color,
				],
			],
			false
		);

		$this->field_element(
			'row',
			$field,
			[
				'slug'    => 'ink_color',
				'content' => $lbl . $fld,
				'class'   => 'color-picker-row',
			]
		);

		// Custom CSS classes.
		$this->field_option( 'css', $field );

		// Size.
		$this->field_option( 'size', $field );

		// Hide label.
		$this->field_option( 'label_hide', $field );

		// Options close markup.
		$this->field_option(
			'advanced-options',
			$field,
			[
				'markup' => 'close',
			]
		);
	}

	/**
	 * Field preview inside the builder.
	 *
	 * @since 1.0.0
	 *
	 * @param array $field Field settings.
	 */
	public function field_preview( $field ) {

		// Label.
		$this->field_preview_option( 'label', $field );

		// Signature placeholder.
		echo '<div class="wpforms-signature-wrap"></div>';

		// Description.
		$this->field_preview_option( 'description', $field );
	}

	/**
	 * Field display on the form front-end.
	 *
	 * @since 1.0.0
	 *
	 * @param array $field      Field settings.
	 * @param array $field_atts Deprecated array.
	 * @param array $form_data  Form data and settings.
	 */
	public function field_display( $field, $field_atts, $form_data ) {

		// Define data.
		$color = ! empty( $field['ink_color'] ) ? wpforms_sanitize_hex_color( $field['ink_color'] ) : '#000000';
		$size  = ! empty( $field['size'] ) ? ' wpforms-field-' . sanitize_html_class( $field['size'] ) : '';

		if ( version_compare( wpforms()->version, '1.8.1', '>=' ) ) {
			$this->field_display_hidden_input( $field );
		}

		// Signature element wrapper.
		printf( '<div class="wpforms-signature-wrap%s">', esc_attr( $size ) );

		// Signature canvas.
		printf(
			'<canvas class="wpforms-signature-canvas" id="wpforms-%d-field_%d-signature" data-color="%s"></canvas>',
			absint( $form_data['id'] ),
			absint( $field['id'] ),
			esc_attr( $color )
		);

		// Clear button to reset canvas.
		printf(
			'<button class="wpforms-signature-clear" title="%s">%s</button>',
			esc_attr__( 'Clear Signature', 'wpforms-signatures' ),
			esc_html__( 'Clear Signature', 'wpforms-signatures' )
		);

		echo '</div>';

		if ( version_compare( wpforms()->version, '1.8.1', '<' ) ) {
			$this->field_display_hidden_input( $field );
		}
	}

	/**
	 * Display field's hidden input that contains dataURL.
	 *
	 * @since 1.7.0
	 *
	 * @param array $field Field settings.
	 */
	private function field_display_hidden_input( $field ) {

		if ( empty( $field['properties']['inputs']['primary'] ) ) {
			return;
		}

		$primary = $field['properties']['inputs']['primary'];

		// Hidden input that contains dataURL.
		printf(
			'<input type="text" %s %s>',
			wpforms_html_attributes( $primary['id'], $primary['class'], $primary['data'], $primary['attr'] ),
			esc_attr( $primary['required'] )
		);
	}

	/**
	 * Validate signature on form submit.
	 *
	 * @since 1.0.0
	 *
	 * @param int    $field_id     Field ID.
	 * @param string $field_submit Submitted form field value.
	 * @param array  $form_data    Form data and settings.
	 */
	public function validate( $field_id, $field_submit, $form_data ) {

		$form_id = absint( $form_data['id'] );

		// Basic required check - If field is marked as required, check for entry data.
		if ( ! empty( $form_data['fields'][ $field_id ]['required'] ) && empty( $field_submit ) && $field_submit !== '0' ) {

			wpforms()->get( 'process' )->errors[ $form_id ][ $field_id ] = wpforms_get_required_label();

			return;
		}

		// Simple format check.
		if ( ! empty( $field_submit ) && substr( $field_submit, 0, 22 ) !== 'data:image/png;base64,' ) {

			wpforms()->get( 'process' )->errors[ $form_id ][ $field_id ] = esc_html__( 'Invalid signature image format', 'wpforms-signatures' );

			return;
		}

		// Image check.
		$base = str_replace( 'data:image/png;base64,', '', $field_submit );

		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
		if ( ! empty( $field_submit ) && ! imagecreatefromstring( base64_decode( $base ) ) ) {
			wpforms()->get( 'process' )->errors[ $form_id ][ $field_id ] = esc_html__( 'Invalid signature image format', 'wpforms-signatures' );
		}
	}

	/**
	 * Delete uploaded files on entry or form delete.
	 *
	 * @since 1.8.0
	 *
	 * @param int|string $column_value Column value.
	 * @param string     $column_name  Column name.
	 */
	public function delete_uploaded_files( $column_value, string $column_name ) {

		switch ( $column_name ) {
			case 'entry_id':
				$this->delete_uploaded_files_by_entry_id( $column_value );
				break;

			case 'form_id':
				$this->delete_uploaded_files_by_form_id( $column_value );
				break;
		}
	}

	/**
	 * Delete uploaded files by entry ID.
	 *
	 * @since 1.9.0
	 *
	 * @param int|string $entry_id Entry ID.
	 */
	private function delete_uploaded_files_by_entry_id( $entry_id ) {

		$entry_obj = wpforms()->get( 'entry' );

		if ( ! $entry_obj ) {
			return;
		}

		$entry = $entry_obj->get( (int) $entry_id );

		if ( empty( $entry ) ) {
			return;
		}

		$this->delete_uploaded_signature_files_from_entry( $entry );
	}

	/**
	 * Delete uploaded files by form ID.
	 *
	 * @since 1.9.0
	 *
	 * @param int|string $form_id Form ID.
	 */
	private function delete_uploaded_files_by_form_id( $form_id ) {

		$entry_obj = wpforms()->get( 'entry' );

		if ( ! $entry_obj ) {
			return;
		}

		$all_entries = $entry_obj->get_entries(
			[
				'form_id' => (int) $form_id,
				'number'  => '-1',
			]
		);

		if ( empty( $all_entries ) ) {
			return;
		}

		foreach ( $all_entries as $entry ) {
			$this->delete_uploaded_signature_files_from_entry( $entry );
		}
	}

	/**
	 * Delete uploaded files from an entry.
	 *
	 * @since 1.8.0
	 *
	 * @depecated 1.9.0
	 *
	 * @param int|mixed $entry_id Entry ID.
	 */
	public function delete_uploaded_files_from_entry( $entry_id ) {

		_deprecated_function( __METHOD__, '1.9.0 of the WPForms plugin' );

		$entry_id  = (int) $entry_id;
		$entry_obj = wpforms()->get( 'entry' );

		if ( ! $entry_obj ) {
			return;
		}

		$entry = $entry_obj->get( $entry_id );

		if ( empty( $entry ) ) {
			return;
		}

		$fields_to_delete = (array) wpforms_decode( $entry->fields );
		$form_id          = (int) $entry->form_id;
		$files_path       = $this->get_form_files_path( $form_id );

		if ( ! is_dir( $files_path ) ) {
			$files_path = $this->get_form_files_path_backward_fallback( $form_id );
		}

		foreach ( $fields_to_delete as $field ) {
			if ( isset( $field['type'] ) && $field['type'] === 'signature' ) {
				$this->delete_uploaded_file( $files_path, $field );
			}
		}
	}

	/**
	 * Delete uploaded files from an entry.
	 *
	 * @since 1.9.0
	 *
	 * @param object $entry Entry object.
	 */
	private function delete_uploaded_signature_files_from_entry( $entry ) {

		$fields_to_delete = (array) wpforms_decode( $entry->fields );
		$form_id          = (int) $entry->form_id;
		$files_path       = $this->get_form_files_path( $form_id );

		if ( ! is_dir( $files_path ) ) {
			$files_path = $this->get_form_files_path_backward_fallback( $form_id );
		}

		foreach ( $fields_to_delete as $field ) {
			if ( isset( $field['type'] ) && $field['type'] === 'signature' ) {
				$this->delete_uploaded_file( $files_path, $field );
			}
		}
	}

	/**
	 * Get the form files path.
	 *
	 * @since 1.8.0
	 *
	 * @param int $form_id Form ID.
	 *
	 * @return string
	 */
	private function get_form_files_path( int $form_id ): string {

		$form_obj = wpforms()->get( 'form' );

		if ( ! $form_obj ) {
			return '';
		}

		$form_data  = $form_obj->get( $form_id );
		$upload_dir = wpforms_upload_dir();

		return (
			trailingslashit( $upload_dir['path'] ) .
			( new Upload() )->get_form_directory( $form_data->ID, $form_data->post_date )
		);
	}

	/**
	 * Fallback method to get Form files path for already existing uploads with incorrectly generated hashes
	 * (files uploaded before version 1.7.6 ).
	 *
	 * @since 1.8.0
	 *
	 * @param int $form_id Form ID.
	 *
	 * @return string
	 */
	private function get_form_files_path_backward_fallback( int $form_id ): string {

		$form_obj = wpforms()->get( 'form' );

		if ( ! $form_obj ) {
			return '';
		}

		$form_data  = $form_obj->get( $form_id );
		$upload_dir = wpforms_upload_dir();

		return (
			trailingslashit( $upload_dir['path'] ) .
			absint( $form_data->ID ) .
			'-' . md5( $form_data->post_date . $form_data->ID )
		);
	}

	/**
	 * Delete an uploaded file.
	 *
	 * @since 1.8.0
	 *
	 * @param string $files_path Path to files.
	 * @param array  $field      Field data.
	 */
	private function delete_uploaded_file( string $files_path, array $field ) {

		if ( empty( $field['value'] ) ) {
			return;
		}

		$file = trailingslashit( $files_path ) . pathinfo( $field['value'], PATHINFO_BASENAME );

		if ( is_file( $file ) ) {
			wp_delete_file( $file );
		}
	}

	/**
	 * Format and sanitize field.
	 *
	 * @since 1.0.0
	 *
	 * @param int    $field_id     Field ID.
	 * @param string $field_submit Submitted form data.
	 * @param array  $form_data    Form data and settings.
	 *
	 * @noinspection NonSecureUniqidUsageInspection
	 */
	public function format( $field_id, $field_submit, $form_data ) { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.TooHigh

		// Define data.
		$name                 = ! empty( $form_data['fields'][ $field_id ]['label'] ) ? sanitize_text_field( $form_data['fields'][ $field_id ]['label'] ) : '';
		$value                = sanitize_text_field( $field_submit );
		$uploads              = wp_upload_dir();
		$form_directory       = absint( $form_data['id'] ) . '-' . md5( $form_data['id'] . $form_data['created'] );
		$wpforms_uploads_root = trailingslashit( $uploads['basedir'] ) . 'wpforms';
		$wpforms_uploads_form = trailingslashit( $wpforms_uploads_root ) . $form_directory;
		$file_name            = sanitize_file_name( esc_html__( 'signature', 'wpforms-signatures' ) . '-' . uniqid() . '.png' );
		$file_new             = trailingslashit( $wpforms_uploads_form ) . $file_name;
		$file_url             = trailingslashit( $uploads['baseurl'] ) . 'wpforms/' . trailingslashit( $form_directory ) . $file_name;

		// Double check we have a image passed.
		if ( ! empty( $value ) && substr( $value, 0, 22 ) === 'data:image/png;base64,' ) {

			// Check for form upload directory destination.
			if ( ! file_exists( $wpforms_uploads_form ) ) {
				wp_mkdir_p( $wpforms_uploads_form );
			}

			// Check if the index.html exists in the root uploads director, if not create it.
			if ( ! file_exists( trailingslashit( $wpforms_uploads_root ) . 'index.html' ) ) {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_file_put_contents
				file_put_contents( trailingslashit( $wpforms_uploads_root ) . 'index.html', '' );
			}

			// Check if the index.html exists in the form uploads director, if not create it.
			if ( ! file_exists( trailingslashit( $wpforms_uploads_form ) . 'index.html' ) ) {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_file_put_contents
				file_put_contents( trailingslashit( $wpforms_uploads_form ) . 'index.html', '' );
			}

			// Compile image data.
			// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode
			$data = base64_decode( preg_replace( '#^data:image/\w+;base64,#i', '', $value ) );

			// Save image.
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_file_put_contents
			$save_signature = file_put_contents( $file_new, $data );

			if ( $save_signature === false ) {

				$value = '';

				wpforms_log(
					esc_html( 'Upload Error, could not upload signature' ),
					$file_url,
					[
						'type'    => [
							'entry',
							'error',
						],
						'form_id' => $form_data['id'],
					]
				);

			} else {

				// Everything's done, so we provide the URL to the image.
				$value = $file_url;

				// Set correct file permissions.
				$stat  = stat( dirname( $file_new ) );
				$perms = $stat['mode'] & 0000666;

				// phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
				@chmod( $file_new, $perms );
			}
		} else {

			$value = '';
		}

		wpforms()->get( 'process' )->fields[ $field_id ] = [
			'name'  => $name,
			'value' => $value,
			'id'    => absint( $field_id ),
			'type'  => $this->type,
		];
	}

	/**
	 * Enqueue Divi integration styles.
	 *
	 * @since 1.8.0
	 *
	 * @param string $min Minified suffix.
	 */
	private function enqueue_integrations( string $min ) {

		if ( ! Divi::is_divi_loaded() ) {
			return;
		}

		wp_enqueue_style(
			self::HANDLE . '-divi',
			WPFORMS_SIGNATURES_URL . "assets/css/integrations/divi/wpforms-signatures{$min}.css",
			[],
			WPFORMS_SIGNATURES_VERSION
		);
	}
}
