<?php
/**
 * @author WP Cloud Plugins
 * @copyright Copyright (c) 2023, WP Cloud Plugins
 *
 * @since       2.0
 * @see https://www.wpcloudplugins.com
 */

namespace TheLion\OutoftheBox;

use TheLion\OutoftheBox\API\Dropbox\Dropbox;
use TheLion\OutoftheBox\API\Dropbox\DropboxApp;

class App
{
    /**
     * The single instance of the class.
     *
     * @var App
     */
    protected static $_instance;

    /**
     * @var bool
     */
    private $_own_app = false;

    /**
     * @var string
     */
    private $_app_key = 'm3n3zyvyr59cdjb';

    /**
     * @var string
     */
    private $_app_secret = 'eu73x5upk7ehes4';

    /**
     * @var \TheLion\OutoftheBox\API\Dropbox\Dropbox
     */
    private static $_sdk_client;

    /**
     * @var \TheLion\OutoftheBox\API\Dropbox\DropboxApp
     */
    private static $_sdk_client_app;

    /**
     * @var \TheLion\OutoftheBox\Account
     */
    private static $_current_account;

    /**
     * We don't save your data or share it.
     * It is used for an easy and one-click  authorization process that will always work!
     *
     * @var string
     */
    private $_auth_url = 'https://www.wpcloudplugins.com:443/out-of-the-box/_AuthorizeApp.php';

    public function __construct()
    {
        require_once OUTOFTHEBOX_ROOTDIR.'/vendors/dropbox-sdk/vendor/autoload.php';

        // Call back for refresh token function in SDK client
        add_action('out-of-the-box-refresh-token', [$this, 'refresh_token'], 10, 1);

        $own_key = Processor::instance()->get_setting('dropbox_app_key');
        $own_secret = Processor::instance()->get_setting('dropbox_app_secret');

        if (
            (!empty($own_key))
            && (!empty($own_secret))
        ) {
            $this->_app_key = Processor::instance()->get_setting('dropbox_app_key');
            $this->_app_secret = Processor::instance()->get_setting('dropbox_app_secret');
            $this->_own_app = true;
        }

        add_filter('outofthebox-set-root-namespace-id', [$this, 'get_root_namespace_id'], 10, 1);
    }

    /**
     * App Instance.
     *
     * Ensures only one instance is loaded or can be loaded.
     *
     * @return App - App instance
     *
     * @static
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            $app = new self();
        } else {
            $app = self::$_instance;
        }

        if (empty($app::$_sdk_client)) {
            try {
                $app->start_sdk_client(App::get_current_account());
            } catch (\Exception $ex) {
                self::$_instance = $app;

                return self::$_instance;
            }
        }

        self::$_instance = $app;

        if (null !== App::get_current_account()) {
            $app->get_sdk_client(App::get_current_account());
        }

        return self::$_instance;
    }

    public function process_authorization()
    {
        $redirect = admin_url('admin.php?page=OutoftheBox_settings');
        if (isset($_GET['network']) || Processor::instance()->is_network_authorized()) {
            $redirect = network_admin_url('admin.php?page=OutoftheBox_network_settings');
        }

        if (empty($_GET['state']) || empty($_GET['ver'])) {
            // Close oAuth popup and refresh admin page. Only possible with inline javascript.
            echo '<script type="text/javascript">window.opener.parent.location.href = "'.$redirect.'"; window.close();</script>';

            exit;
        }
        $state = strtr($_GET['state'], '-_~', '+/=');

        $csrfToken = $state;
        $urlState = null;

        $splitPos = strpos($state, '|');

        if (false !== $splitPos) {
            $csrfToken = substr($state, 0, $splitPos);
            $urlState = substr($state, $splitPos + 1);
        }
        $redirectto = base64_decode($urlState);

        if (false === strpos($redirectto, 'outofthebox_authorization')) {
            return false;
        }

        Processor::reset_complete_cache();

        if (isset($_GET['code'])) {
            $this->create_access_token();
        }

        // Close oAuth popup and refresh admin page. Only possible with inline javascript.
        echo '<script type="text/javascript">window.opener.parent.location.href = "'.$redirect.'"; window.close();</script>';

        exit;
    }

    public function has_plugin_own_app()
    {
        return $this->_own_app;
    }

    public function get_auth_url($params = [])
    {
        $auth_helper = self::get_sdk_client()->getAuthHelper();

        if (Processor::instance()->is_network_authorized() || is_network_admin()) {
            $redirect = network_admin_url('admin.php?page=OutoftheBox_network_settings&action=outofthebox_authorization');
        } else {
            $redirect = admin_url('admin.php?page=OutoftheBox_settings&action=outofthebox_authorization');
        }

        $redirect .= '&license='.(string) License::get();
        $redirect .= '&siteurl='.License::get_home_url();

        $encodedredirect = strtr(base64_encode($redirect), '+/=', '-_~');

        return $auth_helper->getAuthUrl($this->_auth_url, $params, $encodedredirect);
    }

    public function start_sdk_client(Account $account = null)
    {
        self::$_sdk_client = new Dropbox($this->get_dropbox_app($account), ['persistent_data_store' => new \TheLion\OutoftheBox\API\Dropbox\Store\DatabasePersistentDataStore()]);

        return $this->get_sdk_client($account);
    }

    public function refresh_token(Account $account = null)
    {
        $authorization = $account->get_authorization();
        $access_token = $authorization->get_access_token();

        if (!flock($authorization->get_token_file_handle(), LOCK_EX | LOCK_NB)) {
            error_log('[WP Cloud Plugin message]: '.sprintf('Wait till another process has renewed the Authorization Token'));

            /*
             * If the file cannot be unlocked and the last time
             * it was modified was 1 minute, assume that
             * the previous process died and unlock the file manually
             */
            $requires_unlock = ((filemtime($authorization->get_token_location()) + 60) < time());

            // Temporarily workaround when flock is disabled. Can cause problems when plugin is used in multiple processes
            if (false !== strpos(ini_get('disable_functions'), 'flock')) {
                $requires_unlock = false;
            }

            if ($requires_unlock) {
                $authorization->unlock_token_file();
            }

            if (flock($authorization->get_token_file_handle(), LOCK_SH)) {
                clearstatcache();
                rewind($authorization->get_token_file_handle());
                $token = fread($authorization->get_token_file_handle(), filesize($authorization->get_token_location()));
                $decrypted_token = Helpers::decrypt($token);
                $token_object = unserialize($decrypted_token);
                error_log('[WP Cloud Plugin message]: '.sprintf('New Authorization Token has been received by another process.'));
                self::$_sdk_client->setAccessToken($token_object);
                $authorization->unlock_token_file();

                return self::$_sdk_client;
            }
        }

        // Stop if we need to get a new AccessToken but somehow ended up without a refreshtoken
        $refresh_token = $access_token->getRefreshToken();

        if (empty($refresh_token)) {
            error_log('[WP Cloud Plugin message]: '.sprintf('No Refresh Token found during the renewing of the current token. We will stop the authorization completely.'));
            $authorization->set_is_valid(false);
            $authorization->unlock_token_file();
            $this->revoke_token($account);

            return false;
        }

        // Refresh token
        try {
            $new_accesstoken = self::$_sdk_client->getAuthHelper()->refreshToken();

            // Store the new token
            $authorization->set_access_token($new_accesstoken);
            $authorization->unlock_token_file();
            self::get_sdk_client()->setAccessToken($new_accesstoken);

            if (false !== ($timestamp = wp_next_scheduled('outofthebox_lost_authorisation_notification', ['account_id' => $account->get_id()]))) {
                wp_unschedule_event($timestamp, 'outofthebox_lost_authorisation_notification', ['account_id' => $account->get_id()]);
            }
        } catch (\Exception $ex) {
            $authorization->set_is_valid(false);
            $authorization->unlock_token_file();
            error_log('[WP Cloud Plugin message]: '.sprintf('Cannot refresh Authorization Token'));

            if (!wp_next_scheduled('outofthebox_lost_authorisation_notification', ['account_id' => $account->get_id()])) {
                wp_schedule_event(time(), 'daily', 'outofthebox_lost_authorisation_notification', ['account_id' => $account->get_id()]);
            }

            Processor::reset_complete_cache();

            throw $ex;
        }

        return self::$_sdk_client;
    }

    public function create_access_token()
    {
        try {
            $code = $_REQUEST['code'];
            $state = $_REQUEST['state'];

            // Fetch the AccessToken
            $accessToken = self::get_sdk_client()->getAuthHelper()->getAccessToken($code, $state, $this->_auth_url);
            self::$_sdk_client->setAccessToken($accessToken);

            $account_data = self::get_sdk_client()->getCurrentAccount();
            $root_info = $account_data->getRootInfo();
            $root_namespace_id = $root_info['root_namespace_id'];

            $account = new Account($account_data->getAccountId(), $account_data->getDisplayName(), $account_data->getEmail(), $root_namespace_id, $account_data->getAccountType(), $account_data->getProfilePhotoUrl());
            $account->get_authorization()->set_access_token($accessToken);
            $account->get_authorization()->unlock_token_file();

            if ($account_data->emailIsVerified()) {
                $account->set_is_verified(true);
            }

            Accounts::instance()->add_account($account);

            delete_transient('outofthebox_'.$account->get_id().'_is_authorized');
        } catch (\Exception $ex) {
            error_log('[WP Cloud Plugin message]: '.sprintf('Cannot generate Access Token: %s', $ex->getMessage()));

            return new \WP_Error('broke', esc_html__('Error communicating with API:', 'wpcloudplugins').$ex->getMessage());
        }

        return true;
    }

    public function revoke_token(Account $account)
    {
        error_log('[WP Cloud Plugin message]: Lost authorization');

        // Reset Private Folders Back-End if the account it is pointing to is deleted
        $private_folders_data = Processor::instance()->get_setting('userfolder_backend_auto_root', []);
        if (is_array($private_folders_data) && isset($private_folders_data['account']) && $private_folders_data['account'] === $account->get_id()) {
            Processor::instance()->set_setting('userfolder_backend_auto_root', []);
        }

        Processor::reset_complete_cache();

        if (false !== ($timestamp = wp_next_scheduled('outofthebox_lost_authorisation_notification', ['account_id' => $account->get_id()]))) {
            wp_unschedule_event($timestamp, 'outofthebox_lost_authorisation_notification', ['account_id' => $account->get_id()]);
        }

        Core::instance()->send_lost_authorisation_notification($account->get_id());

        try {
            $this->get_sdk_client($account)->getAuthHelper()->revokeAccessToken();
            Accounts::instance()->remove_account($account->get_id());
        } catch (\Exception $ex) {
            error_log('[Out-of-the-Box  message]: '.$ex->getMessage());
        }

        delete_transient('outofthebox_'.$account->get_id().'_is_authorized');
    }

    public function get_app_key()
    {
        return $this->_app_key;
    }

    public function get_app_secret()
    {
        return $this->_app_secret;
    }

    public function set_app_key($_app_key)
    {
        $this->_app_key = $_app_key;
    }

    public function set_app_secret($_app_secret)
    {
        $this->_app_secret = $_app_secret;
    }

    /**
     * @param null|\TheLion\OutoftheBox\Account $account
     *
     * @return \TheLion\OutoftheBox\API\Dropbox\Dropbox
     */
    public static function get_sdk_client($account = null)
    {
        if (!empty($account)) {
            self::set_current_account($account);
        }

        return self::$_sdk_client;
    }

    /**
     * @deprecated
     *
     * @return \TheLion\OutoftheBox\API\Dropbox\Dropbox
     */
    public function get_client()
    {
        Helpers::is_deprecated('function', 'get_client()', 'get_sdk_client($account = null)');

        return self::get_sdk_client();
    }

    /**
     * @param null|mixed $account
     *
     * @return \TheLion\OutoftheBox\API\Dropbox\DropboxApp
     */
    public function get_dropbox_app($account = null)
    {
        if (empty(self::$_sdk_client_app)) {
            if (!empty($account)) {
                self::$_sdk_client_app = new DropboxApp($this->get_app_key(), $this->get_app_secret(), $account->get_authorization()->get_access_token());
            } else {
                self::$_sdk_client_app = new DropboxApp($this->get_app_key(), $this->get_app_secret());
            }
        }

        return self::$_sdk_client_app;
    }

    public function get_root_namespace_id($id = '')
    {
        $use_app_folder = Processor::instance()->get_setting('use_app_folder', 'No');
        if ('Yes' === $use_app_folder) {
            return '';
        }

        $use_team_folders = Processor::instance()->get_setting('use_team_folders', 'No');

        if ('No' === $use_team_folders) {
            return $id;
        }

        $current_account = App::get_current_account();

        if (empty($current_account)) {
            return $id;
        }

        return $current_account->get_root_namespace_id();
    }

    /**
     * @return \TheLion\OutoftheBox\Accounts
     */
    public function get_accounts()
    {
        return Accounts::instance();
    }

    /**
     * @return \TheLion\OutoftheBox\Account
     */
    public static function get_current_account()
    {
        if (empty(self::$_current_account)) {
            if (null !== Processor::instance()->get_shortcode()) {
                $account = Accounts::instance()->get_account_by_id(Processor::instance()->get_shortcode_option('account'));
                if (!empty($account)) {
                    self::set_current_account($account);
                }
            }
        }

        return self::$_current_account;
    }

    public static function set_current_account(Account $account)
    {
        if (self::$_current_account !== $account) {
            self::$_current_account = $account;
            Cache::instance_unload();

            if ($account->get_authorization()->has_access_token()) {
                if (empty(self::$_sdk_client)) {
                    self::instance();
                }

                self::$_sdk_client->setAccessToken($account->get_authorization()->get_access_token());
            }
        }

        return self::$_current_account;
    }

    public static function set_current_account_by_id($account_id)
    {
        $account = Accounts::instance()->get_account_by_id($account_id);

        if (empty($account)) {
            error_log(sprintf('[WP Cloud Plugin message]: APP Error on line %s: Cannot use the requested account (ID: %s) as it is not linked with the plugin. Plugin falls back to primary account.', __LINE__, $account_id));
            $account = Accounts::instance()->get_primary_account();
        }

        return self::set_current_account($account);
    }

    public static function clear_current_account()
    {
        self::$_current_account = null;
        Cache::instance_unload();
    }

    public function get_auth_uri()
    {
        return $this->_auth_url;
    }
}
