<?php
/**
 * @author WP Cloud Plugins
 * @copyright Copyright (c) 2023, WP Cloud Plugins
 *
 * @since       2.0
 * @see https://www.wpcloudplugins.com
 */

namespace TheLion\OutoftheBox;

class UserFolders
{
    /**
     * The single instance of the class.
     *
     * @var UserFolders
     */
    protected static $_instance;

    /**
     * @var string
     */
    private $_user_name_template;
    private $_user_folder_name;

    public function __construct()
    {
        $this->_user_name_template = Processor::instance()->get_setting('userfolder_name');

        $shortcode = Processor::instance()->get_shortcode();
        if (!empty($shortcode) && !empty($shortcode['user_folder_name_template'])) {
            $this->_user_name_template = $shortcode['user_folder_name_template'];
        }
    }

    /**
     * UserFolders Instance.
     *
     * Ensures only one instance is loaded or can be loaded.
     *
     * @return UserFolders - UserFolders instance
     *
     * @static
     */
    public static function instance()
    {
        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }

        return self::$_instance;
    }

    public function get_auto_linked_folder_name_for_user()
    {
        $shortcode = Processor::instance()->get_shortcode();
        if (!isset($shortcode['user_upload_folders']) || 'auto' !== $shortcode['user_upload_folders']) {
            return false;
        }

        if (!empty($this->_user_folder_name)) {
            return $this->_user_folder_name;
        }

        if (is_user_logged_in()) {
            $current_user = wp_get_current_user();
            $userfoldername = $this->get_user_name_template($current_user);
        } else {
            $userfoldername = $this->get_guest_user_name();
        }

        $this->_user_folder_name = $userfoldername;

        return $userfoldername;
    }

    public function get_auto_linked_folder_for_user()
    {
        // Add folder if needed
        $result = $this->create_user_folder($this->get_auto_linked_folder_name_for_user(), Processor::instance()->get_shortcode(), 5000000);

        if (false === $result) {
            exit;
        }

        return $result->get_path();
    }

    public static function get_manually_linked_folder_for_user()
    {
        $userfolder = get_user_option('out_of_the_box_linkedto');
        if (is_array($userfolder) && isset($userfolder['foldertext'])) {
            if (false === isset($userfolder['accountid'])) {
                $linked_account = Accounts::instance()->get_primary_account();
            } else {
                $linked_account = Accounts::instance()->get_account_by_id($userfolder['accountid']);
            }

            App::set_current_account($linked_account);

            return $userfolder['folderid'];
        }
        $defaultuserfolder = get_site_option('out_of_the_box_guestlinkedto');
        if (is_array($defaultuserfolder) && isset($defaultuserfolder['folderid'])) {
            if (false === isset($defaultuserfolder['accountid'])) {
                $linked_account = Accounts::instance()->get_primary_account();
            } else {
                $linked_account = Accounts::instance()->get_account_by_id($defaultuserfolder['accountid']);
            }

            App::set_current_account($linked_account);

            return $defaultuserfolder['folderid'];
        }

        exit(-1);
    }

    public function manually_link_folder($user_id, $linkedto)
    {
        App::set_current_account_by_id($linkedto['accountid']);
        $node = Client::instance()->get_folder($linkedto['folderid'], false);
        $linkedto['foldertext'] = $node->get_name();

        if ('GUEST' === $user_id) {
            $result = update_site_option('out_of_the_box_guestlinkedto', $linkedto);
        } else {
            $result = update_user_option($user_id, 'out_of_the_box_linkedto', $linkedto, false);
        }

        $linkedto['path'] = $node->get_path_display();
        echo json_encode($linkedto);

        exit;
    }

    public function manually_unlink_folder($user_id)
    {
        if ('GUEST' === $user_id) {
            $result = delete_site_option('out_of_the_box_guestlinkedto');
        } else {
            $result = delete_user_option($user_id, 'out_of_the_box_linkedto', false);
        }

        if (false !== $result) {
            exit('1');
        }
    }

    public function create_user_folder($userfoldername, $shortcode, $mswaitaftercreation = 0)
    {
        if (false !== strpos($shortcode['root'], '%user_folder%')) {
            $userfolder_path = Helpers::clean_folder_path(str_replace('%user_folder%', $userfoldername, $shortcode['root']));
        } else {
            $userfolder_path = Helpers::clean_folder_path($shortcode['root'].'/'.$userfoldername);
        }

        try {
            $api_entry = App::instance()->get_sdk_client()->getMetadata($userfolder_path);

            return new Entry($api_entry);
        } catch (\Exception $ex) {
            // Folder doesn't exists, so continue
        }

        $user_template_path = $shortcode['user_template_dir'];

        try {
            if (empty($user_template_path)) {
                $api_entry_new = App::instance()->get_sdk_client()->createFolder($userfolder_path);
            } else {
                $api_entry_new = App::instance()->get_sdk_client()->copy($user_template_path, $userfolder_path);

                // New Meta data isn't fully available directly after copy command
                usleep($mswaitaftercreation);
            }
        } catch (\Exception $ex) {
            error_log('[WP Cloud Plugin message]: '.sprintf('Failed to add user folder: %s', $ex->getMessage()));

            return false;
        }

        $user_folder = new Entry($api_entry_new);
        do_action('outofthebox_log_event', 'outofthebox_created_entry', $user_folder);

        do_action('outofthebox_dynamic_folder_created', $user_folder, $shortcode);

        return $user_folder;
    }

    public function create_user_folders_for_shortcodes($user_id)
    {
        $new_user = get_user_by('id', $user_id);

        $outoftheboxlists = Shortcodes::instance()->get_all_shortcodes();
        $current_account = App::get_current_account();

        foreach ($outoftheboxlists as $list) {
            if (!isset($list['user_upload_folders']) || 'auto' !== $list['user_upload_folders']) {
                continue;
            }

            if (!isset($list['account']) || $current_account->get_id() !== $list['account']) {
                continue; // Skip shortcodes that don't belong to the account that is being processed
            }

            if (false === Helpers::check_user_role($list['view_role'], $new_user)) {
                continue; // Skip shortcodes that aren't accessible for user
            }

            if (false !== strpos($list['user_upload_folders'], 'disable-create-private-folder-on-registration')) {
                continue; // Skip shortcodes that explicitly have set to skip automatic folder creation
            }

            if (!empty($list['user_folder_name_template'])) {
                $this->_user_name_template = $list['user_folder_name_template'];
            } else {
                $this->_user_name_template = Processor::instance()->get_setting('userfolder_name');
            }

            $new_userfoldersname = $this->get_user_name_template($new_user);

            $this->create_user_folder($new_userfoldersname, $list);
        }
    }

    public function create_user_folders($users = [])
    {
        if (0 === count($users)) {
            return;
        }

        foreach ($users as $user) {
            $userfoldersname = $this->get_user_name_template($user);
            $this->create_user_folder($userfoldersname, Processor::instance()->get_shortcode());
        }
    }

    public function remove_user_folder($user_id)
    {
        $deleted_user = get_user_by('id', $user_id);

        $outoftheboxlists = Shortcodes::instance()->get_all_shortcodes();
        $current_account = App::get_current_account();

        foreach ($outoftheboxlists as $list) {
            if (!isset($list['user_upload_folders']) || 'auto' !== $list['user_upload_folders']) {
                continue;
            }

            if (!isset($list['account']) || $current_account->get_id() !== $list['account']) {
                continue; // Skip shortcodes that don't belong to the account that is being processed
            }

            if (false === Helpers::check_user_role($list['view_role'], $deleted_user)) {
                continue; // Skip shortcodes that aren't accessible for user
            }

            if (!empty($list['user_folder_name_template'])) {
                $this->_user_name_template = $list['user_folder_name_template'];
            } else {
                $this->_user_name_template = Processor::instance()->get_setting('userfolder_name');
            }

            $userfoldername = $this->get_user_name_template($deleted_user);

            if (false !== strpos($list['root'], '%user_folder%')) {
                $userfolder_path = Helpers::clean_folder_path(str_replace('%user_folder%', $userfoldername, $list['root']));
            } else {
                $userfolder_path = Helpers::clean_folder_path($list['root'].'/'.$userfoldername);
            }

            try {
                $api_entry_deleted = App::instance()->get_sdk_client()->delete($userfolder_path);
            } catch (\Exception $ex) {
                return false;
            }
        }

        return true;
    }

    public function update_user_folder($user_id, $old_user)
    {
        $updated_user = get_user_by('id', $user_id);

        $outoftheboxlists = Shortcodes::instance()->get_all_shortcodes();
        $current_account = App::get_current_account();

        foreach ($outoftheboxlists as $list) {
            if (!isset($list['user_upload_folders']) || 'auto' !== $list['user_upload_folders']) {
                continue;
            }

            if (!isset($list['account']) || $current_account->get_id() !== $list['account']) {
                continue; // Skip shortcodes that don't belong to the account that is being processed
            }

            if (false === Helpers::check_user_role($list['view_role'], $updated_user)) {
                continue; // Skip shortcodes that aren't accessible for user
            }

            if (!empty($list['user_folder_name_template'])) {
                $this->_user_name_template = $list['user_folder_name_template'];
            } else {
                $this->_user_name_template = Processor::instance()->get_setting('userfolder_name');
            }

            $new_userfoldersname = $this->get_user_name_template($updated_user);
            $old_userfoldersname = $this->get_user_name_template($old_user);

            if ($new_userfoldersname === $old_userfoldersname) {
                continue;
            }

            if (defined('out_of_the_box_update_user_folder_'.$list['root'].'_'.$new_userfoldersname)) {
                continue;
            }

            define('out_of_the_box_update_user_folder_'.$list['root'].'_'.$new_userfoldersname, true);

            if (false !== strpos($list['root'], '%user_folder%')) {
                $new_userfolder_path = Helpers::clean_folder_path(str_replace('%user_folder%', $new_userfoldersname, $list['root']));
                $old_userfolder_path = Helpers::clean_folder_path(str_replace('%user_folder%', $old_userfoldersname, $list['root']));
            } else {
                $new_userfolder_path = Helpers::clean_folder_path($list['root'].'/'.$new_userfoldersname);
                $old_userfolder_path = Helpers::clean_folder_path($list['root'].'/'.$old_userfoldersname);
            }

            try {
                $api_entry_move = App::instance()->get_sdk_client()->move($old_userfolder_path, $new_userfolder_path);
            } catch (\Exception $ex) {
                return false;
            }
        }

        return true;
    }

    public function get_user_name_template($user_data)
    {
        $user_folder_name = Helpers::apply_placeholders($this->_user_name_template, Processor::instance(), ['user_data' => $user_data]);

        return apply_filters('outofthebox_private_folder_name', $user_folder_name, Processor::instance());
    }

    public function get_guest_user_name()
    {
        $username = $this->get_guest_id();

        $current_user = new \stdClass();
        $current_user->user_login = md5($username);
        $current_user->display_name = $username;
        $current_user->ID = $username;
        $current_user->user_role = esc_html__('Anonymous user', 'wpcloudplugins');

        $user_folder_name = $this->get_user_name_template($current_user);

        $prefix = Processor::instance()->get_setting('userfolder_name_guest_prefix');

        return apply_filters('outofthebox_private_folder_name_guests', $prefix.$user_folder_name, Processor::instance());
    }

    public function get_guest_id()
    {
        $id = uniqid();
        if (!isset($_COOKIE['OftB-ID'])) {
            $expire = time() + 60 * 60 * 24 * 7;
            Helpers::set_cookie('OftB-ID', $id, $expire, COOKIEPATH, COOKIE_DOMAIN, false, false, 'strict');
        } else {
            $id = $_COOKIE['OftB-ID'];
        }

        return $id;
    }
}
