<?php
/**
 * WP Courseware Achievement Controller.
 *
 * @package WPCW
 * @subpackage Controllers
 * @since 4.3.0
 */
namespace WPCW\Controllers;


// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Achievement.
 *
 * @since 4.3.0
 */
class Achievements extends Controller {

	/**
	 * @var string The post type slug.
	 * @since 4.3.0
	 */
	public $post_type_slug = 'wpcw_achievement';

	/**
	 * @var string The category slug.
	 * @since 4.3.0
	 */
	public $taxonomy_category_slug = 'wpcw_achievement_category';

	/**
	 * @var string The tag slug.
	 * @since 4.3.0
	 */
	public $taxonomy_tag_slug = 'wpcw_achievement_tag';

	/**
	 * @var array The unit permalinks.
	 * @since 4.4.0
	 */
	protected $permalinks;

	/**
	 * @var Module The module object.
	 * @since 4.4.0
	 */
	protected $module;

	/**
	 * Units Constructor.
	 *
	 * @since 4.4.0
	 */
	public function __construct() {
	}

	/**
	 * Units Load.
	 *
	 * @since 4.3.0
	 */
	public function load() {
	
			 // Set UI labels for Custom Post Type
			 $labels = array(
				'name'                => _x( 'Achievements', 'Post Type General Name', 'wp-courseware' ),
				'singular_name'       => _x( 'Achievement', 'Post Type Singular Name', 'wp-courseware' ),
				'menu_name'           => __( 'Achievements', 'wp-courseware' ),
				'parent_item_colon'   => __( 'Parent Achievement', 'wp-courseware' ),
				'all_items'           => __( 'All Achievements', 'wp-courseware' ),
				'view_item'           => __( 'View Achievement', 'wp-courseware' ),
				'add_new_item'        => __( 'Add New Achievement', 'wp-courseware' ),
				'add_new'             => __( 'Add New', 'wp-courseware' ),
				'edit_item'           => __( 'Edit Achievement', 'wp-courseware' ),
				'update_item'         => __( 'Update Achievement', 'wp-courseware' ),
				'search_items'        => __( 'Search Achievement', 'wp-courseware' ),
				'not_found'           => __( 'Not Found', 'wp-courseware' ),
				'not_found_in_trash'  => __( 'Not found in Trash', 'wp-courseware' ),
			);
			 
		// Set other options for Custom Post Type
			 
			$args = array(
				'label'               => __( 'Achievements', 'wp-courseware' ),
				'description'         => __( 'Manage achievements', 'wp-courseware' ),
				'labels'              => $labels,
				// Features this CPT supports in Post Editor
				'supports'            => array( 'title', ),
				// You can associate this CPT with a taxonomy or custom taxonomy. 
				// 'taxonomies'          => array( ),
				/* A hierarchical CPT is like Pages and can have
				* Parent and child items. A non-hierarchical CPT
				* is like Posts.
				*/ 
				'hierarchical'        => false,
				'public'              => true,
				'show_ui'             => true,
				'show_in_menu' 		  => false,
				'show_in_nav_menus'   => true,
				'show_in_admin_bar'   => true,
				'menu_position'       => 5,
				'can_export'          => true,
				'has_archive'         => true,
				'exclude_from_search' => false,
				'publicly_queryable'  => true,
				'capability_type'     => 'post',
				'show_in_rest' => true,
				'rewrite' => false,
			);
			 
			// Registering your Custom Post Type
			register_post_type( 'wpcw_achievements', $args );

			//register sub menu
			$this->add_sub_menu();

			//register custom fields
			add_action( 'add_meta_boxes', array( $this, 'add_custom_fields' ), 1 );

			//add javascript
			add_action( 'admin_enqueue_scripts', array( $this, 'achievement_enqueue_scripts' ), 10 );

			//save meta
			add_action('save_post', array( $this, 'achievement_save_meta') );
	}

	public function add_custom_fields(){
		add_meta_box(
			'achievement_desc',
			'<strong>'. __( 'Achievement Text', 'wp-courseware' )  .'</strong> - <small>'. __('This text is displayed when a user get the achievement.', 'wp-courseware').'</small>',
			'wpwcachievements_descrition_editor', 'wpcw_achievements', 'normal', 'high'
		  );

		add_meta_box(
			'achievement_details',
			'<strong>'. __( 'Achievement Details', 'wp-courseware' )  .'</strong> - <small>'. __('Setup the details of the achievement.', 'wp-courseware').'</small>',
			array( $this, 'add_award_points_field' ), 'wpcw_achievements', 'normal', 'high'
		  );

		  add_meta_box(
			'achievement_image',
			'<strong>'. __( 'Achievement Image', 'wp-courseware' )  .'</strong>',
			'wpwcachievements_image_box', 'wpcw_achievements', 'side', 'high', array( 'title' => __("Achievement", "wp-courseware"), 'id' => 'achievement' )
		  );
	}

	public function add_sub_menu(){
		add_action( 'admin_menu', function(){
			add_submenu_page( 'wpcw', __( 'Achievements', 'wp-courseware' ), __( 'Achievements', 'wp-courseware' ), 'manage_options','edit.php?post_type=wpcw_achievements', '', 3);
		  } );
	}

	public function add_award_points_field() {
	  $points = isset($_GET['post']) ? get_post_meta( $_GET['post'], 'achievement_award_points', true )
	  : 0;
	  echo '<span class="pullleft first-select">
      <label for="wpachievements_achievements_data_rank">'.__('Points Awarded', 'wp-courseware').':</label><br/>
      <input type="number" name="wpcw_achievements_award_points" id="wpcw_achievements_award_points" value="'.$points.'" />';
      echo '</span>';
	}

	public function achievement_enqueue_scripts() {
		wp_enqueue_media();
		wp_enqueue_style( 'admin', WPCW_URL. 'assets/css/achievement.css' );
		wp_enqueue_script( 'admin', WPCW_URL.'assets/js/achievement.js' );
	}

	public 	function achievement_save_meta( $post_id ) {

		$nonce = filter_input( INPUT_POST, 'wpachievements_achievement_nonce' );
		if (! $nonce || ! wp_verify_nonce( $nonce, 'wpachievements_achievement_save' ) || ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) ) {
		  return $post_id;
		}

		if ( !current_user_can( 'edit_post', $post_id ))
		  return;

		if ( isset($_POST['upload_image']) ) {  
			$ach_img = $_POST['upload_image'];
			update_post_meta( $post_id, '_achievement_image', $ach_img );
		}

		if ( isset($_POST['wpcw_achievements_award_points']) ) {        
		  update_post_meta($post_id, 'achievement_award_points',  sanitize_text_field($_POST['wpcw_achievements_award_points']));      
		}

		if ( isset($_POST['achievement_desc_editor']) ) {  
			$ach_desc = $_POST['achievement_desc_editor'];
	
			remove_action('save_post', array( $this, 'achievement_save_meta' ) );

			$wpa_args = array(
			'ID'           => $post_id,
			'post_content' => $ach_desc,
			'post_status'  => 'publish'
			);

    		wp_update_post( $wpa_args );

    		add_action('save_post', array( $this, 'achievement_save_meta' ) );
		}
	}

	/**
	 * Leaderboard Page.
	 *
	 * @since 4.8.0
	 */
	public function leaderboard_page() {
		wpcw_print_notices();
		global $wpdb, $wpcwdb;
		$result = $wpdb->get_results( 
			"SELECT u.id, u.display_name, SUM(ua.points) as total_points 
			FROM $wpcwdb->user_achievements ua 
			INNER JOIN $wpdb->users u 
			ON ua.user_id = u.ID 
			GROUP BY u.ID 
			ORDER BY total_points DESC"
		);
		wp_enqueue_style( 'admin', WPCW_URL. 'assets/css/achievement.css' );
		wpcw_get_template( 'account/account-leaderboard.php', array( 'leaderboard' => $result ) );
	}

}
