<?php
/**
 * WP Courseware Course Notes Controller.
 *
 * @since 4.9.4
 * @subpackage Controllers
 * @package WPCW
 */
namespace WPCW\Controllers;

use WPCW\Controllers\Controller;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class Course Notes.
 *
 * @since 4.9.4
 */
class Course_Notes extends Controller {

	/**
	 * @var string The course post type slug.
	 * @since 4.4.0
	 */
	public $post_type_slug = 'wpcw_course_note';

	function __construct() {
		// code...
	}

	/**
	 * Load Course Notes Controller.
	 *
	 * @since 4.9.4
	 */
	public function load() {
		add_filter( 'wpcw_ajax_api_events', array( $this, 'notes_ajax_events' ) );
	}

	/**
	 * Course Note Ajax Events.
	 *
	 * @since 4.9.4
	 *
	 * @param array $ajax_events The ajax Events.
	 */
	public function notes_ajax_events( $ajax_events ) {
		$checkout_ajax_events = array(
			'create_wpcw_course_note' => array( $this, 'ajax_create_notes' ),
			'wpcw_delete_course_note' => array( $this, 'ajax_wpcw_delete_note' ),
			'wpcw_course_note_filter' => array( $this, 'ajax_wpcw_course_note_filter' ),
		);
		return array_merge( $ajax_events, $checkout_ajax_events );
	}

	/**
	 * Create/Update Course Note.
	 *
	 * @since 4.9.4
	 *
	 * @param None.
	 *
	 * @return json.
	 */
	public function ajax_create_notes() {
		if ( ! wpcw()->ajax->verify_nonce( true ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'An error occurred during verify nonce. Please try again.' ) ) );
		}
		$editor_content = stripslashes( $_POST['editor_content'] );
		if ( ! $editor_content ) {
			wp_send_json_error( array( 'message' => esc_html__( 'An error occurred. Please try again.' ) ) );
		}
		$unit_id   = $_POST['unit_id'];
		$note_id   = isset( $_POST['note_id'] ) ? $_POST['note_id'] : '';
		$note_data = array(
			'post_author'  => get_current_user_id(),
			'post_content' => $editor_content,
			'post_title'   => 'Student Note',
			'post_status'  => 'publish',
			'post_type'    => $this->post_type_slug,
			'post_parent'  => $unit_id,
		);
		if ( $note_id ) {
			$note_data['ID']       = $note_id;
			$course_note_id        = wp_update_post( $note_data );
			$res_arr['message']    = esc_html__( 'Note updated successfully.', 'wp-courseware' );
			$res_arr['textAreaId'] = 'new_wpcw_note_content_' . $course_note_id;
		} else {
			$course_note_id        = wp_insert_post( $note_data );
			$res_arr['message']    = esc_html__( 'Note created successfully.', 'wp-courseware' );
			$res_arr['textAreaId'] = 'new_wpcw_note_content_' . $course_note_id;

		}
		ob_start();
		$editor = get_editor_code( $course_note_id, $editor_content );
		ob_get_clean();
		$res_arr['html'] = sprintf( '<div class="wpcw-content-div">
							<div class="wpcw-content-img">
								<span>
									%s %s
								</span>
								<i class="wpcw-edit-note wpcw-fa wpcw-fa-edit" data-note-id=%d></i>
								<i class="wpcw-delete-note wpcw-fa wpcw-fa-trash" data-note-id=%d></i>
							</div>
							<div class="note-content">%s
							</div>
							<div class="edit-note-editor">
								%s
							</div>
						</div>', __( 'Posted on', 'wp-courseware'),
						 date( 'd/m/Y' ),
						 $course_note_id,
						 $course_note_id,
						 $editor_content,
						 $editor );

		if ( $course_note_id ) {
			wp_send_json_success(
				$res_arr
			);

		} else {
			wp_send_json_error( array( 'message' => esc_html__( 'An error occurred. Please try again.' ) ) );
		}
	}

	/**
	 * Delete Course Note.
	 *
	 * @since 4.9.4
	 *
	 * @param None.
	 *
	 * @return json.
	 */
	public function ajax_wpcw_delete_note() {
		$note_id = $_POST['note_id'];
		if ( ! $note_id ) {
			wp_send_json_error( array( 'message' => esc_html__( 'An error occurred. Please try again.', 'wp-courseware' ) ) );
		}
		if ( ! wpcw()->ajax->verify_nonce( true ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'An error occurred during verify nonce. Please try again.', 'wp-courseware' ) ) );
		}
		wp_delete_post( $note_id );
		wp_send_json_success(
			array(
				'message' => esc_html__( 'Note deleted successfully.', 'wp-courseware' ),
			)
		);
	}

	/**
	 * Filter Course Note.
	 *
	 * @since 4.9.4
	 *
	 * @param None.
	 *
	 * @return json.
	 */
	public function ajax_wpcw_course_note_filter() {
		$select_desc_asc = $_POST['select_desc_asc'];
		$select_unit     = $_POST['select_unit'];
		$unit_id         = $_POST['unit_id'];
		$course_notes    = array();
		if ( $unit_id ) {
			$parentData = get_wpcw_units_associated_course( $unit_id );
			$course_id  = $parentData->parent_course_id;
		}

		if ( $select_unit == 'all_unit' ) {
			$course_notes = get_wpcw_course_notes( '', get_current_user_id(), $select_desc_asc, $course_id );
		} elseif ( $select_unit == 'single_unit' ) {
			$course_notes = get_wpcw_course_notes( $unit_id, get_current_user_id(), $select_desc_asc );
		}
		ob_start();
		$course_html = '';
		if ( $course_notes ) {
			$text_area_ids = array();
			foreach ( $course_notes as $key => $note_post ) {
				$note_id         = $note_post->ID;
				$post_date       = $note_post->post_date;
				$post_content    = $note_post->post_content;
				$editor          = get_editor_code( $note_id, $post_content );
				$text_area_ids[] = 'new_wpcw_note_content_' . $note_id;
				$course_html    .= sprintf( '<div class="wpcw-content-div">
							<div class="wpcw-content-img">
								<span>
									%s %s
								</span>
								<i class="wpcw-edit-note wpcw-fa wpcw-fa-edit" data-note-id=%d></i>
								<i class="wpcw-delete-note wpcw-fa wpcw-fa-trash" data-note-id=%d></i>
							</div>
							<div class="note-content">%s
							</div>
							<div class="edit-note-editor">
								%s
							</div>
						</div>', __( 'Posted on', 'wp-courseware'),
						 date( 'd/m/Y', strtotime( $post_date ) ),
						 $note_id,
						 $note_id,
						 $post_content,
						 $editor );
			}
		}
		ob_get_clean();
		wp_send_json_success(
			array(
				'html'        => $course_html,
				'textAreaIds' => $text_area_ids,
			)
		);
	}
}
