<?php
/**
 * WP Courseware License.
 *
 * @packcage WPCW
 * @subpackage Core
 * @since 4.9.13
 * 
 */
namespace WPCW\Core;

use WP_Error;
use WP_REST_Request;
use WP_REST_Response;
use WPCW\Admin\Pages;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

/**
 * Class License.
 *
 * @since 4.9.13
 * 
 */
final class translation {

	/**
	 * Options Key for store translations data.
	 *
	 * @var string $translation_key.
	 */
	private static $translation_key = 'wpcw-translations';


	/**
	 * Options Key for store translations data.
	 *
	 * @var string $translation_track_key.
	 */
	private static $translation_track_key = 'wpcw-translations-track';


	/**
	 * Options Key for store translations data.
	 *
	 * @var string $project_slugs.
	 */
	private static $project_slugs = 'wp-courseware';

	/**
	 * Options Key for languageDir.
	 *
	 * @var string $languageDir.
	 */
	private static $languageDir = "";


	/**
	 * translation class Constructor.
	 *
	 */
	function __construct() 
	{
		self::$languageDir = WPCW_LANG_PATH;
		add_action("admin_init", array($this, "handle_translation_actions") );
		add_action("admin_init", array($this, "load_translations_initially") );

	}
	
	/**
	 * POT file URL
	*/
	public static function pot_file_path(){
		return self::$languageDir . '' . self::$project_slugs . '.pot';
	}


	/**
	 * Create Option and update ontion values initial pluin load 
	*/
	public static function load_translations_initially(){
		$wpcw_translations = get_option( self::$translation_key );
		if(!$wpcw_translations){
			self::refresh_available_translations();
		}
	}

	/**
	 * Get last updated translations.
	 */
	public static function translation_action_url( $action = '', $action_url = '',  $locale = '') 
	{
		if ( ! empty( $action ) && $action_url) {
			$action_url = esc_attr( $action_url );

			$nonce_key  = 'wpcw-translation-' . $action;
			$action_url = add_query_arg( array( 'translation_action' => $action ), $action_url );

			if ( ! empty( $project ) ) {
				$nonce_key .= '-' . $project;
				$action_url = add_query_arg( array( 'project' => $project ), $action_url );
			}

			if ( ! empty( $locale ) ) {
				$nonce_key .= '-' . $locale;
				$action_url = add_query_arg( array( 'locale' => $locale ), $action_url );
			}
			
			$action_nonce = wp_create_nonce( $nonce_key );
			$action_url   = add_query_arg( array( 'wpcw-translation-nonce' => $action_nonce ), $action_url );

			return $action_url;
		}
		return '';
	}

	/**
	 * Get last updated translations.
 	*/
	public static function get_translations() {
		$wpcw_translations = get_option( self::$translation_key );
		if ( !empty( $wpcw_translations && isset( $wpcw_translations['translation_sets']) ) ) {
			return $wpcw_translations['translation_sets'];
		}
	}

	/**
	 * Check if language directory is writable for Project Slug.
	 *
	 * @param string $project_slug Project Slug.
	 * @return boolean true if directory is writable.
	 */
	public static function is_language_directory_writable( $project_slug = '' ) {
		if ( ! empty( $project_slug ) ) 
		{
			$translations_dir = self::$languageDir;
			if ( ( ! empty( $translations_dir ) ) && ( is_writable( $translations_dir ) ) ) {
				return true;
			}
		}
		return false;
	}


	/**
	 * Refresh Translations
 	*/
	public static function refresh_translations() {
		$wpcw_translations               = get_option( self::$translation_key );
		$wpcw_translations['last_check'] = 0;
		update_option( self::$translation_key, $wpcw_translations );
		self::refresh_available_translations();
	}


	/**
	 * Download .pot file
	 *
	 * @param string $project_slug Translation project slug.
	 */
	public static function download_pot_file(  ) {
		
		$pot_filename = self::pot_file_path();

		if ( file_exists( $pot_filename ) ) 
		{
			$pot_file_contents = file_get_contents( $pot_filename );
			header( 'Content-type: text/x-gettext-translation' );
			header( 'Content-Length: ' . strlen( $pot_file_contents ) );
			header( 'Cache-Control: no-cache, must-revalidate' );
			header( 'Pragma: no-cache' );
			header( 'Expires: 0' );
			header( 'Content-Disposition: attachment; filename="' . basename( $pot_filename ) . '"; modification-date="' . date( 'r' ) . '";' ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
			echo $pot_file_contents; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			die();
		}
	}

	/**
	 * Download .pot file
	 *
	 * @param string $project_slug Translation project slug.
	*/
	public static function download_po_file( $project = '', $locale = ''   ) {
		
		if ( ( ! empty( $project ) ) && ( ! empty( $locale ) ) ) 
		{
			$installed_translations = self::get_installed_translations( $project );

			if ( isset( $installed_translations[ $locale ] ) )
			{
				$installed_set = $installed_translations[ $locale ];
				$po_filename   = self::$languageDir . $installed_set['po'];
				if ( file_exists( $po_filename ) ) {
					$po_file_contents = file_get_contents( $po_filename ); 

					header( 'Content-type: text/x-gettext-translation' );
					header( 'Content-Length: ' . strlen( $po_file_contents ) );
					header( 'Cache-Control: no-cache, must-revalidate' );
					header( 'Pragma: no-cache' );
					header( 'Expires: 0' );
					header( 'Content-Disposition: attachment; filename="' . basename( $po_filename ) . '"; modification-date="' . date( 'r' ) . '";' ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
					echo $po_file_contents; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Need to output HTML.
					die();
				}
			}
		}
	}

	/**
	 * Get Settings Fields.
	 *
	 * @since 4.9.13
	 * 	 *
	 * @return array The license settings fields.
	 */
	public function get_settings_fields() {
		return apply_filters( 'wpcw_license_settings_fields', array(
			array(
				'type'  => 'translation_table',
				'key'   => 'translation_table',
				'title' => esc_html__( 'Translations', 'wp-courseware' ),
				'desc'  => __( 'All translations are stored into the directory: <code><site root>/wp-content/plugins/wp-courseware/languages/</code>', 'wp-courseware' ),
			),
			
		) );
	}


	/** Return available translations for project slug.
	 *
	 * @param string $project_slug Project .
	 * @param string $locale Locale for translations.
	 * @return mixed Translation set array.
	 */
	public static function project_get_available_translations( $project = '', $locale = '' ) {

		$ld_translations = get_option( self::$translation_key );
		
		if ( isset( $ld_translations['translation_sets'] ) ) {
			if ( ! empty( $locale ) ) {
				foreach ( $ld_translations['translation_sets'] as $translation_set ) {
					if ( $translation_set['wp_locale'] == $locale ) {
						return $translation_set;
					}
				}
			}
			return $ld_translations['translation_sets'];
		}
	}

	/**
	 * Get last update from option key.
	 *
	 * @since 4.9.13
	 * 	 *
	 * @return timestamp of update date.
	 */
	public static function get_last_update() {
		$wpcw_translations = get_option( self::$translation_key );
		if ( isset( $wpcw_translations['last_check'] ) ) {
			return $wpcw_translations['last_check'];
		}
	}

	public static function refresh_available_translations() 
	{
		$url = add_query_arg(
				array(
					'wpcw-glotpress' => 1,
					'action'         => 'translation_sets',
					'project'        => self::$project_slugs,
					'version'        => time(),
				),
				trailingslashit( WPCW_TRANSLATIONS_BASE_URL )
			);

		$url_args = array( 'timeout' => 10 );
	
		$response = wp_remote_get( $url, $url_args );

		if ( ( is_array( $response ) ) && ( wp_remote_retrieve_response_code( $response ) == '200' ) ) 
		{
			$response_body = wp_remote_retrieve_body( $response );

			if ( ! empty( $response_body ) ) {
				$wpcw_translation_sets = json_decode( $response_body, true );

				$wpcw_translation = array(
					'last_check'       => time(),
					'translation_sets' => $wpcw_translation_sets,
				);
				update_option( self::$translation_key, $wpcw_translation );
			}
			return json_decode($response_body);
		}
	}


	/**
	 * Get installed translations.
	 *
	 * @param string $project_slug Translation project slug.
	 */
	public static function get_installed_translations( $project_slug = '' ) {
		
		$translation_files = array();

		if ( ! empty( $project_slug ) ) {

			$languages_plugins_dir    = self::$languageDir;
			$languages_plugins_dir_mo = $languages_plugins_dir . $project_slug . '-*.mo';

			$mo_files = glob( $languages_plugins_dir_mo );

			if ( ! empty( $mo_files ) ) {
				foreach ( $mo_files as $mo_file ) {
					$mo_file       = basename( $mo_file );
					$mo_file_local = str_replace( array( $project_slug . '-', '.mo' ), '', $mo_file );
					if ( ! empty( $mo_file_local ) ) {

						if ( ! isset( $translation_files[ $mo_file_local ] ) ) {
							$translation_files[ $mo_file_local ]             = array();
							$translation_files[ $mo_file_local ]['mo']       = $mo_file;
							$translation_files[ $mo_file_local ]['mo_mtime'] = filemtime( $languages_plugins_dir . $mo_file );

							$po_file                  = str_replace( '.mo', '.po', $mo_file );
							$languages_plugins_dir_po = $languages_plugins_dir . $po_file;
							if ( file_exists( $languages_plugins_dir_po ) ) {
								$translation_files[ $mo_file_local ]['po']       = $po_file;
								$translation_files[ $mo_file_local ]['po_mtime'] = filemtime( $languages_plugins_dir . $po_file );
							}
						}
					}
				}
			}
		}
		
		return $translation_files;
	}


	/**
	 * Install Translations
	 *
	 * @param string $project Project Slug.
	 * @param string $locale Translation Locale.
	 */
	public static function install_translation( $project = '', $locale = '' ) {
		$reply_data = array();
		
		if ( ( ! empty( $project ) ) && ( ! empty( $locale ) ) ) {

			if ( self::is_language_directory_writable( $project ) ) 
			{
				$translation_set = self::project_get_available_translations( $project, $locale );
				
				if ( ( isset( $translation_set['links'] ) ) && ( ! empty( $translation_set['links'] ) ) ) {
					foreach ( $translation_set['links'] as $link_key => $link_url ) {
						
						$url_args = array( 'timeout' => "10" ) ;

						$dest_filename = self::$languageDir . $project . '-' . $locale . '.' . $link_key;

						if ( file_exists( $dest_filename ) ) {
							unlink( $dest_filename );
						}
						
						$response = wp_remote_get( $link_url, $url_args );

						if ( ( is_array( $response ) ) && ( wp_remote_retrieve_response_code( $response ) == '200' ) ) {
							$response_body = wp_remote_retrieve_body( $response );
							if ( ! empty( $response_body ) ) {
								$fp = fopen( $dest_filename, 'w+' ); //phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen
								if ( false !== $fp ) {
									fwrite( $fp, $response_body ); //phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fwrite
									fclose( $fp ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fclose
									$reply_data['status']  = true;
									$reply_data['message'] = '<p>' . sprintf(
										// translators: placeholders: Language Name, Language code.
										esc_html_x( 'Translation installed: %1$s (%2$s)', 'placeholders: Language Name, Language code', 'wp-courseware' ),
										$translation_set['english_name'],
										$translation_set['wp_locale']
									) . '</p>';
									$reply_data['translation_set'] = $translation_set;

									// Translation track
									$trsTrack = get_option(self::$translation_track_key);	
									$trsLocale['last_update'] = time();
									$trsLocale['locale'][$locale] = $locale;


									if( !empty($trsTrack) && isset($trsTrack['locale'] ) )
									{
										$trsLocale['locale'] = array_merge( $trsLocale['locale'], $trsTrack['locale'] );
										$trsLocale['locale'] = array_unique($trsLocale['locale']);
									}
									update_option(self::$translation_track_key, $trsLocale);	
								}
							}
						}
					}
				}
			} else {
				$reply_data["error"] = __("Langauge directory is not Writable.","wp-courseware");
			}
		}
		return $reply_data;
	}


	/**
	 * Remove translations for Project.
	 *
	 * @param string $project Project Slug.
	 * @param string $locale Locale for translation.
	 */
	public static function delete_translation( $project = '', $locale = '' ) {
		$reply_data = array();
		
		if ( ( ! empty( $project ) ) && ( ! empty( $locale ) ) ) {

			if ( self::is_language_directory_writable( $project ) ) {
				$translation_set = self::project_get_available_translations( $project, $locale );

				if ( ( isset( $translation_set['links'] ) ) && ( ! empty( $translation_set['links'] ) ) ) {
					foreach ( $translation_set['links'] as $link_key => $link_url ) {
						/** This filter is documented in includes/class-ld-translations.php */
						$url_args = apply_filters( 'learndash_translations_url_args', array() );

						$dest_filename = self::$languageDir . $project . '-' . $locale . '.' . $link_key;
						if ( file_exists( $dest_filename ) ) {
							unlink( $dest_filename );
							$reply_data['status']  = true;
							$reply_data['message'] = '<p>' . sprintf(
								// translators: placeholders: Language Name, Language code.
								esc_html_x( 'Translation removed: %1$s (%2$s)', 'placeholders: Language Name, Language code', 'learndash' ),
								$translation_set['english_name'],
								$translation_set['wp_locale']
							) . '</p>';
						}
					}
				}

				$trsTrack = get_option(self::$translation_track_key);
				if( !empty ($trsTrack ) && is_array( $trsTrack ) ) {
					if( isset($trsTrack['locale'][$locale]) ) {

						$trsTrack['last_update'] = time();
						unset( $trsTrack['locale'][$locale] );
						update_option(self::$translation_track_key, $trsTrack);
					}
				}	
			}
		}

		return $reply_data;
	}

	/**
	 * Update Translations for Project
	 *
	 * @param string $project Project Slug.
	 * @param string $locale Locale.
	 */
	public static function update_translation( $project = '', $locale = '' ) 
	{
		$response = array();
		if ( ( ! empty( $project ) ) && ( ! empty( $locale ) ) ) {

			if ( self::is_language_directory_writable( $project ) ) 
			{
				$translation_set = self::project_get_available_translations( $project, $locale );


				if ( ( isset( $translation_set['links'] ) ) && ( ! empty( $translation_set['links'] ) ) ) {
					foreach ( $translation_set['links'] as $link_key => $link_url ) 
					{
						$dest_filename = self::$languageDir . $project . '-' . $locale . '.' . $link_key;
						if ( file_exists( $dest_filename ) ) {
							unlink( $dest_filename );
						}

						$response = wp_remote_get( $link_url);

						if ( ( is_array( $response ) ) && ( wp_remote_retrieve_response_code( $response ) == '200' ) ) 
						{
							$response_body = wp_remote_retrieve_body( $response );
							if ( ! empty( $response_body ) ) {

								$fp = fopen( $dest_filename, 'w+' ); //phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_read_fopen
								if ( false !== $fp ) {
									fwrite( $fp, $response_body );
									fclose( $fp ); 
									$response['status']  = true;
									$response['message'] = '<p>' . sprintf(
										
										esc_html_x( 'Translation updated: %1$s (%2$s)', 'placeholders: Language Name, Language code', 'learndash' ),
										$translation_set['english_name'],
										$translation_set['wp_locale']
									) . '</p>';
								}
							}
						}
					}
				}
			}
		}
		return $response;
	}


	/**
	 * Get last update information.
	 */
	public static function handle_translation_actions() 
	{
		if ( isset( $_GET['translation_action'] ) ) 
		{ 

			$project = self::$project_slugs;

			$action = isset( $_GET['translation_action'] ) ? esc_attr( $_GET['translation_action'] ) : "" ;	
			$nonce =  isset( $_GET['wpcw-translation-nonce'] ) ? esc_attr( $_GET['wpcw-translation-nonce'] ) : "" ;
			$locale =  isset( $_GET['locale'] ) ? esc_attr( $_GET['locale'] ) : "" ;
			
			if ( ! empty( $action ) ) {
				switch ( $action ) {

					case 'refresh':
						if ( ( ! empty( $project ) ) && ( ! empty( $nonce ) ) ) {
							if ( wp_verify_nonce( $nonce, 'wpcw-translation-' . $action ) ) {
							 	$response = self::refresh_translations();
							}
						}
						break;

					case 'pot_download':
						if ( ( ! empty( $project ) ) && ( ! empty( $nonce ) ) ) {
							if ( wp_verify_nonce( $nonce, 'wpcw-translation-' . $action ) ) {
								$response = self::download_pot_file();
							}
						}
						break;
						
					case 'po_download':
						if ( ( ! empty( $project ) ) && ( ! empty( $locale ) ) && ( ! empty( $nonce ) ) ) {
							if ( wp_verify_nonce( $nonce, 'wpcw-translation-' . $action . '-' . $locale ) ) {
								$response = self::download_po_file( $project, $locale );
							}
						}
						break;

					case 'install':
						
						if ( ( ! empty( $locale ) ) && ( ! empty( $nonce ) ) ) {
							if ( wp_verify_nonce( $nonce, 'wpcw-translation-' . $action . '-' . $locale ) ) {
								$response = self::install_translation($project , $locale);
							}
						}
						break;

					case 'update':
						if ( ( ! empty( $project ) ) && ( ! empty( $locale ) ) && ( ! empty( $nonce ) ) ) {
							if ( wp_verify_nonce( $nonce, 'wpcw-translation-' . $action . '-' . $locale ) ) {
								$response = self::update_translation($project , $locale);
							}
						}
						break;

					case 'delete':
						if ( ( ! empty( $project ) ) && ( ! empty( $locale ) ) && ( ! empty( $nonce ) ) ) {
							if ( wp_verify_nonce( $nonce, 'wpcw-translation-' . $action . '-' . $locale ) ) {
								$response = self::delete_translation($project , $locale);
							}
						}
						break;

					default:
						break;
				}


				if( !empty($response) && !isset( $response['error'] ) ) {
					$redirect_url = remove_query_arg( array( 'translation_action', 'wpcw-translation-nonce', 'locale') );
					if ( ! empty( $redirect_url ) ) {
						wp_safe_redirect( $redirect_url );
					}
					
				} else {
					if( isset( $response['error']) ){
						wpcw_display_admin_notice( $response['error'] , 'error' );
					}
				}
			}
		}
	}
}