<?php
/**
 * WP Courseware Quiz Upload Question Type.
 *
 * @package WPCW
 * @since 1.0.0
 */

if ( ! class_exists( 'WPCW_quiz_Sorting' ) ) {
	/**
	 * Class WPCW_quiz_Sorting.
	 *
	 * The class that represents a question where you can upload a file.
	 *
	 * @since 1.0.0
	 */
	class WPCW_quiz_Sorting extends WPCW_quiz_base {

		/**
		 * Default constructor
		 *
		 * @param Object $quizItem The quiz item details.
		 */
		public function __construct( $quizItem ) {
			parent::__construct( $quizItem );
			$this->questionType = 'sorting';
			$this->cssClasses   = 'wpcw_question_type_sorting';

			$this->hint = __( '(Optional) Use this to guide the user what they should upload.', 'wp-courseware' );

			if ( $this->disabled ) {
				$this->cssClasses .= ' wpcw_question_disabled';
			}
		}

		/**
		 * Output the form that allows questions to be configured.
		 */
		public function editForm_toString() {

			$answerList = false;
			if ( $this->quizItem->question_data_answers ) {
				$answerList = WPCW_quizzes_decodeAnswers( $this->quizItem->question_data_answers );
			}

			$html = false;

			// Extra CSS for errors
			$errorClass_Question  = false;
			$errorClass_CorAnswer = false;

			// Error Check - Have we got an issue with a lack of question?
			if ( $this->showErrors ) {
				if ( ! $this->quizItem->question_question ) {
					$errorClass_Question = 'wpcw_quiz_missing';
					$this->gotError      = true;
				}
			}

			// Track columns needed to show question details
			$columnCount = 4;

			// Render just the question area
			$html .= sprintf( '<li id="wpcw_quiz_details_%s" class="%s"><table class="wpcw_quiz_details_questions_wrap" cellspacing="0">', $this->quizItem->question_id, $this->cssClasses );

			// Details of the question - top of the question details.
			$html .= $this->getSection_processHeader( $columnCount );

			// Check for being disabled.
			if ( $this->disabled ) {
				$html .= $this->getSection_disabledQuestionNotice( $columnCount );
			}

			// Main question details here...
			$html .= sprintf( '<tr class="wpcw_quiz_row_question %s">', $errorClass_Question );

			$html .= sprintf( '<th>%s</th>', __( 'Question', 'wp-courseware' ) );

			$html .= sprintf( '<td>' );
			$html .= sprintf( '<textarea name="question_question_%s">%s</textarea>', $this->quizItem->question_id, htmlspecialchars( $this->quizItem->question_question ) );
			$html .= sprintf( '<input type="hidden" name="question_type_%s" value="sorting" />', $this->quizItem->question_id );

			// Field storing order of question among other questions
			$html .= sprintf(
				'<input type="hidden" name="question_order_%s" value="%s" class="wpcw_question_hidden_order" />', $this->quizItem->question_id, absint( $this->quizItem->question_order )
			);

			$html .= sprintf( '</td>' );

			// Only show column if need correct answers.
			// $html .= sprintf( '<td class="wpcw_quiz_details_tick_correct wpcw_quiz_only_td">%s</td>', __( 'Correct<br/>Answer?', 'wp-courseware' ) );

			// Column for add/remove buttons
			$html .= '<td>&nbsp;</td>';

			$html .= sprintf( '</tr>' );

			// Render the section that allows an image to be shown.
			$html .= $this->getSection_showImageField( $columnCount );

			// Render the field that allows answers to be randomized
			// $html .= $this->getSection_showRandomizeAnswersField( $columnCount );

			// Render the field that allows answers to always be displayed as checkboxes.
			// $html .= $this->getSection_showAnswersAsCheckboxesField( $columnCount );

			// Render the list of answers if we have any.

			if ( $answerList ) {
				$count = 0;
				$odd   = true;
				foreach ( $answerList as $answerItem ) {
					// Extract image if available
					$answerItemImageVal = WPCW_arrays_getValue( $answerItem, 'image' );

					// Exract the answer if available
					$answerItemVal = trim( $answerItem['answer'] );
					++ $count;
					/*
					 * Store the selected result.
					 * @var string
					 */
					$result          = $this->quizItem->question_correct_answer;
					$correct_answers = @unserialize( $result );
					/**
					 * check is the serialised array or not if not set $correct_answers to result value for the single answer quiz.
					 */
					if ( ! $correct_answers ) {
						$correct_answers = array( $result );
					}

					// Show an error if the field is still blank.
					$errorClass_Answer = false;
					if ( $this->showErrors ) {
						// Check that answer contains some characters.
						if ( strlen( $answerItemVal ) == 0 ) {
							$errorClass_Answer = 'wpcw_quiz_missing';
							$this->gotError    = true;
						}
					}

					// Add 'data-answer-id' field to store the ID of this row, and other rows that match this.
					$html .= sprintf( '<tr class="wpcw_quiz_row_answer %s %s" data-answer-id="%d">', $errorClass_Answer, ( $odd ? 'alternate' : '' ), $count );
					$html .= sprintf( '<th>%s <span>%d</span></th>', __( 'Answer', 'wp-courseware' ), $count );
					$html .= sprintf( '<td><input type="text" name="question_answer_%s[%d]" value="%s" /></td>', $this->quizItem->question_id, $count, htmlspecialchars( $answerItemVal ) );

					// Correct answer column
					// $html .= sprintf(
								// 	'<td class="wpcw_quiz_details_tick_correct wpcw_quiz_only_td">
			     //                        <input type="checkbox" name="question_answer_sel_%s[]" id="question_answer_sel_%s_%s" value="%s" %s />
			     //                    </td>',
					// 	$this->quizItem->question_id,
					// 	$this->quizItem->question_id,
					// 	$count,
					// 	$count,
					// 	( in_array( $count, $correct_answers ) ? 'checked="checked"' : false
					// 	) );

					// Buttons for add/remove questions
					$html .= sprintf(
						'
					<td class="wpcw_quiz_add_rem">
						<a href="#" title="%s" class="wpcw_question_add"><img src="%sicon_add_32.png" /></a>
						<a href="#" title="%s" class="wpcw_question_remove"><img src="%sicon_remove_32.png" /></a>
					</td>', __( 'Add a new answer...', 'wp-courseware' ), WPCW_IMG_URL, __( 'Remove this answer...', 'wp-courseware' ), WPCW_IMG_URL
					);

					$html .= sprintf( '</tr>' );

					// Add the image URL for this answer - added as a new row.
					$html .= sprintf( '<tr class="wpcw_quiz_row_answer_image wpcw_quiz_row_answer_image_%d %s %s">', $count, $errorClass_Answer, ( $odd ? 'alternate' : '' ) );
					$html .= sprintf(
						'<th>%s <span class="wpcw_inner_hint">%s</span></th>', __( 'Answer Image URL', 'wp-courseware' ), __( '(Optional) ', 'wp-courseware' )
					);

					$html .= '<td>';
					// Field name - needs to use underscore, as square brackets break the jQuery to find the target.
					$thisAnswerFieldName = 'question_answer_image_' . $this->quizItem->question_id . '_' . $count;

					// The URL field.
					$html .= sprintf(
						'<input type="text" name="question_answer_image_%s[%d]" id="%s" value="%s" />', $this->quizItem->question_id, $count, $thisAnswerFieldName, $answerItemImageVal
					);

					// The insert button.
					$html .= sprintf(
						'<span class="wpcw_insert_image_wrap"><a href="#" class="button wpcw_insert_image" data-uploader_title="%s" data-uploader_btn_text="%s" data-target="%s" title="%s"><span class="wpcw_insert_image_img"></span> %s</a></span>', __( 'Choose an image for this answer...', 'wp-courseware' ), __( 'Select Image...', 'wp-courseware' ), $thisAnswerFieldName, __( 'Select Image', 'wp-courseware' ), __( 'Select Image', 'wp-courseware' )
					);

					$html .= '</td>';

					// Filler for the remaining space
					$html .= '<td colspan="2"></td>';

					$html .= sprintf( '</tr>' );

					$odd = ! $odd;
				}
			}

			// Extra fields at the bottom of a question.
			$html .= $this->getSection_processFooter( $columnCount );

			// All done
			$html .= sprintf( '</table></li>' );

			return $html;
		}


		/**
		 * Shows the field where the instructor can determine if answers are randomly presented
		 * to the user on the page.
		 *
		 * @param Integer $columnCount The number of columns that are being rendered to sh
		 *
		 * @return String The HTML for rendering the randomize answers field.
		 */
		protected function getSection_showRandomizeAnswersField( $columnCount ) {
			$html = '<tr>';
			$html .= sprintf(
				'<th>%s<span class="wpcw_inner_hint">%s</span></th>',
				__( 'Randomize Answers?', 'wp-courseware' ),
				__( '(Optional)', 'wp-courseware' )
			);

			$html .= '<td class="wpcw_quiz_details_randomize_answers">';

			// The checkbox to enable the feature
			$html .= sprintf(
				'<input name="question_multi_random_enable_%s" class="wpcw_quiz_details_enable" type="checkbox" %s />',
				$this->quizItem->question_id,
				( $this->quizItem->question_multi_random_enable > 0 ? 'checked="checked"' : '' ),
				__( 'Yes, randomize the order of these answers.', 'wp-courseware' )
			);

			// The count of the items that will be randomized. Always include, but hide if not enabled.
			$html .= sprintf(
				'<span class="wpcw_quiz_details_count_wrap" %s>
					 <label>%s</label>
					 <input name="question_multi_random_count_%s" class="wpcw_quiz_details_count" type="text" value="%s" size="10" maxlength="10" />
					 <span class="wpcw_quiz_details_count_doc">%s</span>
				 </span>',
				( $this->quizItem->question_multi_random_enable ? '' : 'style="display: none;"' ),
				__( 'Number of answers to display:', 'wp-courseware' ),
				$this->quizItem->question_id,
				$this->quizItem->question_multi_random_count,
				__( 'The correct answers will always appear in the selection of answers.', 'wp-courseware' )
			);

			$html .= '</td>';

			// Works out the space after the text area.
			$columnCount -= 2;
			if ( $columnCount > 0 ) {
				$html .= sprintf( '<td colspan="%d">&nbsp;</td>', $columnCount );
			}

			$html .= '</tr>';

			return $html;
		}


		/**
		 * Shows the field where the instructor can determine if answers are displayed
		 * as checkboxes or radio buttons. Always displays, but is hidden when they select
		 * more than one answer.
		 *
		 * @param Integer    $columnCount The number of columns that are being rendered to sh
		 * @param bool|array $answerList The answer list.
		 *
		 * @return String The HTML for rendering the checkbox or radio field.
		 */
		protected function getSection_showAnswersAsCheckboxesField( $columnCount ) {
			$correct_answers = maybe_unserialize( $this->quizItem->question_correct_answer );

			$hidden = ( ! empty( $correct_answers ) && count( $correct_answers ) > 1 ) ? 'style="display: none;"' : '';

			$html = '<tr class="wpcw_quiz_row_checkboxes_enable"' . $hidden . '>';
			$html .= sprintf(
				'<th>%s<span class="wpcw_inner_hint">%s</span></th>', __( 'Force Checkboxes?', 'wp-courseware' ), __( '(Optional)', 'wp-courseware' )
			);

			$html .= '<td class="wpcw_quiz_details_checkboxes_enable">';

			// The checkbox to enable the feature
			$html .= sprintf(
				'<input name="question_multi_checkboxes_enable_%s" class="wpcw_quiz_multi_checkboxes_enable" type="checkbox" %s /> %s',
				$this->quizItem->question_id,
				( $this->quizItem->question_multi_checkboxes_enable > 0 ? 'checked="checked"' : '' ),
				__( 'Yes, force the answers to display as checkboxes.', 'wp-courseware' )
			);

			$html .= '</td>';

			// Works out the space after the text area.
			$columnCount -= 2;
			if ( $columnCount > 0 ) {
				$html .= sprintf( '<td colspan="%d">&nbsp;</td>', $columnCount );
			}

			$html .= '</tr>';

			return $html;
		}



		/**
		 * Render Form.
		 *
		 * @see WPCW_quiz_base::renderForm_toString()
		 */
		public function renderForm_toString( $parentQuiz, $questionNum, $selectedAnswer, $showAsError, $errorToShow = false ) {
			// Generate the ID of the field, also used for the CSS ID
			$fieldID = sprintf( 'question_%d_%s_%d', $parentQuiz->quiz_id, $this->questionType, $this->quizItem->question_id );

			// Have they already uploaded a file? If so, tell them with a link to open the file.
			if ( $selectedAnswer ) {

				$this->extraQuizHTML .= sprintf( '<div class="wpcw_fe_quiz_q_sorting_wrapper" id="%s">', $fieldID );

				if ( $this->quizItem->question_data_answers ) {
					
					$wpcwSortingQue = WPCW_quizzes_decodeAnswers($this->quizItem->question_data_answers);
					$this->extraQuizHTML .= sprintf('<ul id="%s" class="wpcw_fe_quiz_q_answers wpcw_ques_sorting_ul">', $fieldID );	
						
						if( is_array($wpcwSortingQue) && !empty( $wpcwSortingQue ) ) {

							foreach ($selectedAnswer as $key => $sortQue ) {
				         		$this->extraQuizHTML .= sprintf("<li id='%s_{$key}' class='wpcw_que_sorting_list %s_item'>",$fieldID, $fieldID);

				         		$this->extraQuizHTML .= sprintf("<input style='display:none' type='hidden' value='{$sortQue}' name='%s[]' class='wpcw_que_sorting_list %s_item'>",$fieldID, $fieldID);

				         		if( isset( $sortQue['image'] ) ){
				         			$this->extraQuizHTML .= sprintf( '<img src="%s" class="wpcw_sorting_img %s_img">', $sortQue['image'],$fieldID );
				         		}
				         		$this->extraQuizHTML .=  sprintf("<div class='wpcw_sorting_ans'>%s</div>", $sortQue);
				         		$this->extraQuizHTML .= '<span class="dashicons dashicons-move"></span>';
				         		$this->extraQuizHTML .= "</li>";
				        	} 
						}
			      	$this->extraQuizHTML .=  '</ul>';
		      	}
				$this->extraQuizHTML .= '</div>';

			} // Only show the file upload if we don't have it.
			else {
				
				$this->extraQuizHTML .= sprintf( '<div class="wpcw_fe_quiz_q_sorting_wrapper" id="%s">', $fieldID );

				if ( $this->quizItem->question_data_answers ) {
					
					$wpcwSortingQue = WPCW_quizzes_decodeAnswers($this->quizItem->question_data_answers);
					$this->extraQuizHTML .= sprintf('<ul id="%s" class="wpcw_fe_quiz_q_answers wpcw_ques_sorting_ul">', $fieldID );	
						
						if( is_array($wpcwSortingQue) && !empty( $wpcwSortingQue ) ) {

							shuffle( $wpcwSortingQue);
							
							foreach ($wpcwSortingQue as $key => $sortQue ) {
				         		$this->extraQuizHTML .= sprintf("<li id='%s_{$key}' class='wpcw_que_sorting_list %s_item'>",$fieldID, $fieldID);

				         		$this->extraQuizHTML .= sprintf("<input style='display:none' type='hidden' value='{$sortQue['answer']}' name='%s[]' class='wpcw_que_sorting_list %s_item'>",$fieldID, $fieldID);

				         		if( isset( $sortQue['image'] ) ){
				         			$this->extraQuizHTML .= sprintf( '<img src="%s" class="wpcw_sorting_img %s_img">', $sortQue['image'],$fieldID );
				         		}
				         		$this->extraQuizHTML .=  sprintf("<div class='wpcw_sorting_ans'>%s</div>", $sortQue['answer']);
				         		$this->extraQuizHTML .= '<span class="dashicons dashicons-move"></span>';
				         		$this->extraQuizHTML .= "</li>";
				        	} 
						}
			      	$this->extraQuizHTML .=  '</ul>';
		      	}
				$this->extraQuizHTML .= '</div>';
			}

			// Work out what file types are permitted
			$fileTypes      = WPCW_files_cleanFileExtensionList( $this->quizItem->question_answer_file_types );
			$permittedFiles = false;
			if ( ! empty( $fileTypes ) ) {
				// Show message about permitted file types, which can be customised if needed.
				$permittedFiles = apply_filters( 'wpcw_front_quiz_upload_permitted_files', __( 'Allowed file types: ', 'wp-courseware' ) . implode( ', ', $fileTypes ) . '. ', $fileTypes );
			}

			// Add the hint if there is one
			if ( $this->quizItem->question_answer_hint ) {
				$this->extraQuizHTML .= sprintf( '<div class="wpcw_fe_quiz_q_hint">%s</div>', nl2br( htmlspecialchars( $this->quizItem->question_answer_hint ) ) );
			}

			// Add the file type list if there are any
			if ( $permittedFiles ) {
				$this->extraQuizHTML .= sprintf( '<div class="wpcw_fe_quiz_q_hint wpcw_fe_quiz_q_upload_permitted_files">%s</div>', $permittedFiles );
			}

			return parent::renderForm_toString_withClass( $parentQuiz, $questionNum, $selectedAnswer, $showAsError, 'wpcw_fe_quiz_q_upload', $errorToShow );
		}


		/**
		 * Clean the answer data and return it to the user. Check for true or false answer.
		 *
		 * @param String $rawData The data that's being cleaned.
		 *
		 * @return String The cleaned data.
		 */
		public static function sanitizeAnswerData( $rawData ) {
			return $rawData;
		}

		/**
		 * Validate the files that have been uploaded, checking them against the conditions of the quiz details.
		 *
		 * @param Array $fileList The list of files to be checked for this quiz.
		 * @param Array $quizDetails The details of the quiz to check
		 *
		 * @return Array The results of the file upload (upload_errors, upload_missing, upload_valid), which contain a list of the question ID and error messages.
		 */
		public static function validateFiles( $fileList, $quizDetails ) {
			// Assume that we have quiz details at this point.

			// Get a list of the questions that are expecting files.
			$questionsWithUploads = array();
			foreach ( $quizDetails->questions as $qID => $qObj ) {
				if ( 'upload' == $qObj->question_type ) {
					$questionsWithUploads[ $qID ] = $qObj;
				}
			}

			// No questions to check for.
			if ( count( $questionsWithUploads ) == 0 ) {
				return false;
			}

			// Generate a unique path for the file uploads that uses the user's private directory.
			$userPathDetails = WPCW_files_getFileUploadDirectory_forUser( $quizDetails, get_current_user_id() );

			// Prepare results data
			$results = array(
				'upload_errors'  => array(),
				'upload_missing' => array(),
				'upload_valid'   => array(),
			);

			// Check for each expected upload file that's in the list of questions
			// and do a little more validation (and handle moving the file too).
			foreach ( $questionsWithUploads as $qID => $qObj ) {
				// Generate the name of the file key to check e.g. question_16_upload_73
				$keyName = sprintf( 'question_%d_upload_%d', $quizDetails->quiz_id, $qID );

				// File was found, so need to some further checks to make sure the extension is valid
				// and then we can move the file to the right place.
				if ( isset( $fileList[ $keyName ] ) ) {
					// Uploaded file details
					$file_name  = $fileList[ $keyName ]['name'];
					$file_tmp   = $fileList[ $keyName ]['tmp_name'];
					$file_error = $fileList[ $keyName ]['error'];
					$file_size  = $fileList[ $keyName ]['size'];

					// Got a PHP upload error?
					if ( $file_error > 0 ) {
						$errMsg = __( 'Error. An unknown file upload error occurred.', 'wp-courseware' );

						switch ( $file_error ) {
							case UPLOAD_ERR_FORM_SIZE:
							case UPLOAD_ERR_INI_SIZE:
								$errMsg = sprintf( __( 'Error. The uploaded file exceeds the maximum file upload size (%s).', 'wp-courseware' ), WPCW_files_getMaxUploadSize() );
								break;

							case UPLOAD_ERR_PARTIAL:
								$errMsg = __( 'Error. The uploaded file was only partially uploaded.', 'wp-courseware' );
								break;

							case UPLOAD_ERR_NO_FILE:
								$errMsg = __( 'Error. No file was uploaded.', 'wp-courseware' );
								break;

							case UPLOAD_ERR_NO_TMP_DIR:
								$errMsg = __( 'Error. The temporary upload directory does not exist.', 'wp-courseware' );
								break;

							case UPLOAD_ERR_CANT_WRITE:
								$errMsg = __( 'Error. Could not write the uploaded file to disk.', 'wp-courseware' );
								break;

							case UPLOAD_ERR_EXTENSION:
								$errMsg = __( 'Error. An extension stopped the file upload.', 'wp-courseware' );
								break;
						}

						// Store error and don't process file further
						$results['upload_errors'][ $qID ] = $errMsg;
						continue;
					}

					// Check the valid file extensions
					$extensionTypes    = WPCW_files_cleanFileExtensionList( $qObj->question_answer_file_types );
					$thisFileExtension = pathinfo( $file_name, PATHINFO_EXTENSION );

					// File extension is not valid, so abort and move to next file.
					if ( ! in_array( strtolower( $thisFileExtension ), $extensionTypes ) ) {
						$results['upload_errors'][ $qID ] = sprintf( __( 'Error. Extension of file does not match allowed file types of %s.', 'wp-courseware' ), implode( ', ', $extensionTypes ) );
						continue;
					}

					// Move file to the new location, which is USERPATH/question_16_upload_73_user_4.ext so that we can ensure we have
					// completely safe URL for the file. And the naming convention helps the admin to a certain degree.
					$newFilename = $keyName . '_user_' . get_current_user_id() . '.' . $thisFileExtension;
					if ( move_uploaded_file( $file_tmp, $userPathDetails['dir_path'] . $newFilename ) !== false ) {
						// Store relative path of file as being a valid upload.
						$results['upload_valid'][ $qID ] = $userPathDetails['path_only'] . $newFilename;
					} // Could not move file - might be out of space, or a write error.
					else {
						$results['upload_errors'][ $qID ] = __( 'Error. Could not move file to your training directory.', 'wp-courseware' );
						continue;
					}
				} // Keep track of files that are missing.
				else {
					$results['upload_missing'][ $qID ] = true;
				}
				// end check of question in file list.
			} // end foreach

			return $results;
		} // end fn
	}
}
