/* globals GravityKit, jQuery, ajaxurl, Qs, google, window */
/**
 * Setups the main variable for this file.
 *
 * @since 3.1.0
 *
 * @type   {Object}
 */
GravityKit.GravityMaps.Controllers = GravityKit.GravityMaps.Controllers || {};
GravityKit.GravityMaps.Controllers.Map = {};

( ( $, obj ) => {
    "use strict";
	/**
	 * Creates a local alias for Utils.className() method.
	 */
	const { className } = GravityKit.GravityMaps.Utils;

	obj.selectors = {
		mapCanvas: '.gv-map-canvas',
		stickyContainer: '.gv-map-sticky-container',
		multipleEntryMap: '.gk-multi-entry-map',
		multipleEntryMapAvoidRebound: '.gk-multi-entry-map-avoid-rebound',
		generatedMap: '.gk-map-generated',
		generatedMapMarkers: '.gk-map-markers-generated',
		visibleMap: '.gk-map-visible',
		mapsEntriesContainer: '.gk-map-entries',

		viewWrapper: `[id^="gv-view-"]`,
		viewWrapperById: ( id ) => `[id^="gv-view-${id}-"]`,

		viewContainer: '.gv-container',
		viewContainerById: ( id ) => `${obj.selectors.viewContainer}-${id}`,

		viewEntry: '.gv-map-view',
	};

	const {
		doAction,
		addAction,
		applyFilters,
		addFilter,
	} = GravityKit.GravityMaps.hooks;

	/**
	 * Pull from the window object the localize variable for the GravityMaps Factory.
	 *
	 * @since 3.1.0
	 *
	 * @type {*|{}}
	 */
	obj.defaultArguments = window.gravitykit_gravitymaps_factory_default_arguments || {};

	/**
	 * The storage for the scroll observer, used for loading maps conditionally.
	 *
	 * @since 3.1.0
	 *
	 * @type {null|IntersectionObserver}
	 */
	obj.observer = null;

	/**
	 * Determine if the window was scrolled.
	 *
	 * @since 3.1.0
	 *
	 * @type {boolean}
	 */
	obj.didScroll = false;

	/**
	 * Attach all the hooks into the Actions and Filters used by the maps system.
	 *
	 * @since 3.1.0
	 */
	obj.hook = () => {

	};

	/**
	 * Triggers an error for all maps passed or in the page.
	 *
	 * @param {string} errorText
	 * @param {jQuery} $maps
	 * @param {boolean} hideMap
	 *
	 * @return {void}
	 */
	obj.triggerError = ( errorText, $maps, hideMap = true ) => {
		if ( $maps.length === 0 ) {
			$maps = obj.getMaps();
		}

		$maps.each( ( index, map ) =>{
			const $map = $( map );

			if ( hideMap ) {
				$map.hide();
			}

			const settings = obj.getSettings( $map );

			let $notice;

			errorText = $( '<div/>' ).html( errorText ).text();

			if ( settings.display_errors ) {
				$notice = $( '<div/>', {
					'class': 'gv-notice gv-error error'
				} ).html( errorText );
			} else {
				$notice = $( document.createComment( errorText ) );
			}

			$notice.insertBefore( $map );
		} );
	};

	/**
	 * Initializes the observer for the maps.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery|null} $container The container to look for maps.
	 *
	 * @return {void}
	 */
	obj.bindObserverToMaps = ( $container= null ) => {
		const $maps = obj.getMaps( $container );

		/**
		 * @action gk.maps.controllers.map.bind_observer_to_maps Allows the filtering of the maps before binding the observer.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $maps The maps to bind the observer to.
		 * @param {jQuery} $container The container to look for maps.
		 */
		doAction( 'gk.maps.controllers.map.bind_observer_to_maps', $maps, $container );

		$maps.each( ( key, map ) => obj.getObserver().observe( map ) );
	};

	/**
	 * Updates the data for a map.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 * @param {object} data The data to update.
	 *
	 * @return void
	 */
	obj.updateData = ( $map, data ) => {
		/**
		 * @filter gk.maps.controllers.map.update_data Allows the filtering of the data before updating the DOM of the map
		 *
		 * @since 3.1.0
		 *
		 * @param {object} data The data for the map.
		 * @param {jQuery} $map The map container.
		 */
		data = applyFilters( 'gk.maps.controllers.map.update_data', data, $map );
		$map.data( 'gkMap', data );
	};

	/**
	 * Get the data for a map.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 *
	 * @return {object}
	 */
	obj.getData = ( $map ) => {
		const data = $map.data( 'gkMap' );

		/**
		 * @filter gk.maps.controllers.map.get_data Allows the filtering of the data for a map.
		 *
		 * @since 3.1.0
		 *
		 * @param {object} data The data for the map.
		 * @param {jQuery} $map The map container.
		 */
		return applyFilters( 'gk.maps.controllers.map.get_data', data, $map );
	};

	/**
	 * Determines if a map is valid.
	 *
	 * @since 3.1.4
	 *
	 * @param {jQuery} $map The map container.
	 */
	obj.isValid = ( $map ) => {
		if ( ! $map || 0 === $map.length ) {
			return false;
		}

		const data = obj.getData( $map );

		if ( ! data || ! data.map ) {
			return false;
		}

		return $map.hasClass( className( obj.selectors.visibleMap ) );
	};

	/**
	 * Updates the settings for a map.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map     The map container.
	 * @param {object} settings The settings to update.
	 *
	 * @return void
	 */
	obj.updateSettings = ( $map, settings ) => {
		/**
		 * @filter gk.maps.controllers.map.update_settings Allows the filtering of the settings before updating the DOM of the map
		 *
		 * @since 3.1.0
		 *
		 * @param {object} settings The settings for the map.
		 * @param {jQuery} $map The map container.
		 */
		settings = applyFilters( 'gk.maps.controllers.map.update_settings', settings, $map );
		$map.data( 'gkMapSettings', settings );
	};

	/**
	 * Get the settings for a map.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 *
	 * @return {object}
	 */
	obj.getSettings = ( $map ) => {
		const settings = $map.data( 'gkMapSettings' );

		/**
		 * @filter gk.maps.controllers.map.get_settings Allows the filtering of the settings for a map.
		 *
		 * @since 3.1.0
		 *
		 * @param {object} settings The settings for the map.
		 * @param {jQuery} $map The map container.
		 */
		return applyFilters( 'gk.maps.controllers.map.get_settings', settings, $map );
	};

	/**
	 * Create a single map instance.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 */
	obj.create = ( $map ) => {
		// Prevents loading of all maps.
		if ( ! $map.hasClass( className( obj.selectors.visibleMap ) ) ) {
			return;
		}

		if ( $map.hasClass( className( obj.selectors.generatedMap ) ) ) {
			return;
		}
		const markGeneratedMap = () => $map.addClass( className( obj.selectors.generatedMap ) );

		/**
		 * Allow other plugins to hook into the map creation process.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $map The map container.
		 * @param {function} markGeneratedMap The callback to call when the map is created.
		 */
		doAction( 'gk.maps.controllers.map.create_map', $map, markGeneratedMap );

		/**
		 * Allow other plugins to hook into the map after finishing the creation process.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $map The map container.
		 * @param {function} markGeneratedMap The callback to call when the map is created.
		 */
		doAction( 'gk.maps.controllers.map.after_create_map', $map, markGeneratedMap );

		const settings = obj.getSettings( $map );

		// Trigger errors.
		if (
			settings.errors &&
			settings.errors.length > 0 &&
			$map.hasClass( className( obj.selectors.multipleEntryMap ) )
		) {
			settings.errors.map( ( error ) => {
				const hideMap = error.hide_map || false;
				obj.triggerError( error.message, $map, hideMap );
			} );
		}
	};

	/**
	 * Fixes issue where fitBounds() zooms in too far after adding markers
	 *
	 * @see http://stackoverflow.com/a/4065006/480856
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 */
	obj.setZoom = ( $map ) => {
		doAction( 'gk.maps.controllers.map.set_zoom', $map );
	};

	/**
	 * Centers the map properly.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 */
	obj.setCenter = (  $map ) => {
		doAction( 'gk.maps.controllers.map.set_center', $map );
	};

	/**
	 * Determine if we are in the single entry view.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 *
	 * @return {boolean}
	 */
	obj.isSingleEntry = ( $map ) => {
		return ! obj.isMultiEntry( $map );
	};

	/**
	 * Determine if we are in the single entry view.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $map The map container.
	 *
	 * @return {boolean}
	 */
	obj.isMultiEntry = ( $map ) => {
		return $map.is( obj.selectors.multipleEntryMap );
	};

	/**
	 * Given a Container gets all maps.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $container
	 * @param {boolean} filterByGenerated
	 *
	 * @returns {jQuery}
	 */
	obj.getMaps = ( $container = null, filterByGenerated = false ) => {
		if ( ! $container || 0 === $container.length ) {
			$container = $( document );
		}

		let $maps = $container.find( obj.selectors.mapCanvas );

		if ( filterByGenerated ) {
			$maps = $maps.filter( obj.selectors.generatedMap );
		}

		/**
		 * @filter gk.maps.controllers.map.get_maps Allows the filtering of the maps before returning them.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $maps The maps to filter.
		 * @param {boolean} filterByGenerated If we should filter by generated maps.
		 */
		return applyFilters( 'gk.maps.controllers.map.get_maps', $maps, filterByGenerated );
	};

	/**
	 * Given a Container gets all Search enabled Maps.
	 *
	 * @since 3.1.0
	 *
	 * @param {jQuery} $container
	 * @param {boolean} filterByGenerated
	 *
	 * @returns {jQuery}
	 */
	obj.getSearchMaps = ( $container, filterByGenerated = false ) => {
		const $maps = obj.getMaps( $container, filterByGenerated );

		/**
		 * @filter gk.maps.controllers.map.get_search_maps Allows the filtering of the maps before returning them.
		 *
		 * @since 3.1.0
		 *
		 * @param {jQuery} $maps The maps to filter.
		 * @param {boolean} filterByGenerated If we should filter by generated maps.
		 */
		return applyFilters( 'gk.maps.controllers.map.get_search_maps', $maps.filter( obj.selectors.multipleEntryMap ), filterByGenerated );
	};

	/**
	 * Gets the observer stored for this page, if not set it will create a new one.
	 *
	 * @since 3.1.0
	 *
	 * @returns {IntersectionObserver}
	 */
	obj.getObserver = () => {
		if ( ! obj.observer ) {
			// define an observer instance
			obj.observer = new IntersectionObserver( obj.onIntersection, {
				root: null,   // default is the viewport
				rootMargin: '-50px',
				threshold: 0.01, // percentage of target's visible area. Triggers "onMapIntersection"
			} );
		}

		return obj.observer;
	};

	/**
	 * Listener for the IntersectionObserver, it will only trigger when the map is visible.
	 *
	 * @since 3.1.0
	 *
	 * @param entries
	 * @param opts
	 *
	 * @return {void}
	 */
	obj.onIntersection = ( entries, opts ) => {
		entries.forEach( entry => {
			const $map = $( entry.target );

			entry.target.classList.toggle( className( obj.selectors.visibleMap ), entry.isIntersecting );

			if ( ! entry.isIntersecting ) {
				return;
			}

			if ( ! $map.hasClass( className( obj.selectors.generatedMap ) ) ) {
				obj.create( $map );
			}
		} );
	};

	/**
	 * Determines if we have any sticky maps on this page.
	 *
	 * @since 3.1.0
	 *
	 * @return {boolean}
	 */
	obj.hasStickyMaps = () => obj.$stickyMaps.length > 0;

	/**
	 * Set properties for sticky map and make sure Map Canvas height is less than 50% of window height viewport
	 * Default Canvas height = 400 px (@see assets/css/gv-maps.css )
	 *
	 * @since 3.1.0
	 *
	 * @return {void}
	 */
	obj.setupStickyMaps = () => {
		// set map container (just for sticky purposes)
		obj.$stickyMaps = $( obj.selectors.stickyContainer );
		if ( ! obj.hasStickyMaps() ) {
			return;
		}

		const windowHeight = $( window ).height();

		obj.$stickyMaps.each( ( index, element ) => {
			const $stickyMapContainer = $( element );
			const doubleCanvasHeight = $stickyMapContainer.height() * 2;

			// if viewport height is less than 2x 400 px
			if ( windowHeight < doubleCanvasHeight ) {
				$stickyMapContainer.find( obj.selectors.mapCanvas ).height( windowHeight / 2 );
			}
		} );

		obj.bindScrollCheck();
		obj.setupMobile();
	};

	/**
	 * Set offset for sticky map calculations, important to avoid jumping.
	 *
	 * @since 3.1.0
	 *
	 * @return {void}
	 */
	obj.setupScrollOffset = () => {
		obj.$stickyMaps.each( ( index, element ) => {
			const $stickyMapContainer = $( element );
			$stickyMapContainer.data( 'offset', $stickyMapContainer.offset().top );
		} );
	};

	/**
	 * Flag that the window has scrolled, prevents unnecessary calculations.
	 *
	 * @since 3.1.0
	 *
	 * @return {boolean}
	 */
	obj.flagScroll = () => obj.didScroll = true;

	/**
	 * Bind scroll check to window for Sticky Maps.
	 *
	 * @since 3.1.0
	 *
	 * @return {void}
	 */
	obj.bindScrollCheck = () => {
		if ( ! obj.hasStickyMaps() ) {
			return;
		}

		$( window ).one( 'scroll', obj.setupScrollOffset );
		setInterval( obj.onWindowScroll, 250 );
		$( window ).on( 'scroll', obj.flagScroll );
	};

	/**
	 * Check if window has scrolled and if so, update sticky map position.
	 *
	 * @since 3.1.0
	 *
	 * @return {void}
	 */
	obj.onWindowScroll = () => {
		if ( ! obj.didScroll ) {
			return;
		}

		if ( ! obj.hasStickyMaps() ) {
			return;
		}

		obj.didScroll = false;
		const scroll = $( window ).scrollTop();

		obj.$stickyMaps.each( ( index, element ) => {
			const $stickyMapContainer = $( element );
			const offset = $stickyMapContainer.data( 'offset' );
			const $stickyMap = $stickyMapContainer.find( obj.selectors.mapCanvas );
			const canvasWidth = $stickyMap.width();
			const canvasHeight = $stickyMap.height();
			const $viewWrapper = $stickyMapContainer.parents( obj.selectors.viewWrapper ).eq( 0 );
			const $entriesList = $viewWrapper.find( obj.selectors.mapsEntriesContainer );
			const settings = obj.getSettings( $stickyMap );

			if ( scroll >= offset ) {
				$stickyMap.width( canvasWidth );
				$stickyMapContainer.addClass( 'gv-sticky' );
				if ( settings.template_layout === 'top' ) {
					$entriesList.css( 'margin-top', canvasHeight + 'px' );
				}
			} else {
				$stickyMap.width( '100%' );
				$stickyMapContainer.removeClass( 'gv-sticky' );
				if ( settings.template_layout === 'top' ) {
					$entriesList.css( 'margin-top', '' );
				}
			}
		} );
	};

	/**
	 * Check if the page is being loaded in a mobile environment.
	 *
	 * @since 3.1.0
	 *
	 * @return {void}
	 */
	obj.setupMobile = () => {
		// only apply this logic for the map template containing the sticky map (even if it is not pinned)
		if ( ! obj.hasStickyMaps() ) {
			return;
		}

		const windowWidth = $( window ).width();

		obj.$stickyMaps.each( ( index, element ) => {
			const $stickyMapContainer = $( element );
			const $map = $stickyMapContainer.find( obj.selectors.mapCanvas );
			const settings = obj.getSettings( $map );

			if ( windowWidth > parseInt( settings.mobile_breakpoint, 10 ) ) {
				return;
			}

			const $parent = $stickyMapContainer.parent();
			const $container = $stickyMapContainer.parents( obj.selectors.viewContainer ).eq( 0 );

			if ( $parent.hasClass( 'gv-grid-col-1-3' ) && 1 === $parent.index() ) {
				$parent.detach().prependTo( $container );
			}
		} );
	};

	/**
	 * Triggers when the window finishes loading.
	 *
	 * @since 3.1.0
	 *
	 * @return {void}
	 */
	obj.load = () => {
		obj.setupStickyMaps();
		obj.bindObserverToMaps();
	};

	/**
	 * Hooking needs to happen as early as possible.
	 */
	obj.hook();

	$( window ).on( 'load', obj.load );

} )( jQuery, GravityKit.GravityMaps.Controllers.Map );
